/* FIGARO'S PASSWORD MANAGER (FPM)
 * Copyright (C) 2000 John Conneely
 * 
 * FPM is open source / free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * FPM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * callbacks.c
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>
#include <string.h>
#include <stdlib.h>

#include "fpm.h"
#include "fpm_gpw.h"
#include "passfile.h"
#include "fpm_crypt.h"
#include "callbacks.h"
#include "interface.h"
#include "support.h"

void
on_preferences1_activate               (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  fpm_prefs_init();
}


void
on_about1_activate                     (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  GtkWidget *about_dialog;
	
  about_dialog = create_aboutdialog1();
  gtk_window_set_icon (GTK_WINDOW(about_dialog), gtk_widget_render_icon (about_dialog, GTK_STOCK_ABOUT, GTK_ICON_SIZE_MENU, NULL));
  gtk_window_set_transient_for (GTK_WINDOW(about_dialog), GTK_WINDOW(glb_win_app));
  gtk_widget_show (about_dialog);
}

void
on_app_safe_destroy                    (GtkObject       *object,
                                        gpointer         user_data)
{
    gtk_main_quit();
}

void
on_button_ok_clicked                   (GtkButton       *button,
                                        gpointer         user_data)
{
  fpm_save_passitem(glb_win_edit, glb_edit_data);
  gtk_widget_destroy(glb_win_edit);
  gtk_tree_view_set_cursor(glb_clist_main, gtk_tree_path_new_from_string(g_strdup_printf("%i", glb_cur_row)), NULL, FALSE);
  gtk_widget_set_sensitive(glb_win_app, TRUE);
  gtk_widget_grab_focus(GTK_WIDGET(glb_clist_main));
}


void
on_button_cancel_clicked               (GtkButton       *button,
                                        gpointer         user_data)
{
  gtk_widget_destroy(glb_win_edit);
  gtk_tree_view_set_cursor(glb_clist_main, gtk_tree_path_new_from_string(g_strdup_printf("%i", glb_cur_row)), NULL, FALSE);
  gtk_widget_set_sensitive(glb_win_app, TRUE);
  gtk_widget_grab_focus(GTK_WIDGET(glb_clist_main));
}


void
on_button_new_clicked                  (GtkButton       *button,
                                        gpointer         user_data)
{
  fpm_new_passitem(&glb_win_edit, &glb_edit_data);
}



void
on_button_edit_clicked                 (GtkButton       *button,
                                        gpointer         user_data)
{
  if (glb_edit_data != NULL)
    fpm_edit_passitem(&glb_win_edit, glb_edit_data);
  else
    fpm_message(GTK_WINDOW(glb_win_app), _("\nPlease select a row."), GTK_MESSAGE_INFO);
}


gboolean
on_app_safe_delete_event               (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
    if (glb_fpm_ini->enable_tray_icon)
	fpm_tr_toggle_win_app(FALSE);
    else
	fpm_quit();
	
  return TRUE;
}


gboolean
on_clist_main_button_press_event       (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
  
 // Limit dblclick on mouse button 1 only
 if (event->button == 1) {
  /* I couldn't see a way to handle double clicks with a CList,
   * so I implemented my own using the button press and release events.
   * This is the press events.  We check to see how many clicks we get
   * within FPM_MAX_DOUBLE_CLICK_TIME seconds, currently set to 0.5.
   */
  if((g_timer_elapsed(glb_timer_click, NULL) < FPM_MAX_DOUBLE_CLICK_TIME) && (glb_click_row == glb_cur_row))
  {
    glb_click_count++;
  }
  else
  {
    g_timer_reset(glb_timer_click);
    glb_click_count = 1;
    glb_click_row = glb_cur_row;
  }
 }

 if (event->button == 3) {

    glb_click_btn = TRUE;
 }
  return FALSE;
}


gboolean
on_clist_main_button_release_event     (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
 // Limit dblclick on mouse button 1 only
 if (event->button == 1) {
  /* Handle double clicks
   * Here we are potentially releasing from a double click.  If we've clicked
   * more than once in the time allowed, consider it a double click.
   */
  if((g_timer_elapsed(glb_timer_click, NULL) < FPM_MAX_DOUBLE_CLICK_TIME) &&
     (glb_click_count > 1))
  {
    fpm_double_click(glb_edit_data);
  }
 }
  return FALSE;
}

void
on_clist_main_select_row               (GtkTreeView *treeview,
					gpointer data)
{
  GtkTreeSelection *selection;
  GtkTreeIter selectedIter;
  GtkTreeModel *model;
  
  selection = gtk_tree_view_get_selection(treeview);
  if (gtk_tree_selection_get_selected(selection, &model, &selectedIter)) {

	gtk_tree_model_get(model, &selectedIter,
	  FPM_COL_DATA_POINTER, &glb_edit_data,
	  -1);
    
    glb_cur_row = atoi(gtk_tree_model_get_string_from_iter(model, &selectedIter));

    if (glb_click_btn) {
        GtkMenu *menu;
    
	menu = GTK_MENU(create_context_menu());
    
        gtk_menu_popup (menu, NULL, NULL, NULL, NULL,
	    	        0, gtk_get_current_event_time ());

	glb_click_btn = FALSE;
    }
  }

}


void
on_save1_activate                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  passfile_save(glb_filename);
}


void
on_exit1_activate                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  fpm_quit();
}


void
on_button_save_clicked                 (GtkButton       *button,
                                        gpointer         user_data)
{
  passfile_save(glb_filename);
}


gboolean
on_dialog_edit_passitem_delete_event   (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
  GtkTextIter start, end;
  GtkTextView* notes;
  GtkTextBuffer*  buffer;
  gchar *entry_notes;

  notes = GTK_TEXT_VIEW(lookup_widget(glb_win_edit, "text_notes"));
  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (notes));
  gtk_text_buffer_get_bounds (buffer, &start, &end);
  entry_notes = gtk_text_iter_get_text (&start, &end);

  if (strcmp(glb_edit_data->title, fpm_get_entry(glb_win_edit, "entry_title"))
	|| strcmp(glb_edit_data->arg, fpm_get_entry(glb_win_edit, "entry_arg"))
	|| strcmp(glb_edit_data->user, fpm_get_entry(glb_win_edit, "entry_user"))
	|| strcmp(glb_edit_data->category, fpm_get_entry(glb_win_edit, "entry_edit_category"))
	|| strcmp(glb_edit_data->launcher, fpm_get_entry(glb_win_edit, "entry_edit_launcher"))
	|| strcmp(glb_edit_data->notes, entry_notes)
	|| glb_edit_data->default_list != GTK_TOGGLE_BUTTON(lookup_widget(glb_win_edit, "checkbutton_default"))->active
     ) {
	
	fpm_question(GTK_WINDOW(glb_win_edit), _("Are you sure you want discard changes?"),
		 G_CALLBACK(fpm_dialog_answer_edit));

	return TRUE;
  }	
  gtk_widget_destroy (glb_win_edit);
  gtk_widget_set_sensitive (glb_win_app, TRUE);
  gtk_widget_grab_focus (GTK_WIDGET(lookup_widget(glb_win_app, "clist_main")));

  return FALSE;
}


void
on_dialog_edit_passitem_destroy        (GtkObject       *object,
                                        gpointer         user_data)
{  
  /* Make main window active again */
  gtk_widget_set_sensitive(glb_win_app, TRUE);
 
}


void
on_dialog_edit_passitem_show           (GtkWidget       *widget,
                                        gpointer         user_data)
{

  /* Make main window inactive to simulate a modal edit window. 
   * We don't want to make the edit window really modal because
   * then anoying things happen like confirmation dialog boxes'
   * buttons don't work.
   */
  gtk_widget_set_sensitive(glb_win_app, FALSE);

}


void
on_togglebutton_showpass_toggled       (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
    GtkEntry* entry_password;
  
    entry_password = GTK_ENTRY(lookup_widget (glb_win_edit, "entry_password"));
    gtk_entry_set_visibility(entry_password, togglebutton->active);
}

void
on_button_jump_clicked                 (GtkButton       *button,
                                        gpointer         user_data)
{
  if (glb_edit_data != NULL)
    fpm_jump(glb_edit_data);
  else
    fpm_message(GTK_WINDOW(glb_win_app), _("\nPlease select a row."), GTK_MESSAGE_INFO);
}


void
on_button_pass_clicked                 (GtkButton       *button,
                                        gpointer         user_data)
{
  if (glb_edit_data != NULL)
    fpm_select_password(glb_edit_data, TRUE);
  else
    fpm_message(GTK_WINDOW(glb_win_app), _("\nPlease select a row."), GTK_MESSAGE_INFO);
}


void
on_button_user_clicked                 (GtkButton       *button,
                                        gpointer         user_data)
{
  if (glb_edit_data != NULL)
    fpm_select(glb_edit_data->user, TRUE);
  else
    fpm_message(GTK_WINDOW(glb_win_app), _("\nPlease select a row."), GTK_MESSAGE_INFO);
}


void
on_button_password_ok_clicked          (GtkButton       *button,
                                        gpointer         user_data)
{
  fpm_check_password();
}


void
on_button_password_cancel_clicked      (GtkButton       *button,
                                        gpointer         user_data)
{
  gtk_main_quit();
}


void
on_gpw_toggled                         (GtkEditable     *editable,
                                        gpointer         user_data)
{
  GtkWidget* widget;

  widget=lookup_widget(glb_win_misc, "entry_bits");
  if(strcmp("", gtk_entry_get_text(GTK_ENTRY(widget))))
  {
	  fpm_gpw_set_from_dialog(glb_win_misc);
  }

}


void
on_button_generate_clicked             (GtkButton       *button,
                                        gpointer         user_data)
{

  glb_win_misc = create_dialog_password_gen();
  fpm_gpw_start_screen(glb_win_misc);
  gtk_widget_show(glb_win_misc);

}

void
on_button_gen_generate_now_clicked     (GtkButton       *button,
                                        gpointer         user_data)
{
  GtkWidget* widget;

  widget = GTK_WIDGET(create_window_entropy());

  gtk_window_set_transient_for (GTK_WINDOW(widget), GTK_WINDOW(glb_win_app));
  gtk_widget_show(widget);
  while (gtk_events_pending()) gtk_main_iteration();

  fpm_gpw_fillin_password(glb_win_misc);

  gtk_widget_hide(widget);
}


void
on_button_gen_ok_clicked               (GtkButton       *button,
                                        gpointer         user_data)
{
  GtkEntry *entry1, *entry2;
  gchar* text;

  entry1 = GTK_ENTRY(lookup_widget(glb_win_misc, "entry_gen_password"));
  entry2 = GTK_ENTRY(lookup_widget(glb_win_edit, "entry_password"));

  text = gtk_editable_get_chars (GTK_EDITABLE(entry1), 0, -1);
  if (strcmp(text, "")) gtk_entry_set_text(entry2, text);

  gtk_widget_destroy(glb_win_misc);
}


void
on_button_gen_cancel_clicked           (GtkButton       *button,
                                        gpointer         user_data)
{
  gtk_widget_destroy(glb_win_misc);
}


void
on_entry_password_activate             (GtkEditable     *editable,
                                        gpointer         user_data)
{
  fpm_check_password();
}


void
on_button_cpw_ok_clicked               (GtkButton       *button,
                                        gpointer         user_data)
{
  fpm_set_password();
}


void
on_button_cpw_cancel_clicked           (GtkButton       *button,
                                        gpointer         user_data)
{
  if(old_context == NULL)
  {
    /* We must be running for the first time.  Cancel quits. */
    gtk_main_quit();
  }
  else
  {
    /* User is canceling from changing a password.  Just close the screen. */
    gtk_widget_destroy(glb_win_misc);
  }
}



void
on_copy_pass_activate                  (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  if (glb_edit_data != NULL)
    fpm_select_password(glb_edit_data, TRUE);
  else
    fpm_message(GTK_WINDOW(glb_win_app), _("\nPlease select a row."), GTK_MESSAGE_INFO);
}


void
on_copy_user_activate                  (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  if (glb_edit_data != NULL)
    fpm_select(glb_edit_data->user, TRUE);
  else
    fpm_message(GTK_WINDOW(glb_win_app), _("\nPlease select a row."), GTK_MESSAGE_INFO);
}


void
on_item_password_activate              (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
    glb_win_misc = create_dialog_cpw();
    gtk_widget_show(glb_win_misc);
}


void
on_button_prefs_ok_clicked             (GtkButton       *button,
                                        gpointer         user_data)
{
  fpm_prefs_save_launcher();
}


void
on_button_prefs_cancel_clicked         (GtkButton       *button,
                                        gpointer         user_data)
{
  gtk_widget_destroy(glb_win_misc);
}

void
on_item_delete_activate                (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  if (glb_edit_data != NULL)
  {
    fpm_question(GTK_WINDOW(glb_win_app), _("Are you sure you want to delete this item?"),
		 G_CALLBACK(fpm_dialog_answer_delete));
  }
  else
    fpm_message(GTK_WINDOW(glb_win_app), _("\nPlease select a row."), GTK_MESSAGE_INFO);
}

gboolean
on_dialog_prefs_destroy_event          (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{

  return FALSE;
}


void
on_dialog_prefs_destroy                (GtkObject       *object,
                                        gpointer         user_data)
{
  gtk_widget_set_sensitive(glb_win_app, TRUE);
}


void
on_export1_activate                    (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
    passfile_export(glb_filename);

}


void
on_import_passwords1_activate          (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
    GtkWidget *dialog;
    GtkFileFilter *filter;
    const gchar *fpm_directory;

    dialog = gtk_file_chooser_dialog_new (_("Import Password File"),
    				      GTK_WINDOW(glb_win_app),
    				      GTK_FILE_CHOOSER_ACTION_OPEN,
				      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				      GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
				      NULL);
    filter = gtk_file_filter_new ();
    gtk_file_filter_add_pattern (filter, "*");
    gtk_file_filter_set_name (filter, _("All Files"));
    gtk_file_chooser_add_filter (GTK_FILE_CHOOSER(dialog), filter);

    fpm_directory = g_get_home_dir();
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), fpm_directory);

    if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
    {
	char *filename;

        filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
	printf("Choose %s file.\n", filename);
	passfile_import(filename);
        g_free (filename);
    }
    gtk_widget_destroy (dialog);


}

static void on_button_import_ok_clicked_cb (GtkWidget *widget, gpointer data)
{
	gint *index = data;

	g_print("%p\n", widget);
	gtk_object_set_user_data(GTK_OBJECT(widget), GINT_TO_POINTER(*index));
	++*index;
}

void
on_button_import_ok_clicked            (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkEntry* entry;
	GtkOptionMenu *optmenu;
	GtkWidget* menu;
	GtkWidget* menuitem;
	gchar *fname;
	gint format;
	gint index;

	entry = GTK_ENTRY(lookup_widget (glb_win_import,
					 "import_password_file"));
	fname = gtk_editable_get_chars (GTK_EDITABLE(entry), 0, -1);

	optmenu = GTK_OPTION_MENU(lookup_widget (glb_win_import,
						 "import_password_format"));
	menu = gtk_option_menu_get_menu (optmenu);
	menuitem =  gtk_menu_get_active (GTK_MENU(menu));

	
	index = 0;
	gtk_container_foreach (GTK_CONTAINER(menu),
			       (GtkCallback)on_button_import_ok_clicked_cb,
			       &index);

	format = GPOINTER_TO_INT(
			gtk_object_get_user_data(GTK_OBJECT(menuitem)));

	switch (format) {
		case 0:
			g_print("tab delimited choosen\n");
			passfile_import(fname);
			break;
		case 1:
			g_print("gpasman choosen\n");
			break;
	}
	gtk_widget_destroy(glb_win_import);
	gtk_widget_set_sensitive(glb_win_app, TRUE);
}


void
on_button_import_cancel_clicked        (GtkButton       *button,
                                        gpointer         user_data)
{
  gtk_widget_destroy(glb_win_import);
  gtk_widget_set_sensitive(glb_win_app, TRUE);

}


void
on_current_password_file1_activate     (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	gchar *dia_str;

	dia_str = g_strdup_printf(_("Current Location of Password file:\n"
		"%s"), glb_filename);
	fpm_message(GTK_WINDOW(glb_win_app), dia_str, GTK_MESSAGE_INFO);
	g_free(dia_str);

}

void
on_open1_activate                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{

    GtkWidget *dialog;
    gchar *fpm_directory;

    if (glb_dirty) {
    fpm_question(GTK_WINDOW(glb_win_app), _("Do you want to save changes?"),
		 G_CALLBACK(fpm_dialog_answer_save));
    }

    dialog = gtk_file_chooser_dialog_new (_("Open Password File"),
				      GTK_WINDOW(glb_win_app),
				      GTK_FILE_CHOOSER_ACTION_OPEN,
				      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				      GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
				      NULL);

    fpm_directory = g_strjoin(NULL, g_get_home_dir(), "/.fpm/", NULL);
    
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), fpm_directory);
    
    if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {
        glb_filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
	unlock_fpm_file();
	fpm_init(glb_filename);
    }
    gtk_widget_destroy (dialog);
}


void
on_save_as1_activate                   (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
    GtkWidget *dialog;
    gchar *fpm_directory;

    dialog = gtk_file_chooser_dialog_new (_("Save Password File"),
    				      GTK_WINDOW(glb_win_app),
    				      GTK_FILE_CHOOSER_ACTION_SAVE,
				      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				      GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
				      NULL);
    gtk_file_chooser_set_do_overwrite_confirmation (GTK_FILE_CHOOSER (dialog), TRUE);

    fpm_directory = g_strjoin(NULL, g_get_home_dir(), "/.fpm/", NULL);
    
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), fpm_directory);
    gtk_file_chooser_set_current_name (GTK_FILE_CHOOSER (dialog), _("Untitled password file"));
    
    if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
    {
	char *filename;

        filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
	passfile_save(filename);
        g_free (filename);
    }
    gtk_widget_destroy (dialog);
}

void
on_preferences2_activate               (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
    glb_win_misc = create_dialog_preferences();
    gtk_window_set_transient_for (GTK_WINDOW(glb_win_misc), GTK_WINDOW(glb_win_app));
//    gtk_window_set_destroy_with_parent (GTK_WINDOW(glb_win_misc), TRUE);

    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "save_on_add")), glb_fpm_ini->save_on_add);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "save_on_change")), glb_fpm_ini->save_on_change);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "save_on_delete")), glb_fpm_ini->save_on_delete);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "save_on_quit")), glb_fpm_ini->save_on_quit);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "create_backup")), glb_fpm_ini->create_backup);
    gtk_spin_button_set_value (GTK_SPIN_BUTTON(lookup_widget (glb_win_misc, "number_backup_files")), glb_fpm_ini->number_backup_files);

    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "search_in_title")), glb_fpm_ini->search_in_title);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "search_in_url")), glb_fpm_ini->search_in_url);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "search_in_user")), glb_fpm_ini->search_in_user);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "search_in_notes")), glb_fpm_ini->search_in_notes);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "search_match_case")), glb_fpm_ini->search_match_case);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "search_limit_category")), glb_fpm_ini->search_limit_category);

    if (glb_fpm_ini->create_backup)
	gtk_widget_set_sensitive (GTK_WIDGET(lookup_widget (glb_win_misc, "backup_files_hbox")), TRUE);

    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "enable_tray_icon")), glb_fpm_ini->enable_tray_icon);
    if (glb_fpm_ini->enable_tray_icon) {
	gtk_widget_set_sensitive (GTK_WIDGET(lookup_widget (glb_win_misc, "tr_always_visible")), TRUE);
	gtk_widget_set_sensitive (GTK_WIDGET(lookup_widget (glb_win_misc, "tr_auto_hide")), TRUE);
	gtk_widget_set_sensitive (GTK_WIDGET(lookup_widget (glb_win_misc, "tr_auto_lock")), TRUE);
}
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "tr_always_visible")), glb_fpm_ini->tr_always_visible);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "tr_auto_hide")), glb_fpm_ini->tr_auto_hide);
    gtk_spin_button_set_value (GTK_SPIN_BUTTON(lookup_widget (glb_win_misc, "tr_auto_hide_minutes")), glb_fpm_ini->tr_auto_hide_minutes);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "tr_auto_lock")), glb_fpm_ini->tr_auto_lock);
    gtk_spin_button_set_value (GTK_SPIN_BUTTON(lookup_widget (glb_win_misc, "tr_auto_lock_minutes")), glb_fpm_ini->tr_auto_lock_minutes);
    if (!glb_fpm_ini->enable_tray_icon) {
	gtk_widget_set_sensitive (GTK_WIDGET(lookup_widget (glb_win_misc , "tr_auto_hide_align")), FALSE);
	gtk_widget_set_sensitive (GTK_WIDGET(lookup_widget (glb_win_misc , "tr_auto_lock_align")), FALSE);
    }
    
    if (gtk_dialog_run (GTK_DIALOG (glb_win_misc)) == GTK_RESPONSE_OK)
    {
	glb_fpm_ini->save_on_add = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "save_on_add")));
	glb_fpm_ini->save_on_change = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "save_on_change")));
	glb_fpm_ini->save_on_delete = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "save_on_delete")));
	glb_fpm_ini->save_on_quit = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "save_on_quit")));
	glb_fpm_ini->create_backup = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "create_backup")));
	glb_fpm_ini->number_backup_files = gtk_spin_button_get_value (GTK_SPIN_BUTTON(lookup_widget (glb_win_misc, "number_backup_files")));

	glb_fpm_ini->search_in_title = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "search_in_title")));
	glb_fpm_ini->search_in_url = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "search_in_url")));
	glb_fpm_ini->search_in_user = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "search_in_user")));
	glb_fpm_ini->search_in_notes = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "search_in_notes")));
	glb_fpm_ini->search_match_case = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "search_match_case")));
	glb_fpm_ini->search_limit_category = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "search_limit_category")));

	glb_fpm_ini->enable_tray_icon = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "enable_tray_icon")));
	glb_fpm_ini->tr_always_visible = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "tr_always_visible")));
	glb_fpm_ini->tr_auto_hide = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "tr_auto_hide")));
	glb_fpm_ini->tr_auto_hide_minutes = gtk_spin_button_get_value (GTK_SPIN_BUTTON(lookup_widget (glb_win_misc, "tr_auto_hide_minutes")));
	glb_fpm_ini->tr_auto_lock = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "tr_auto_lock")));
	glb_fpm_ini->tr_auto_lock_minutes = gtk_spin_button_get_value (GTK_SPIN_BUTTON(lookup_widget (glb_win_misc, "tr_auto_lock_minutes")));

	if (glb_fpm_ini->enable_tray_icon)
	    fpm_tray_icon();
	else {
	    fpm_tr_cleanup();
	    gtk_widget_hide (GTK_WIDGET(lookup_widget (glb_win_app , "hide_to_tray")));
//	    g_signal_handlers_disconnect_by_func (glb_win_app, G_CALLBACK (on_app_safe_event), NULL);
	    g_source_remove_by_user_data (glb_win_app);
	}

	fpm_ini_save();
    }
    gtk_widget_destroy (glb_win_misc);
}

void
on_create_backup_toggled               (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
    gtk_widget_set_sensitive (GTK_WIDGET(lookup_widget (glb_win_misc , "backup_files_hbox")), togglebutton->active);
}

void
on_add_item1_activate                   (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	fpm_new_passitem(&glb_win_edit, &glb_edit_data);
}

void
on_edit_item1_activate                   (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  if (glb_edit_data != NULL)
    fpm_edit_passitem(&glb_win_edit, glb_edit_data);
  else
    fpm_message(GTK_WINDOW(glb_win_app), _("\nPlease select a row."), GTK_MESSAGE_INFO);
}

gboolean
on_dialog_password_delete_event        (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{

  gtk_main_quit();
  return(TRUE);
}

void
on_optionmenu_category_changed         (GtkComboBox     *combobox,
                                        gpointer         user_data)
{
    gchar *dia_str;
    
    glb_last_category = gtk_combo_box_get_active_text(combobox);
    fpm_clist_create_view(glb_last_category);

    dia_str = g_strdup_printf(_("Passwords in category: %d"), glb_num_row);
    fpm_statusbar_push(dia_str);
}

void
on_find_activate                       (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{

    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lookup_widget (glb_win_app, "find_match_case")), glb_fpm_ini->search_match_case);

    gtk_widget_show (GTK_WIDGET(lookup_widget (glb_win_app, "findbar")));
    gtk_widget_grab_focus (GTK_WIDGET(lookup_widget (glb_win_app, "find_entry")));
}


void
on_find_close_clicked                  (GtkToolButton   *toolbutton,
                                        gpointer         user_data)
{
    gtk_widget_hide (GTK_WIDGET(lookup_widget (glb_win_app, "findbar")));
    fpm_clist_create_view(glb_last_category);
}

gboolean
on_find_entry_key_press_event          (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
    // Escape
    if (event->keyval == 0xff1b) {
	gtk_widget_hide (GTK_WIDGET(lookup_widget (glb_win_app, "findbar")));
	fpm_clist_create_view(glb_last_category);
    	return TRUE;
    }

  return FALSE;
}

void
on_find_entry_changed                  (GtkEditable     *editable,
                                        gpointer         user_data)
{
    fpm_search(gtk_editable_get_chars(editable, 0, -1), FALSE);
}


void
on_find_match_case_toggled             (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
    glb_fpm_ini->search_match_case = togglebutton->active;
    gtk_widget_activate(GTK_WIDGET(lookup_widget (glb_win_app, "find_entry")));
}


void
on_find_entry_activate                 (GtkEntry        *entry,
                                        gpointer         user_data)
{
    fpm_search(gtk_editable_get_chars(GTK_EDITABLE(entry), 0, -1), TRUE);
}

void
tray_icon_on_click			(GtkStatusIcon *status_icon, 
					 gpointer user_data)
{
    if (glb_locked) {
	gtk_status_icon_set_visible(glb_tray_icon, FALSE);
	glb_locked = FALSE;
	unlock_fpm_file();
	fpm_init(glb_filename);
    } else {
	fpm_tr_toggle_win_app(FALSE);
    }
}

void
tray_icon_on_menu			(GtkStatusIcon *status_icon, guint button, 
                    			 guint activate_time, gpointer user_data)
{
    GtkMenu *menu;

    menu = GTK_MENU(create_tray_menu());

    if (!glb_locked) {
	gtk_widget_hide (GTK_WIDGET(lookup_widget (GTK_WIDGET(menu) , "unlock1")));
	gtk_widget_show (GTK_WIDGET(lookup_widget (GTK_WIDGET(menu) , "lock1")));
	gtk_widget_show (GTK_WIDGET(lookup_widget (GTK_WIDGET(menu) , "show1")));
	gtk_widget_show (GTK_WIDGET(lookup_widget (GTK_WIDGET(menu) , "tr_about1")));
    } else {
	gtk_widget_show (GTK_WIDGET(lookup_widget (GTK_WIDGET(menu) , "unlock1")));
	gtk_widget_hide (GTK_WIDGET(lookup_widget (GTK_WIDGET(menu) , "show1")));
	gtk_widget_hide (GTK_WIDGET(lookup_widget (GTK_WIDGET(menu) , "lock1")));
	gtk_widget_hide (GTK_WIDGET(lookup_widget (GTK_WIDGET(menu) , "tr_about1")));
    }

    gtk_menu_popup (menu, NULL, NULL, gtk_status_icon_position_menu, status_icon,
    0, gtk_get_current_event_time());
}

void
on_enable_tray_icon_toggled            (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
    gtk_widget_set_sensitive (GTK_WIDGET(lookup_widget (glb_win_misc , "tr_always_visible")), togglebutton->active);
    gtk_widget_set_sensitive (GTK_WIDGET(lookup_widget (glb_win_misc , "tr_auto_hide")), togglebutton->active);
    gtk_widget_set_sensitive (GTK_WIDGET(lookup_widget (glb_win_misc , "tr_auto_lock")), togglebutton->active);

    gtk_widget_set_sensitive (GTK_WIDGET(lookup_widget (glb_win_misc , "tr_auto_hide_align")), togglebutton->active & glb_fpm_ini->tr_auto_hide);
    gtk_widget_set_sensitive (GTK_WIDGET(lookup_widget (glb_win_misc , "tr_auto_lock_align")), togglebutton->active & glb_fpm_ini->tr_auto_lock);
}

void
on_lock1_activate                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
    fpm_lock();
}


void
on_tr_auto_hide_toggled                (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
    gtk_widget_set_sensitive (GTK_WIDGET(lookup_widget (glb_win_misc , "tr_auto_hide_align")), togglebutton->active);
    glb_fpm_ini->tr_auto_hide = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "tr_auto_hide")));
}


void
on_tr_auto_lock_toggled                (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
    gtk_widget_set_sensitive (GTK_WIDGET(lookup_widget (glb_win_misc , "tr_auto_lock_align")), togglebutton->active);
    glb_fpm_ini->tr_auto_lock = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(lookup_widget (glb_win_misc, "tr_auto_lock")));
}

void
on_hide_to_tray_activate               (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
    fpm_tr_toggle_win_app(FALSE);
}


gboolean
on_app_safe_window_state_event         (GtkWidget       *widget,
                                        GdkEventWindowState	 *event,
                                        gpointer         user_data)
{

    if (event->changed_mask & GDK_WINDOW_STATE_ICONIFIED)
	glb_minimized = (event->new_window_state & GDK_WINDOW_STATE_ICONIFIED) ? 1 : 0;

    return FALSE;
}

void
on_show1_activate                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
    fpm_tr_toggle_win_app(FALSE);
}

gboolean
on_app_safe_focus_in_event             (GtkWidget       *widget,
                                        GdkEventFocus   *event,
                                        gpointer         user_data)
{
    if (glb_fpm_ini->tr_auto_hide)
	g_source_remove_by_user_data (glb_win_app);

    return FALSE;
}


gboolean
on_app_safe_focus_out_event            (GtkWidget       *widget,
                                        GdkEventFocus   *event,
                                        gpointer         user_data)
{
    if (!fpm_window_check() && glb_fpm_ini->tr_auto_hide && !glb_minimized)
	g_timeout_add_seconds(glb_fpm_ini->tr_auto_hide_minutes*60, (GSourceFunc) fpm_auto_hide, glb_win_app);

  return FALSE;
}


void
on_unlock1_activate                    (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
    gtk_status_icon_set_visible(glb_tray_icon, FALSE);
    glb_locked = FALSE;
    unlock_fpm_file();
    fpm_init(glb_filename);
}
