
/*
 * DLCICFG	A cfguration program for Data Link Connection Identifiers
 *		used along with Frame Relay Access Devices.  Used to create, 
 *		cfg and delete DLCI devices.  
 *
 * Version:	@(#)dlcicfg.c	0.20	25 Jan 1998
 *
 * Author:	Mike McLagan <mike.mclagan@linux.org>
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 */

#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <ctype.h>
#include <unistd.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>

#if __GLIBC__
#define _LINUX_SOCKET_H
#endif

#include <linux/if.h>
#include <linux/if_frad.h>

#include "cfgfile.h"
#include "tags.h"

char *banner = "DLCI configuration, v0.20(25 Jan 1998) by mike.mclagan@linux.org";

void usage(char *progname, char *invalid)
{
   char *temp;

   temp = strrchr(progname, '/');
   if (temp)
   {
      *temp = 0;
      progname = temp + 1;
   }

   if (invalid)
      printf("Invalid argument specified: %s\n", invalid);

   printf("Usage: %s [-Q] [-?rtq -cn -bn -en -Cn -Bn -En] add <dlci> <frad>\n", progname);
   printf("       %s [-Q] [-?rtq -cn -bn -en -Cn -Bn -En] conf <device>\n", progname);
   printf("       %s [-Q] del <device>\n", progname);
   printf("       %s [-Q] file cfgfile\n", progname);
   printf("       -?        - this message\n");
   printf("       -Q        - run quietly\n");
   printf("       -r        - ignore incoming CIR\n");
   printf("       -t        - ignore outgoing CIR\n");
   printf("       -q        - queue all outgoing packets\n\n");
   printf("       -c        - Committed information rate, outgoing\n");
   printf("       -b        - Committed burst size, outgoing\n");
   printf("       -e        - Excess burst size, outgoing\n\n");
   printf("       -C        - Committed information rate, incoming\n");
   printf("       -B        - Committed burst size, incoming\n");
   printf("       -E        - Excess burst size, incoming\n\n");
   printf("       dlci      - the DLCI number\n");
   printf("       frad      - FRAD device for data transmission\n");
   printf("       device    - a created DLCI device: dlci00 - dlci99\n");
   printf("       cfgfile   - dlci configuration file\n");
   exit(-1); 
}

void config(char *dev, struct dlci_conf *conf, int quiet)
{
   int              i, sock;
   struct ifreq     ifr;

   strcpy(ifr.ifr_name, dev);
   ifr.ifr_data = (void *) conf;

   sock = socket(AF_INET, SOCK_STREAM, 0);
   i = ioctl(sock, DLCI_SET_CONF, &ifr); 
   close(sock);

   if (i < 0)
   { 
      fprintf(stderr, "Error configuring device %s: %s.\n", dev, strerror(errno));
      exit(errno);
   }
}

void add(int dlci, char *dev, struct dlci_conf *conf, int quiet)
{
   int              i, sock;
   struct dlci_add  add;

   memset(&add, 0, sizeof(add));

   add.dlci = dlci;
   strcpy(add.devname, dev);

   sock = socket(AF_INET, SOCK_STREAM, 0);
   i = ioctl(sock, SIOCADDDLCI, &add); 

   close(sock);
 
   if (i < 0)
   { 
      fprintf(stderr, "Error creating/attaching DLCI %i to %s: %s.\n", dlci, dev, strerror(errno));
      exit(errno);
   }

   if (!quiet)
      printf("Allocated device %s for DLCI %i and attached to %s.\n", add.devname, dlci, dev);

   if (conf)
      config(add.devname, conf, quiet);
}

void del(char *dev, int quiet)
{
   int              i, sock;
   struct dlci_add  add;

   memset(&add, 0, sizeof(add));

   strcpy(add.devname, dev);
 
   sock = socket(AF_INET, SOCK_STREAM, 0);
   i = ioctl(sock, SIOCDELDLCI, &add); 
   close(sock);

   if (i < 0)
   { 
      fprintf(stderr, "Error deleting device %s: %s.\n", dev, strerror(errno));
      exit(errno);
   }

   if (!quiet)
      printf("Device %s deleted.\n", dev);
}

void cfgfile(char *filename, int quiet)
{
   struct dlci_conf def = {0, 64, 64, 0, 64, 64, 0};
   struct dlci_conf conf;
   void             *cfg_info;
   char             *temp, *dev;
   int              count, i, j, dcount, dlci, config, len;
   char             buf[32], flags[128];

   cfg_info = config_read(filename);
   if (!cfg_info)
   {
      fprintf(stderr, "Config file missing.\n");
      exit(-EINVAL);
   }

   temp = config_value(cfg_info, ATAG_MAIN, KTAG_DEVCOUNT);
   if (!temp)
   {
      fprintf(stderr, "Device count not found.\n");
      exit(-EINVAL);
   }

   count = atoi(temp);
   if (!quiet)
      printf("Reading configuration for %i devices.\n", count);

   for(i=1;i<=count;i++)
   {
      sprintf(buf, KTAG_DEV, i);
      dev = config_value(cfg_info, ATAG_MAIN, buf);
      if (!dev)
      {
         fprintf(stderr, "Device %i not found.\n", i);
         exit(-EINVAL);
      }

      if (!quiet)
         printf("Configuring device %s.\n", dev);

      temp = config_value(cfg_info, dev, KTAG_DLCICOUNT);
      if (temp)
      {
         dcount = atoi(temp);
         for(j=1;j<=dcount;j++)
         {
            sprintf(buf, KTAG_DLCI, j);
            temp = config_value(cfg_info, dev, buf);
            if (!temp)
            {
               fprintf(stderr, "DLCI %i not found for device %s.\n", j, dev);
               exit(-EINVAL);
            }
            dlci = atoi(temp);
            memcpy(&conf, &def, sizeof(def));

            sprintf(buf, ATAG_DLCI, i, dlci);

            config = 0;
            
            temp = config_value(cfg_info, buf, KTAG_F_CIR);
            if (temp)
            {
               conf.CIR_fwd = atoi(temp);
               config = 1;
            } 

            temp = config_value(cfg_info, buf, KTAG_B_CIR);
            if (temp)
            {
               conf.CIR_bwd = atoi(temp);
               config = 1;
            } 

            temp = config_value(cfg_info, buf, KTAG_F_BC);
            if (temp)
            {
               conf.Bc_fwd = atoi(temp);
               config = 1;
            } 

            temp = config_value(cfg_info, buf, KTAG_B_BC);
            if (temp)
            {
               conf.Bc_bwd = atoi(temp);
               config = 1;
            } 

            temp = config_value(cfg_info, buf, KTAG_F_BE);
            if (temp)
            {
               conf.Be_fwd = atoi(temp);
               config = 1;
            } 

            temp = config_value(cfg_info, buf, KTAG_B_BE);
            if (temp)
            {
               conf.Be_bwd = atoi(temp);
               config = 1;
            } 

            temp = config_value(cfg_info, buf, KTAG_DLCIFLAG);
            if (temp)
            {
               strcpy(flags, temp);
               for(temp = strtok(flags, ",");temp;temp = strtok(NULL, ","))
               {
                   len = strlen(temp);
                   if (strncasecmp(VFLAG_TXIGNORE, temp, len) == 0)
                   {
                      config = 1;
                      conf.flags |= DLCI_IGNORE_CIR_OUT;
                      continue;
                   }

                   if (strncasecmp(VFLAG_RXIGNORE, temp, len) == 0)
                   {
                      config = 1;
                      conf.flags |= DLCI_ACCOUNT_CIR_IN;
                      continue;
                   }

                   if (strncasecmp(VFLAG_BUFFERIF, temp, len) == 0)
                   {
                      config = 1;
                      conf.flags |= DLCI_BUFFER_IF;
                      continue;
                   }

                   fprintf(stderr, "Unknown or ambiguous flag %s for DLCI %i on device %s\n", temp, dlci, dev);
                   exit(-EINVAL);
               }
            }
            add(dlci, dev, config ? &conf : NULL, quiet);
         }
      }
   }
   config_clean(cfg_info);
}

int main(int argc, char *argv[])
{
   int              temp, cfg, quiet;
   struct dlci_conf conf = {0, 64, 64, 0, 64, 64, 0};
   char             c;

   cfg = quiet = 0;

   while ((c=getopt(argc, argv, "bBcCeErtqQ")) > 0)
      switch (c)
      {
         case 'r' : 
            conf.flags |= DLCI_IGNORE_CIR_OUT;
            cfg = 1;
            break;

         case 't' : 
            conf.flags |= DLCI_ACCOUNT_CIR_IN;
            cfg = 1;
            break;

         case 'q' :
            conf.flags |= DLCI_BUFFER_IF;
            cfg = 1;
            break;

         case 'b' :
            conf.Bc_fwd = atoi(optarg);
            cfg = 1;
            break;

         case 'c' :
            conf.CIR_fwd = atoi(optarg);
            cfg = 1;
            break;

         case 'e' :
            conf.Be_fwd = atoi(optarg);
            cfg = 1;
            break;

         case 'B' :
            conf.Bc_bwd = atoi(optarg);
            cfg = 1;
            break;

         case 'C' :
            conf.CIR_bwd = atoi(optarg);
            cfg = 1;
            break;

         case 'E' :
            conf.Be_bwd = atoi(optarg);
            cfg = 1;
            break;

         case 'Q' : 
            quiet = 1;
            break;

         default:
            usage(argv[0], argv[optind - 1]);
            break;
      }
  
   if (!quiet)
      printf("%s.\n", banner);

   if (argc <= optind)
      usage(argv[0], NULL);

   temp = strlen(argv[optind]);
   if ((strncasecmp(argv[optind], "add", temp) != 0) && 
       (strncasecmp(argv[optind], "conf", temp) != 0) && 
       (strncasecmp(argv[optind], "del", temp) != 0) &&
       (strncasecmp(argv[optind], "file", temp) != 0))
      usage(argv[0], argv[optind]);

   switch (*argv[optind++])
   {
      case 'a' :
         if (argc < optind + 2)
            usage(argv[0], NULL);
 
         if (argc > optind + 2)
            usage(argv[0], argv[optind + 1]);

          if (!isnumeric(argv[optind]))
            usage(argv[0], argv[optind]);

         temp = atoi(argv[optind]);

         if ((temp < 16) || (temp > 991))
            usage(argv[0], argv[optind]);

         if (strlen(argv[optind + 1]) > IFNAMSIZ)
            usage(argv[0], argv[optind + 1]);
 
         add(temp, argv[optind + 1], cfg ? &conf : NULL, quiet); 
         break;

      case 'c' :
         if (argc < optind) 
            usage(argv[0], NULL);
 
         if (argc > optind + 1)
            usage(argv[0], argv[optind+1]);
 
         if (strlen(argv[optind]) > IFNAMSIZ)
            usage(argv[0], argv[optind]);
 
         if (!cfg)
            usage(argv[0], NULL);

         config(argv[optind], &conf, quiet);
         break;

      case 'd' :
         if (argc < optind)
            usage(argv[0], NULL);

         if (argc > optind + 1)
            usage(argv[0], argv[optind+1]);

         if (strlen(argv[optind]) > IFNAMSIZ)
            usage(argv[0], argv[optind]);
 
         del(argv[optind], quiet);
         break;

      case 'f' :
         if (argc < optind)
            usage(argv[0], argv[optind]);

         if (argc > optind + 1)
            usage(argv[0], argv[optind+1]);

         if (cfg)
            usage(argv[0], NULL);

         cfgfile(argv[optind], quiet);
         break;
   }
   return(0);
}

