package freenet;
import freenet.crypt.EntropySource;
import freenet.session.*;
import freenet.support.*;
import freenet.support.io.*;
import freenet.thread.*;
//import freenet.transport.VoidAddress;
import java.io.*;
import java.io.EOFException;
/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU General Public Licence (GPL) 
  version 2.  See http://www.gnu.org/ for further details of the GPL.
 */

/**
 * Handles both sending and receiving messages on a connection.
 *
 * @author oskar (90%)
 * @author <A HREF="mailto:I.Clarke@strs.co.uk">Ian Clarke</A>
 * @author <a href="mailto:blanu@uts.cc.utexas.edu">Brandon Wiley</a>
 */

public class ConnectionHandler implements Runnable {

    // Protected/Private Fields

    private final OpenConnectionManager ocm;
    private final Link link;
    private final Presentation p;
    private final Ticker t;
    
    private int meanMessageSize;
    private long startTime;
    private int maxInvalid;
    private int maxPadding;

    private int messages = 0;

    protected boolean outbound;

    // state variables

    /** Last time of noted activity */
    private volatile long lastActiveTime;     // clock time of last activity

    
    /** Whether the ConnectionHandler has been started */
    private final Irreversible started = new Irreversible(false);
    
    /** If no more messages will be read */
    private final Irreversible receiveClosed = new Irreversible(false);
    
    /** If no more message can be sent */
    private final Irreversible sendClosed = new Irreversible(false);

    /** If we should never timeout the connection */
    private final Irreversible persist = new Irreversible(false);

    /** terminate() reached */
    private Irreversible finalized = new Irreversible(false); 


    /** Object to lock on when receiving */
    private final Object receiveLock=new Object();
    /** Count of number of number of reads going on (should be 0 or 1) */
    //private final Count receiving=new Count(0, receiveLock);
    // AFM (another fucking monitor)
    private volatile int receivingCount = 0;

    /** Object to lock on when sending */
    private final Object sendLock=new Object();
    /** Count objects for number of sends in progress of pending */
    //private final Count sending=new Count(0, sendLock);
    // AFM (another fucking monitor)
    private volatile int sendingCount = 0;
    

    private volatile long sendQueueSize = 0;
    //private Thread exec_instance; // execution thread
    private static EntropySource sendTimer=new EntropySource(), 
                                 recvTimer=new EntropySource(); // feed some entropy

    // Constructors
    /**
     * The ConnectionHandler provides the interface between the session,
     * presentation, and application layers. Messages received on l, are
     * parsed using p, and then turned into messageobjects using the 
     * MessageFactory which are scheduled for immediate execution on the 
     * ticker. Message objects given to the sendmessage are serialized
     * using the p and sent on l.
     * <p>
     * The outbound argument is a hint used by diagnostics to
     * differentiate inbound from outbound connections.
     * <p>
     * @param  ocm The cache of open connections to register with
     * @param  p   A presentation for parsing messages
     * @param  l   A live, initialized link, to read messages off
     * @param  t   A ticker to schedule the messages with.
     * @param  maxInvalid  The maximum number of invalid messages to swallow.
     * @param  maxPad      The maximum number of padding bytes to accept
     *                     between two messages.
     * @param  outbound Set true for outbound connections, false otherwise.
     */
    public ConnectionHandler(OpenConnectionManager ocm, 
                             Presentation p, Link l, Ticker t,
                             int maxInvalid, int maxPad, boolean outbound) {

        this.ocm = ocm;
        this.p = p;
        this.meanMessageSize = p.exptMessageSize();
        this.link = l;
        this.t = t;
        lastActiveTime = System.currentTimeMillis();
        this.maxInvalid = maxInvalid;
        this.maxPadding = maxPad;
        this.outbound = outbound;

        if (peerIdentity() != null) ocm.put(this);
        Peer peer = new Peer(link.getPeerIdentity(), link.getPeerAddress(),
                             link.getManager(), p);
        Core.logger.log(this, "New connectionhandler with "+peer, Logger.DEBUG);
    }


    public void start() {
        (new Thread(this, ""+this)).start();
    }

    // Public Methods
    public void run() {

        startTime = System.currentTimeMillis();

        //exec_instance = Thread.currentThread();

        try {
            started.change();
        } catch (IrreversibleException e) {
            Core.logger.log(this, "Can not restart connectionhandlers",
                            Logger.ERROR);
            throw new RuntimeException("C.H. already run once");  
            // ??? -tavin <- what's weird, this implies something is badly fucked
        }

        Core.logger.log(this, "Started listening for messages", 
                        Core.logger.DEBUG);
       
        lastActiveTime = System.currentTimeMillis();
            
        // Get the inputstream for messages (decrypted);
        PushbackInputStream in = 
            new PushbackInputStream(link.getInputStream());
        
        boolean wait;
        int padCount;
        int invalid = 0;

        
        //boolean close = false;
        
        // loop over each message
        message: do {
            
            padCount = 0;
            lastActiveTime = System.currentTimeMillis();
            try {
                link.setTimeout(3000);
            } catch (IOException e) {
                Core.logger.log(this, "Error resetting socket timeout",
                                e, Logger.MINOR);
                break message;
            }

            waiting: while (!receiveClosed.state()) {
                int i;
                try {
                    i = in.read();
                    if (i==-1) {
                        break message;
                    }
                    else if (i==0 && ++padCount >= maxPadding) {
                        Core.logger.log(this, "Too much padding.", Logger.DEBUG);
                        break message;
                    }
                    else {
                        in.unread(i);
                        break waiting;
                    }
                } catch (InterruptedIOException e) {
                    //if (sending.count() == 0 && !persist.state() && 
                    if (sendingCount == 0 && !persist.state() && 
                        System.currentTimeMillis() >= 
                        (lastActiveTime + Core.connectionTimeout)) {
                        
                        if (sendClosed.state()) {
                                // no reply within one timeout of sending
                                // the closing messages...
                            //break waiting;  -- we are certain to break message;
                            //                   a few lines down anyway
                            break message;
                        } else {
                                // timeout
                            close();
                            continue message;
                        }
                    }
                } catch (IOException e) {
                    Core.logger.log(this, "I/O error on socket", e, Logger.MINOR);
                    break message;
                }
            }
            
            if (receiveClosed.state()) 
                break message;

            boolean needToRemoveFromOCM = false;
            try {
                synchronized(receiveLock) {
                    // now that we are synced
                    if (receiveClosed.state()) break message;
                    
                    try {
                        // note the deliberate placement of these variables.
                        // so that the Message variables are inside, which keeps
                        // it possible for the finalizer on the ReceiveInputStream
                        // to be called while we are waiting on it (preventing
                        // deadlock if the stream is just ignored).
                        RawMessage m=null;
                        Message msg = null;

                        //receiving.incCount();
                        ++receivingCount;
                        link.setTimeout(600000);/* 10 minutes */
                    
                        // read message from protocol
                        m = p.readMessage(in);
                        if (m.close) {
                            needToRemoveFromOCM = true;
                            receiveClosed.change(true);
                        }
                        // avoid actually setting the receiveClosed flag
                        // until we've read the trailing field
                        //close = m.close;
                        //if (close) removeFromOCM();
                        if (m.trailingFieldLength > 0) {
                            m.trailingFieldStream = 
                                new ReceiveInputStream(in, m.trailingFieldLength);
                        }
                        if (m.sustain == true) {
                            persist.change(true);
                        }
                    
                        Core.randSource.acceptTimerEntropy(recvTimer);

                        if (Core.logger.shouldLog(Logger.DEBUG))
                            Core.logger.log(this, "Receiving RawMessage:\n"+m, 
                                            Logger.DEBUG);
                    
                        msg = t.getMessageHandler().getMessageFor(this, m);
                        invalid = 0; // succeeded this time.

                        // i think users like this one too..
                        if (Core.logger.shouldLog(Logger.MINOR))
                            Core.logger.log(this,
                                            m.messageType + " @ " 
                                            + Long.toHexString(msg.id())
                                            +" <- " + link.getPeerAddress(),
                                            Logger.MINOR);

                        // Schedule
                    
                        t.add(0, msg);
                        messages++;
                        if (m.trailingFieldLength > 0) // &&
                            //!((ReceiveInputStream) m.trailingFieldStream).done)
                            // ^^^ there is a risk of decrementing receivingCount twice
                            wait = true;
                        else {
                            //receiving.decCount();
                            --receivingCount;
                            wait = false;
                        }
                    } catch (InvalidMessageException e) {
                        Core.logger.log(this,"Invalid message: " + e.toString(),
                                        Logger.MINOR);
                        invalid++;
                        if (invalid >= maxInvalid) {
                            Core.logger.log(this, invalid + 
                                            " consecutive bad messages - closing.",
                                            Logger.DEBUGGING);
                            break message;
                        } else {
                            continue message; 
                        }
                    } catch (EOFException e) {
                        break message; // this stream is over
                    } catch (IOException e) {
                        break message;
                    }
                
                    if (wait) {
                        Core.diagnostics.occurrenceCounting("readLockedConnections",
                                                            1);
                        try {
                            receiveLock.wait();
                        } catch (InterruptedException e) {
                            Core.logger.log(this,"I got interrupted!",
                                            Logger.DEBUGGING);
                        }
                        Core.diagnostics.occurrenceCounting("readLockedConnections",
                                                            -1);
                        //Core.logger.log(this,"I am awake!",Logger.DEBUGGING);
                    }
                    //Core.logger.log(this,"Finished with message",
                    //                Logger.DEBUGGING);
                    lastActiveTime = System.currentTimeMillis();
                }
            }
            finally {
                if (needToRemoveFromOCM) {
                    removeFromOCM();
                }
            }
        } while (!receiveClosed.state());

        // Unlocked.  Groovy.
        removeFromOCM();
        receiveClosed.change(true);  // in case we break;'d out
            
        // should this be in terminate() instead?
        Core.diagnostics.occurrenceContinuous("connectionLifeTime",
                                             System.currentTimeMillis() 
                                             - startTime);
        Core.diagnostics.occurrenceContinuous("connectionMessages",
                                             messages);
        
        Core.logger.log(this, "Receiving terminated.", Logger.DEBUG);
                                                 
        if (sendClosed.state()) { // everybody done.
            // The race condition we need to worry about here is that
            // neither would call terminate(), but that can't happen
            // because both reading and writing sets themselves closed
            // before checking that the other is. So if the sending 
            // thread got that we were still open, it must have set 
            // itself close before we readed sendClosed.state()
            if (sendingCount == 0) terminate(); 
        } else {
            // note that close does nothing if p.getMessage is null   
            close();
        }
    }

    /**
     * Sends a message using this connection. This will return after the 
     * message headers have been sent.
     * @param m  the message to send. If m is null, the method will lock
     *           until m _could_ have been sent, and then return.
     * @return   the OutputStream to write the trailing field to
     *           if there is one, otherwise null.
     */
    public OutputStream sendMessage(Message m) throws SendFailedException {

        // non terminal SFEs if we just stopped sending on this pipe

        if (sendClosed.state())
            throw new SendFailedException(link.getPeerAddress(), false);

        boolean needToRemoveFromOCM = false;
        try {
            synchronized(sendLock) {
                if (sendClosed.state())  // check again when synced
                    throw new SendFailedException(link.getPeerAddress(), false);
            
                //int n = sending.incCount();
                OutputStream send = null;
                try {
                    ++sendingCount;
                    //if (sending.count() > 1) {
                    if (sendingCount > 1) {
                        long time = System.currentTimeMillis();
                        try {
                            sendLock.wait(60000);
                        } catch (InterruptedException e) {
                        }

                        if (sendingCount > 1 &&  // this can give false pos.
                            System.currentTimeMillis() - time >= 60000) {
                            Address peer = link.getPeerAddress();
                            Core.logger.log(this, "Waited one minute on " + 
                                            "connection to " 
                                            + peer + " (" + hashCode() +
                                            ") free up, but it didn't.",
                                            Core.logger.NORMAL);
                            throw new SendFailedException(peer, 
                                                          "Gave up wait");
                        } else if (sendClosed.state())  // check again 
                           throw new SendFailedException(link.getPeerAddress(),
                                                         false);
                    }

                    // if we were just waiting for the connection to become free..
                    if (m == null)
                        return null;
                
                    RawMessage raw = m.toRawMessage(p);
                    //if (raw == null) {
                    //    return null;
                    //}
                    if (raw.close) {
                        needToRemoveFromOCM = true;
                        sendClosed.change(true);
                        //} else if (raw.sustain) {
                        //    persist.change(true);
                        // hm, we only want to persist if we *receive*
                        // a sustain message, right?
                    }
                
                    sendQueueSize += meanMessageSize + raw.trailingFieldLength;
                
                    Core.logger.log(this, sendingCount + " messages in sendqueue",
                                    Core.logger.DEBUG);
                
                    // dont need now
                    //raw.close = (sendClosed.state() && sendingCount <= 1);
                    // defensive coding .. b/c i'm tired
                    // of puzzling over this

                    if (Core.logger.shouldLog(Logger.DEBUG))
                        Core.logger.log(this, "Sending RawMessage:\n"+raw, 
                                        Logger.DEBUG);
                    
                    // users are fond of this log entry
                    if (Core.logger.shouldLog(Logger.MINOR))
                        Core.logger.log(this,
                                        raw.messageType + " @ " + 
                                        Long.toHexString(m.id())
                                        +" -> " + link.getPeerAddress(),
                                        Logger.MINOR);
            
                    OutputStream mout;
                    try {
                        mout = link.getOutputStream();
                        raw.writeMessage(mout);
                        mout.flush();
                        messages++;
                    } catch (IOException e) {
                        needToRemoveFromOCM = true;
                        sendClosed.change(true);
                        // terminal failure
                        throw new SendFailedException(link.getPeerAddress(), true);
                    }
    
                    //sendQueueSize -= Math.min(meanMessageSize, sendQueueSize);
                    // i am ever so slightly worried this syntax is bad with volatiles
                    sendQueueSize = Math.max(sendQueueSize - meanMessageSize, 0);
     
                    if (raw.trailingFieldLength > 0) { // has trailing
                        send = new SendOutputStream(mout, raw.trailingFieldLength, 
                                                    raw.close);
                    } else { // does not
                        //Core.logger.log(this, "Message sent.", Logger.DEBUG);
                        Core.randSource.acceptTimerEntropy(sendTimer);
                        lastActiveTime = System.currentTimeMillis();
                    }
                    return send;
                }
                finally {
                    // only if there is a trailing field to write do we not
                    // decrement sendingCount and notify() immediately
                    if (send == null) {
                        //sending.decCount(); // done
                        --sendingCount;
                        if (receiveClosed.state() && receivingCount == 0
                            && sendClosed.state() && sendingCount == 0) {
                            terminate();
                        }
                        sendLock.notify();
                    }
                }
            }
        }
        finally {
            if (needToRemoveFromOCM) {
                removeFromOCM();
            }
        }
    }

    // IMPORTANT: Don't call this while holding sendLock
    //            or receiveLock or you will cause a
    //            deadlock.
    //
    /** Removes this CH from the OCM.
     */
    private final void removeFromOCM() {
        if (peerIdentity() != null) ocm.remove(this);
    }
    
    /**
     * Initiates a close dialog on this connection by sending a 
     * closeMessage() as specified by the Presentation object if one can
     * be created.
     */
    void close() {
        Message m = p.getCloseMessage();
        if (m != null) {
            try {
                OutputStream out = sendMessage(m);
                if (out != null) // wtf...
                    out.close();
            } catch (IOException e) {
                // wtf wtf...
            } catch (SendFailedException e) {
                Core.logger.log(this, 
                                "Failed to send connection close message: "
                                + m, Core.logger.MINOR);
            }
        } 
    }

    /**
     * Closes the connection utterly and finally.
     */
    public void terminate() {
        try {
            finalized.change();
        } catch (IrreversibleException e) {
            return; // terminate called twice
        }

        if (!sendClosed.state()) {
            // ???  Is there a codepath that leaves sendClose.state()
            // ???  true without removing the connection?
            //removeFromOCM();
            synchronized (sendLock) {          
                sendClosed.change(true);
                sendLock.notifyAll();
            }
        }
        if (!receiveClosed.state()) {
            synchronized (receiveLock) {
                receiveClosed.change(true);           
                receiveLock.notify();
            }
        }
        try {
            link.close();
        } catch (IOException e) {
        }

        // Unconditionally remove the connection handler
        // from the OCM
        removeFromOCM();
        Core.logger.log(this, "ConnectionHandler closed!", Logger.DEBUG);

        //if (exec_instance!=null)
        //    exec_instance.interrupt();
    }

    /**
     * Checks whether the connection is alive and can send messages.
     */
    public final boolean isOpen() {
        return !sendClosed.state();
    }

    int sendingCount() {
        return sendingCount;
    }

    /**
     * Returns the number milliseconds since this connection was active.
     */
    public final long idleTime() {
        //return (sending.count() > 0 || receiving.count() > 0 ? 
        return (sendingCount > 0 || receivingCount > 0 ? 
                0 : System.currentTimeMillis() - lastActiveTime); 
    }

    /**
     * @return the number of bytes waiting to be sent on this connection
     */
    public final long sendQueueSize() {
        //return sendQueueSize(); -- any chance this could have led to an infinite
        //                           loop in OCM that made it never release its
        //                           monitor?
        return sendQueueSize;
    }

    /**
     * @return whether the connection is currently sending something
     */
    public final boolean sending() {
        //return sending.count() > 0;
        return sendingCount > 0;
    }

    /**
     * @return whether the connection is current receiving something
     */
    public final boolean receiving() {
        return receivingCount > 0;
    }

    /** @return  identity of the Peer on the other end of the connection
      */
    public final Identity peerIdentity() {
        return link.getPeerIdentity();
    }

    /** @return  the Transport used for this connection
      */
    public final Transport transport() {
        return link.getPeerAddress().transport();
    }

    public final LinkManager sessionType() {
        return link.getManager();
    }
    
    public final Address peerAddress() {
        return link.getPeerAddress();
    }

    public final Presentation presentationType() {
        return p;
    }

    public final long runTime() {
        return System.currentTimeMillis() - startTime;
    }

    public final long messages() {
        return messages;
    }

    protected void finalize() throws Throwable {
        if (!finalized.state()) {
            Core.logger.log(this, "I wasn't terminated properly! Doing it now..",
                            Logger.ERROR);
            terminate();
        }
    }

    
    
    

    //=========================================================================
    // the rest is inner classes for streams that handle the trailing
    // fields when sending/receiving messages
    //=========================================================================
    
    /**
     * An InputStream that is allows reading of a limited number of bytes,
     * and unlocks the receiving when that many bytes are read or it is closed.
     */
    private class ReceiveInputStream extends DiscontinueInputStream {

        private final long toRead;
        private long read = 0;
        private boolean done = false;
        
        public ReceiveInputStream(InputStream in, long toRead) {
            super(in);
            this.toRead = toRead;
        }

        public int read() throws IOException {
            try {
                if (read >= toRead) {
                    return -1;
                }
                int i = in.read();
                if (i > -1)
                    read++;
                if (read == toRead)
                    done();
                return i;
            }
            catch (IOException e) {
                close();
                throw (IOException) e.fillInStackTrace();
            }
        }

        public int read(byte[] b, int off, int length) throws IOException {
            try {
                if (read >= toRead) {
                    return -1;
                } else if (read + length > toRead) {
                    length = (int) (toRead - read);
                }
                // System.err.println("CONN STREAM READING SUPER IN: " + in.toString());
                int i = in.read(b, off, length);
                if (i > -1)
                    read += i;
                if (read == toRead)
                    done();
                return i;
            }
            catch (IOException e) {
                close();
                throw (IOException) e.fillInStackTrace();
            }
        } 
        
        public void close() throws IOException {
            //if (read < toRead) {
            //    // not good, bad data present on Connection
            //    Core.logger.log(ConnectionHandler.this, "Close after " + read 
            //                    + " of " + toRead + " bytes", Logger.MINOR);
            //}
            done();
            //discontinue();
            //super.close();
        }

        public final void discontinue() throws IOException {
            read = toRead;
            done();
        }

        private void done() {
            if (!done) {
                synchronized(receiveLock) {
                    done = true;
                    lastActiveTime = System.currentTimeMillis();
                    //receiving.decCount();
                    if (read < toRead)               // can't read any more
                        receiveClosed.change(true);  // messages on this conn.
                    --receivingCount;
                    receiveLock.notify(); // wake read thread
                }
                if (read < toRead) {
                    Core.logger.log(ConnectionHandler.this, "Close after "+read 
                                    +" of "+toRead+" bytes", Logger.MINOR);
                }
                else {
                    Core.logger.log(ConnectionHandler.this, 
                                    "Trailing field fully received.",
                                    Logger.DEBUG);
                }
            }
        }

        protected void finalize() throws Throwable {
            if (!done) {
                Core.logger.log(ConnectionHandler.this,
                    "I was finalized without being properly deallocated: "+this,
                    Logger.ERROR);
                done();
            }
            super.finalize();
        }
    }

    /**
     * An OutputStream that only allows writing of so many bytes,
     * and that releases the sendLock that many bytes are written
     * or it closed.
     */
    private class SendOutputStream extends FilterOutputStream {

        private long written = 0;
        private final long toWrite;
        private boolean close;
        private boolean done = false;

        public SendOutputStream(OutputStream out, long toWrite, 
                                boolean close) {
            super(out);
            this.toWrite = toWrite;
            this.close   = close;
        }

        public void write(int i) throws IOException {
            try {
                if (written >= toWrite)
                    throw new IOException("Overwriting: " + written +
                                          " + 1 > " + toWrite);
                out.write(i);
                written++;
                sendQueueSize = Math.max(sendQueueSize - 1, 0);
                if (toWrite == written)
                    done();
            }
            catch (IOException e) {
                close();
                throw (IOException) e.fillInStackTrace();
            }
        }

        public void write(byte[] b, int off, int length) throws IOException {
            try {
                if (written + length > toWrite)
                    throw new IOException("Overwriting: " + written + " + " + length
                                          + " > " + toWrite);
                out.write(b, off, length);
                written += length;
                //sendQueueSize -= Math.min(length,sendQueueSize);
                // i am ever so slightly worried this syntax is bad with volatiles
                sendQueueSize = Math.max(sendQueueSize - length, 0);
                if (written == toWrite)
                    done();
            }
            catch (IOException e) {
                close();
                throw (IOException) e.fillInStackTrace();
            }
        }

        public void close() throws IOException {
            if (!done && written < toWrite) {
                Core.logger.log(ConnectionHandler.this,
                    "Close after "+written+" of "+toWrite+" bytes on: "+this,
                    Logger.MINOR);
            }
            //sendQueueSize -= Math.min((toWrite - written), sendQueueSize);
            // i am ever so slightly worried this syntax is bad with volatiles
            sendQueueSize = Math.max(sendQueueSize - (toWrite - written), 0);
            written = toWrite;
            done();
        }

        public void done() {
            if (!done) {
                done = true;
                try {
                    out.flush();
                }
                catch (IOException e) {
                    Core.logger.log( ConnectionHandler.this,
                                     "I/O error when flushing SendOutputStream",
                                     e, Logger.MINOR );
                }
                synchronized(sendLock) {
                    //sending.decCount(); // done!
                    --sendingCount;
                    //if (close) {
                    //    removeFromOCM();
                    //    sendClosed.change(true);
                    //}
                    // ^^^ this has already happened
                    if (receiveClosed.state() && receivingCount == 0
                        && sendClosed.state() && sendingCount == 0) {
                        terminate();
                    }
                    
                    Core.logger.log(ConnectionHandler.this, 
                                    "Message and trailing field sent.",
                                    Logger.DEBUGGING);
                    
                    Core.randSource.acceptTimerEntropy(sendTimer);
                    
                    lastActiveTime = System.currentTimeMillis();
                    sendLock.notify(); // wake a possible waiting thread. 
                }
            }
        }

        /* Removed because it prevents early GCing of these objects.
        protected void finalize() throws Throwable {
            if (!done) {
                Core.logger.log( ConnectionHandler.this,
                    "I was finalized without being properly deallocated: "+this,
                    Logger.ERROR);
                done();
            }
            super.finalize();
        }
        */
    }
}







