package freenet;

import freenet.support.io.DiscontinueInputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.IOException;
import java.io.EOFException;

/**
 * Implementations of Presentation are used to handle connections
 * with a certain message protocol. It is used used to initialize
 * new connections, read messages off the connection into the RawMessage 
 * format, and creating new RawMessages for writing to a stream with
 * this protocol.
 **/

public abstract class Presentation {

    /**
     * These are control byte values used in the streams of all presentations 
     **/
    public static final int
        CB_OK        = 0x00,
        CB_RESTARTED = 0x01,  // parallels QueryRestarted
        CB_ABORTED   = 0x02;  // parallels QueryAborted

    //Values over 128 are internal
    public static final int 
        CB_BAD_DATA       = 0x81,
        CB_SEND_CONN_DIED = 0x82,
        CB_RECV_CONN_DIED = 0x83,
        CB_BAD_KEY        = 0x84,
        CB_CACHE_FAILED   = 0x85,
        CB_CANCELLED      = 0x86;
    
    public static final String getCBname(int cb) {
        switch (cb) {
            case CB_OK:                 return "CB_OK";
            case CB_RESTARTED:          return "CB_RESTARTED";
            case CB_ABORTED:            return "CB_ABORTED";
            case CB_BAD_DATA:           return "CB_BAD_DATA";
            case CB_SEND_CONN_DIED:     return "CB_SEND_CONN_DIED"; 
            case CB_RECV_CONN_DIED:     return "CB_RECV_CONN_DIED";
            case CB_BAD_KEY:            return "CB_BAD_KEY";
            case CB_CACHE_FAILED:       return "CB_CACHE_FAILED";
            case CB_CANCELLED:          return "CB_CANCELLED";
            default:                    return "Unknown control byte";
        }
    }

    public static final String getCBdescription(int cb) {
        return "0x"+Integer.toHexString(cb)+" ("+getCBname(cb)+")";
    }

        
    /**
     * Returns the designator number of this Presentation type.
     */
    public abstract int designatorNum();

    /**
     * Creates a new RawMessage of a stream by reading this presentation. This
     * method locks until an entire message has been read (excluding trailing)
     * @param in    The stream to read from.
     * @return      A new raw message.
     **/
    public abstract RawMessage readMessage(InputStream in) throws InvalidMessageException, EOFException;

    /** 
     * Creates a new RawMessage of a given type that uses this presentation
     * @param messageType   The name of the message type
     * @param close         Whether to keep alive the connection after 
     *                      receiving or sending this message.
     * @param sustain       Whether to sustain the connection.
     * @param fs            A set of message specific fields.
     * @param trailingLength The length of the trailing data, or 0 if there 
     *                       is no trailing data
     * @param trailingName   The name of the trailing field, or null if there
     *                       is no trailing data
     * @param data           An inputstream containing the trailing data,
     *                       straight off the DataStore (decrypted).
     * @return     A new raw message
     **/
    public abstract RawMessage newMessage(String messageType, boolean close, 
                                          boolean sustain, FieldSet fs, 
                                          long trailingLength, 
                                          String trailingName, 
                                          DiscontinueInputStream data);

    /**
     * An estimation of the avergae size of a message (not including
     * trailing) encoded in this presenation in bytes
     */
    public abstract int exptMessageSize();

    /**
     * Subclasses can override this to put a default Message on the stack
     * if the RawMessage is not recognized by the MessageHandler.
     */
    public Message getDefaultMessage() {
        return null;
    }

    /**
     * Subclasses can override this to put have a message get sent when 
     * a Connection has timed out and is ready to close. If a message is
     * returned from this, it will be sent as a close token, and the 
     * connection will not be forced close, otherwise it will.
     */
    public Message getCloseMessage() {
        return null;
    }

    /**
     * Subclasses that support setting connections to be sustained can override
     * this to return a message that is sent down a Connection and sets
     * Sustain to true.
     */
    public Message getSustainMessage() {
        return null;
    }

}


