// REDFLAG: put ref export support back in.
package freenet.client.http;

import java.io.*;
import java.net.*;
import java.util.*;
import java.text.DateFormat;
import javax.servlet.*;
import javax.servlet.http.*;

import freenet.ContactCounter;
import freenet.support.Logger;
import freenet.Core;
import freenet.Key;
import freenet.Address;
import freenet.BadAddressException;
import freenet.FieldSet;
import freenet.node.Node;
import freenet.node.NodeReference;
import freenet.node.rt.RTDiagSnapshot;
import freenet.node.LoadStats;
import freenet.diagnostics.Diagnostics;
import freenet.diagnostics.DiagnosticsFormat;
import freenet.diagnostics.HtmlDiagnosticsFormat;
import freenet.diagnostics.HtmlIndexFormat;
import freenet.diagnostics.RowDiagnosticsFormat;
import freenet.diagnostics.FieldSetFormat;
import freenet.node.rt.RoutingTable;
import freenet.support.servlet.http.HttpServletResponseImpl;
import freenet.support.io.WriteOutputStream;
import freenet.support.io.ReadInputStream;
import freenet.support.URLEncodedFormatException;
import freenet.support.ArrayBucket;
import freenet.support.Bucket;
import freenet.support.sort.*;
import freenet.support.Comparable;
import freenet.support.StringMap;
import freenet.support.SimpleStringMap;


/*
  This code is distributed under the GNU Public Licence (GPL)
  version 2.  See http://www.gnu.org/ for further details of the GPL.
*/

/**
 * Servlet to display and allow downloading
 * of node references as the node is running.
 * <p>
 * Example freenet.conf segment to run this servlet:
 * <pre>
 * nodestatus.class=freenet.client.http.NodeStatusServlet
 * # Change port number if you like.
 * nodestatus.port=8889
 * # Make sure that the servlet is listed in the services line.
 * services=fproxy,nodestatus
 * </pre>
 * <p>
 * @author giannij
 **/
public class NodeStatusServlet extends HttpServlet  {
	private long firstAccess = -1;

    public void init() {
        ServletContext context = getServletContext();
        node = (Node)context.getAttribute("freenet.node.Node");
        if (node != null) {
            rt = node.rt;
            diagnostics = node.diagnostics;
            inboundContacts = node.inboundContacts;
            outboundContacts = node.outboundContacts;
            inboundRequests = node.inboundRequests;
            outboundRequests = node.outboundRequests;
            loadStats = node.loadStats;
        }
    }

    protected void doGet(HttpServletRequest req, HttpServletResponse resp)
	throws ServletException, IOException
    {
        if ((rt == null)) {
            sendError(resp, HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
                      "Couldn't access the freenet.node.Node instance.  " +
                      "This servlet must be run in the same JVM as the node.");
            return;
        }

        try {
            String uri = freenet.support.URLDecoder.decode(req.getRequestURI());
            String baseURL = req.getContextPath() + req.getServletPath();
            if ((baseURL.length() > 0) && (!baseURL.endsWith("/"))) {
                baseURL += "/";
            } 

            if (uri.endsWith("/loadStats.txt")) {
                sendLoadStats(resp);
                return;
            }
            
            if (uri.endsWith("/inboundContacts.txt")) {
                sendPerHostStats(resp, "inboundContacts");
                return;
            }

            if (uri.endsWith("/outboundContacts.txt")) {
                sendPerHostStats(resp, "outboundContacts");
                return;
            }

            if (uri.endsWith("/inboundRequests.txt")) {
                sendPerHostStats(resp, "inboundRequests");
                return;
            }

            if (uri.endsWith("/outboundRequests.txt")) {
                sendPerHostStats(resp, "outboundRequests");
                return;
            }

            if (uri.endsWith("key_histogram.txt")) {
                sendKeyHistogram(resp, false);
                return;
            }

            if (uri.endsWith("key_histogram_data.txt")) {
                sendKeyHistogram(resp, true);
                return;
            }

            if (uri.endsWith("noderefs.txt")) {
                sendRefList(req, resp);
            }
            else if (uri.endsWith("nodestatus.html")) {
                sendStatusPage(resp);
            }
            else if (uri.endsWith("tickerContents.html")) {
                sendTickerContents(resp);
            }
            else if (uri.endsWith("ocmContents.html")) {
                sendOcmContents(resp);

            } else if (uri.endsWith("diagnostics/index.html")) {
                sendDiagnosticsIndex(req, resp, baseURL);
            }
            else if (uri.indexOf(baseURL + "diagnostics/") != -1) {
                // REDFLAG: clean up handling of bad urls
                int pos = uri.indexOf(baseURL + "diagnostics/") +
                                      (baseURL + "diagnostics/").length();
                int endName = uri.indexOf("/", pos);
                String varName = uri.substring(pos, endName);
                String period = uri.substring(endName + 1);
                sendVarData(req, resp, 
                            varName, period);
            }
            else {
                sendIndexPage(resp, baseURL);
            }
        }
        catch (URLEncodedFormatException uee) {
            // hmmm... underwhelming
            throw new IOException(uee.toString());
        }
    }

    private void reportNodeStats(PrintWriter pw) throws IOException {
        if (node == null) {
            return;
        }

        pw.println("<li> Uptime: &nbsp " + getUptime() + " <br> "); 

        // Thread load
        int jobs = node.activeJobs();

        // It's not just thread based.  There's also a hard
        // rate limit.
        boolean rejectingRequests = node.rejectingRequests();

        if (jobs > -1) {
            String color = "black";
            String comment = "";
            if (jobs >= Node.maximumThreads) {
                color = "red"; 
                comment = " &nbsp <b> [Rejecting incoming connections and requests!] </b> ";
            }
            else if (rejectingRequests) {
                color = "red"; 
                comment = " &nbsp <b> [QueryRejecting all incoming requests!] </b> ";
            }

            float threadPoolLoad = (((float)jobs) / Node.maximumThreads) * 100.0f;

            String msg = " " + 
                Integer.toString(jobs) +
                " &nbsp &nbsp (" + 
                Float.toString(threadPoolLoad) + "%) " +
                " <font color=\"" + color + "\"> " +
                comment + " </font>  <br>";
            
            pw.println("<li> Active pooled jobs: " + msg); 

            if ((jobs >= Node.maximumThreads) || rejectingRequests) {
                pw.println("It's normal for the node to sometimes reject connections or requests");
                pw.println("for a limited period.  If you're seeing rejections continuously the node");
                pw.println("is overloaded or something is wrong (i.e. a bug).");
            }

            pw.println(" <p> ");

        }

        pw.println("");
    }


    private final static String appendIntervalToMsg(long count, 
                                                    String msg,
                                                    String singular, String plural ) {

        if (count == 1) {
            return msg + " " + Long.toString(count) + " " + singular;
        }
        else {
            return msg + " " + Long.toString(count) + " " + plural;
        }
    }

    private String getUptime() {

        long deltat = (System.currentTimeMillis() - Node.startupTimeMs) / 1000;

        long days = deltat / 86400l;

        deltat -= days * 86400l;

        long hours = deltat / 3600l;

        deltat -= hours * 3600l;

        long minutes = deltat/ 60l;

        String msg = null;
        msg = appendIntervalToMsg(days, "", "day, &nbsp ", "days, &nbsp");
        msg = appendIntervalToMsg(hours, msg, "hour, &nbsp ", "hours, &nbsp ");
        msg = appendIntervalToMsg(minutes, msg, "minute", "minutes");
        
        if (msg.equals("")) {
            return " &nbsp < 1 minute ";
        }

        return msg;
    }

    private void sendIndexPage(HttpServletResponse resp, String baseURL) throws IOException {
	resp.setStatus(HttpServletResponse.SC_OK);
	resp.setContentType("text/html");
        PrintWriter pw = resp.getWriter();

        pw.println("<html>");
	pw.println("<head>");
	pw.println("<title>");
	pw.println("Node Status Info");
	pw.println("</title>");
	pw.println("</head>");
        pw.println("<body>");
	pw.println("<h1>Node Status Info</h1>");

        pw.println("<ul>");

        // Knows to send list items.
        reportNodeStats(pw);

        pw.println("    <li> <a href=\"" + baseURL +"nodestatus.html\"> Node Reference Status </a><p>");
        pw.println("    <li> Reference Table info ");
        pw.println("        <ul>");
        pw.println("            <li> <a href= \"" + baseURL +"key_histogram.txt\">Histogram</a>");
        pw.println("                 of the keys in the node's reference table. ");
        pw.println("            <li> <a href= \"" + baseURL +"key_histogram_data.txt\">gnuplottable " +
                   "frequency data</a> for the histogram.");
        pw.println("        </ul> <p>");
        pw.println("    <li> <a href=\"" + baseURL +"diagnostics/index.html\"> Diagnostics Values </a> <p>");
        pw.println("    <li> <a href=\"" + baseURL +"inboundContacts.txt\"> Inbound Contact Attempts </a> <br>");
        pw.println("         This shows the per host statistics for inbound connections to the");
        pw.println("         node's FNP interface. <p>");
        pw.println("    <li> <a href=\"" + baseURL +"outboundContacts.txt\"> Outbound Contact Attempts </a> <br>");
        pw.println("         This shows the per host statistics for outbound FNP connections that");
        pw.println("         the node attempts to make to other nodes. There are a couple of things");
        pw.println("         to keep in mind when comparing these numbers to the connection ");
        pw.println("         statistics under <a href=\"" + baseURL +"nodestatus.html\"> Node Reference Status </a>.");
        pw.println("         <p> First these statistics are for <em>physical connections</em> attempts");
        pw.println("         in contrast to the NRS numbers which include contacts made over cached");
        pw.println("         connections.");
        pw.println("         <p> Second, the NRS statistics are only for outbound routing attempts");
        pw.println("         whereas these numbers include all outbound connections.");
        pw.println("         <p>");

        pw.println("    <li> <a href=\"" + baseURL +"inboundRequests.txt\"> Inbound Requests </a> <br>");
        pw.println("         This shows per host aggregated (DataRequest, ");
        pw.println("         InsertRequest, AnnouncementRequest) inbound requests received on the node's");
        pw.println("         FNP interface.");
        pw.println("<p>");

        pw.println("    <li> <a href=\"" + baseURL +"outboundRequests.txt\"> Outbound Requests </a> <br>");
        pw.println("         This shows per host aggregated (DataRequest, ");
        pw.println("         InsertRequest, AnnouncementRequest) requests that your node sent to");
        pw.println("         other nodes.");
        pw.println("<p>");
        pw.println("    <li> <a href=\"" + baseURL +"tickerContents.html\"> Ticker Contents </a> <br>");
        pw.println("         This shows events scheduled on the nodes ticker.");
        pw.println("<p>");
        pw.println("    <li> <a href=\"" + baseURL +"ocmContents.html\"> Connection Manager Contents </a> <br>");
        pw.println("         This shows the connections cached in the nodes Open Connection Manager.");

        pw.println("<p>");
        pw.println("    <li> <a href=\"" + baseURL +"loadStats.txt\"> Global network load stats </a> <br>");
        pw.println("         This shows hourly inbound request rates from other nodes.  ");
        pw.println("         For now it's for informational purposes only.  Eventually these data may be");
        pw.println("         used to implement intra-node load balancing.");
        pw.println("<p>");
        pw.println("</ul>");
        pw.println("</body>");
        pw.println("</html>");
        resp.flushBuffer();
    }

    private final static String drawLine(int binNum, int freq) {
        String ret = Integer.toString(binNum, 16) + " |";
        for (int i = 0; i < freq; i++) {
            ret += "=";
        }
        return ret;
    }

    private final static int[] fillBins(Key[] keys) {
        int[] bins = new int[256];
        int i = 0;
        for (i = 0; i < keys.length; i++) {
            final String keyAsString = keys[i].toString();
            
            // most significant byte.
            final int binNumber = (((int)keys[i].getVal()[0]) & 0xff);

            bins[binNumber]++;
        }
        return bins;
    }


    private String peakValue(int[] bins, int index, float mean) {
        // hmmm... Allow edges to count as peaks?
        int nextCount = 0;
        int prevCount = 0;

        if (index > 0) {
            prevCount = bins[index - 1];
        }

        if (index < bins.length - 1) {
            nextCount = bins[index + 1];
        }

        if ((bins[index] > prevCount) &&
            (bins[index] > nextCount)) {
            return Integer.toString(index, 16) + " --> (" + (((float)bins[index]) / mean) + ")\n";
        }

        return null;
    }


    private void sendStats(PrintWriter pw, int[] bins) {
        int max = 0;
        int sum = 0;
        int i = 0;
        for (i = 0; i < bins.length; i++) {
            if (bins[i] > max) {
                max = bins[i];
            }
            sum += bins[i];
        }
        
        float mean = ((float)sum) / bins.length;

        pw.println("mean: " + mean);
        pw.println("");

        if (mean < 1.0f) {
            return;
        }

        String text = "";
        for (i = 0; i < bins.length; i++) {
            String peakValue = peakValue(bins, i, mean);
            if (peakValue != null) {
                text += peakValue; 
            }
        }

        if (!text.equals("")) {
            pw.println("peaks (count/mean)");
            pw.println(text);
        }
    }

    private void sendKeyHistogram(HttpServletResponse resp, boolean justData) throws IOException {
        final Key[] keys = rt.getSnapshot().keys();

        int[] bins = fillBins(keys);        

        int count = 0;
        int i = 0;
        for (i = 0; i < bins.length; i++) {
            count += bins[i];
        }

        DateFormat df = DateFormat.getDateTimeInstance();
        String date = df.format(new Date(System.currentTimeMillis()));

        resp.setStatus(HttpServletResponse.SC_OK);
  	resp.setContentType("text/plain");
  	PrintWriter pw = resp.getWriter();
        
        if (justData) {
            pw.println("# Histogram of keys in in fred's reference table");
            pw.println("# " + date);
            pw.println("# keys: " + count + " skipped: " + (keys.length - count));

        }
        else {
            pw.println("Histogram of keys in in fred's reference table");
            pw.println(date);
            pw.println("keys: " + count + " skipped: " + (keys.length - count));
            pw.println("");
        }
        for (i = 0; i < bins.length; i++) {
            if (justData) {
                pw.println(i + "\t" + bins[i]);
            }
            else {
                pw.println(drawLine(i, bins[i]));
            }
        }

        pw.println("");

        sendStats(pw, bins);

        resp.flushBuffer(); 
    }

    private void sendStatusPage(HttpServletResponse resp) throws IOException {
        long now = System.currentTimeMillis();
        DateFormat df = DateFormat.getDateTimeInstance();
        String date = df.format(new Date(now));

        RTDiagSnapshot status = rt.getSnapshot();
        final StringMap tableData = status.tableData();
        final StringMap[] refData = status.refData();
        String typeName = null;
        
        if (tableData != null) {
            typeName = (String)tableData.value("Implementation");
        }

  	resp.setStatus(HttpServletResponse.SC_OK);
  	resp.setContentType("text/html");
  	PrintWriter pw = resp.getWriter();

        pw.println("<html>");
  	pw.println("<head>");
  	pw.println("<title>");
  	pw.println("Routing Table status: " + date);
  	pw.println("</title>");
  	pw.println("</head>");
        pw.println("<body>");
  	
  	pw.println("<h1>Routing Table status: " + date + " </h1>");
        pw.println("<p>");

        makePerTableData(status, tableData, pw);

        makeRefDownloadForm(status, tableData, refData, pw);

        pw.println("<p>");

  	pw.println("<table border>"); 
        
        makeTableHeader(status, tableData, refData,  pw);

        makeRefRowEntries(status, refData, typeName, pw);
        
        pw.println("</table>");
        
  	pw.println("</body>");
  	pw.println("</html>");

  	resp.flushBuffer();
    }



    private void makePerTableData(RTDiagSnapshot status, StringMap data, PrintWriter pw) {

        final String[] keys = data.keys();
        final Object[] values = data.values();
        if ((keys == null) && (values == null)) {
            return;
        }
        
        for (int i=0; i < keys.length; i++) {
            pw.println(keys[i] + ": " + values[i] + " <br>");
        }
    }

    private final String PROP_CP = "Contact Probablitity";
    private final String PROP_CONNECTIONS = "Successful Connections";
    private final String PROP_NODEREF = "NodeReference";

    private void makeRefDownloadForm(RTDiagSnapshot status, StringMap tableData, 
                                     StringMap[] refs, PrintWriter pw) {

        if ((refs == null) || (refs.length < 1)) {
            return;
        }
        
        boolean hasCPs = refs[0].value(PROP_CP) != null;
        boolean hasConns = refs[0].value(PROP_CONNECTIONS) != null;

        if (!(hasCPs || hasConns)) {
            return;
        }

        pw.println("<form action=\"/noderefs.txt\" method=\"Get\">");

        if (hasCPs) {
            pw.println("<b>Minimum contact probability:</b> ");
            pw.println(" <input size=\"5\" name=\"minCP\" value=\"0.0\"> <br>");
        }

        if (hasConns) {
            pw.println("<b>Minimum successful connections:</b> ");
            pw.println(" <input size=\"5\" name=\"minConnections\" value=\"1\"> <p> ");
        }

        pw.println(" <input type=\"submit\" value=\"Download References\">");
        pw.println("</from>");
    }

    private void makeRefRowEntries(RTDiagSnapshot status, StringMap[] refs, 
                                   String typeName, PrintWriter pw) {
        
        if ((refs == null) || (refs.length < 1)) {
            return;
        }

        for (int i = 0; i < refs.length; i++) {
            final String[] keys = refs[i].keys();
            final Object[] values = formatRef(typeName, refs[i].values());
            pw.println("<tr>");
            for (int j = 0; j < values.length; j++) {
                // Skip noderef values since we can't render them.
                if (keys[j].equals("NodeReference")) {
                    continue;
                }

                pw.println("    <td> " + values[j] + " </td>");

            }
            pw.println("</tr>");
        }
    }

    private void makeTableHeader(RTDiagSnapshot status, StringMap tableData, 
                                 StringMap[] refData, PrintWriter pw) {
        final StringMap[] refs = status.refData();
        
        if ((refs == null) || (refs.length < 1)) {
            return;
        }

        pw.println("<tr>");
        final String[] keys = refs[0].keys();

        if (keys == null) {
            pw.println("</tr>");
            return;
        }

        for (int j = 0; j < keys.length; j++) {
            // Skip noderef values since we can't render them.
            if (keys[j].equals("NodeReference")) {
                continue;
            }
            pw.println("    <td> <b> " + keys[j] + " </b> </td>");
        }
        pw.println("</tr>");
    }

    private void sendRefList(HttpServletRequest req, HttpServletResponse resp) 
        throws IOException {
        float minCP = (float)0.0;
        try {
            String minCPAsString = req.getParameter("minCP");
            if (minCPAsString != null) {
                minCPAsString = freenet.support.URLDecoder.decode(minCPAsString);
                minCP = Float.valueOf(minCPAsString).floatValue();
            }
        }
        catch (NumberFormatException nfe) {
        }
        catch (URLEncodedFormatException uefe) {
        }

        if (minCP > 1.0 ) {
            minCP = (float)1.0;
        }
        if (minCP < 0.0) {
            minCP = (float)0.0;
        }

        int minConnections = 0;
        try {
            String minConnectionsAsString = req.getParameter("minConnections");
            if (minConnectionsAsString != null) {
                minConnectionsAsString = freenet.support.URLDecoder.decode(minConnectionsAsString);
                minConnections = Integer.parseInt(minConnectionsAsString);
            }
        }
        catch (NumberFormatException nfe) {
        }
        catch (URLEncodedFormatException uefe) {
        }
                
        if (minConnections < 0) {
            minConnections = 0;
        }

        resp.setStatus(HttpServletResponse.SC_OK);
        resp.setContentType("text/plain");

        final StringMap[] refs = rt.getSnapshot().refData();
        if ((refs == null) || (refs.length < 1)) {
            resp.getWriter().println(""); // so that document is not empty.
            resp.flushBuffer();
            return;
        }

        // REDFLAG: doc key / type assumptions

        boolean hasCPs = (refs[0].value(PROP_CP) != null) && 
            (refs[0].value(PROP_CP) instanceof Float);
        boolean hasConns = (refs[0].value(PROP_CONNECTIONS) != null) &&
            (refs[0].value(PROP_CONNECTIONS) instanceof Integer);
        
        WriteOutputStream out = null;
        out = new WriteOutputStream(resp.getOutputStream());
        
        int count = 0;
        for (int i = 0; i < refs.length; i++) {
            if (hasCPs) {
                float cp =  ((Float)refs[i].value(PROP_CP)).floatValue();
                if (cp < minCP) {
                    continue;
                }
            }

            if (hasConns) {
                int conns =  ((Integer)refs[i].value(PROP_CONNECTIONS)).intValue();
                if (conns < minConnections) {
                        continue;
                }
            }
            
            NodeReference ref = (NodeReference)refs[i].value(PROP_NODEREF);
            ref.getFieldSet().writeFields(out);
            count++;
        }
    
        // Don't send an empty document
        // error in browser.
        if (count == 0) {
            out.write('\n');
        }

        resp.flushBuffer(); 
    }

    // REDFLAG: C&P from fproxy
    private void sendError(HttpServletResponse resp, int status,
                             String detailMessage)
        throws IOException {

        // get status string
        String statusString = status + " " +
            HttpServletResponseImpl.getNameForStatus(status);

        // show it
        Core.logger.log(this, "Sending HTTP error: " + statusString,
                        Logger.DEBUGGING);
        PrintWriter pw = resp.getWriter();
        resp.setStatus(status);
        resp.setContentType("text/html");
        pw.println("<html>");
        pw.println("<head><title>" + statusString + "</title></head>");
        pw.println("<body bgcolor=\"#ffffff\">");
        pw.println("<h1>" + statusString + "</h1>");
        pw.println("<p>" + detailMessage);
        pw.println("</body>");
        pw.println("</html>");
        resp.flushBuffer(); 
    }

    ///////////////////////////////////////////////////////////
    // Support for printing contents of ticker and ocm.


    private final void sendTickerContents(HttpServletResponse resp) throws IOException {
        PrintWriter pw = resp.getWriter();
        resp.setContentType("text/html");

        pw.println("<html>");
        pw.println("<head>");
        pw.println("<title>");
        pw.println("Freenet Node Ticker Contents");
        pw.println("</title>");
        pw.println("</head>");
        pw.println("<body>");
        node.ticker().writeEventsHtml(pw);
        pw.println("</body>");
        pw.println("</html>");
    }

    private final void sendOcmContents(HttpServletResponse resp) throws IOException {
        PrintWriter pw = resp.getWriter();
        resp.setContentType("text/html");

        pw.println("<html>");
        pw.println("<head>");
        pw.println("<title>");
        pw.println("Open ConnectionManager Contents");
        pw.println("</title>");
        pw.println("</head>");
        pw.println("<body>");
        node.connections.writeHtmlContents(pw);
        pw.println("</body>");
        pw.println("</html>");
    }

    ////////////////////////////////////////////////////////////
    // Support for sending Diagnostics.

    private final void sendVarData(HttpServletRequest req, 
                                   HttpServletResponse resp, 
                                   String varName, 
                                   String period) throws IOException {
        PrintWriter pw = resp.getWriter();

        DiagnosticsFormat format;
        boolean html = false;
        if (period.equalsIgnoreCase("occurrences")) {
            html = true;
            resp.setContentType("text/html");
            format =  new HtmlDiagnosticsFormat(-1);
        } else if (period.equalsIgnoreCase("raw")) {
            resp.setContentType("text/plain");
            format = new RowDiagnosticsFormat();
        } else if (period.startsWith("raw")) {
            resp.setContentType("text/plain");
            if (period.substring(3).equalsIgnoreCase("occurences"))
                format = new RowDiagnosticsFormat(-1);
            else
                format = 
                    new RowDiagnosticsFormat(Diagnostics.getPeriod(period.substring(3)));

        } else if (period.equalsIgnoreCase("fieldset")) {
            resp.setContentType("text/plain");
            format = new FieldSetFormat();
        } else {
            try {
                resp.setContentType("text/html");
                html = true;
                format = 
                    new HtmlDiagnosticsFormat(Diagnostics.getPeriod(period));
            } catch (IllegalArgumentException e) {
                sendError(resp, 404, "Unknown period type given.");
                return;
            }
        }

        try {
            if (html)
                pw.println("<html><body>");
            pw.println(diagnostics.writeVar(varName, format));
            if (html)
                pw.println("</body></html>");
        } catch (NoSuchElementException e) {
            sendError(resp, 404, "No such diagnostics field");
            return;
        }

        resp.flushBuffer();
    }

    private final void sendDiagnosticsIndex(HttpServletRequest req, HttpServletResponse resp,
                                            String baseURL)
        throws IOException {

        PrintWriter pw = resp.getWriter();
        resp.setContentType("text/html");

        pw.println("<html>");
        pw.println("<head>");
        pw.println("<title>");
        pw.println("Freenet Node Diagnostics Variables");
        pw.println("</title>");
        pw.println("</head>");
        pw.println("<body>");
        pw.println();
        pw.println();

        DiagnosticsFormat indexFormat = new HtmlIndexFormat();
        Core.diagnostics.writeVars(pw, indexFormat);

        pw.println("</body>");
        pw.println("</html>");
        resp.flushBuffer();
    }

    private final void sendLoadStats(HttpServletResponse resp)
        throws IOException {

	resp.setStatus(HttpServletResponse.SC_OK);
	resp.setContentType("text/plain");
        loadStats.dump(resp.getWriter());
        resp.flushBuffer();
    }


    private final int getActive(ContactCounter.Record[] contacts) {
        int activeCount = 0;
        for (int i = 0; i < contacts.length; i++) {
            activeCount += contacts[i].activeContacts;
        }
        return activeCount;
    }

    private final void sendPerHostStats(HttpServletResponse resp, String kind)
        throws IOException {

	resp.setStatus(HttpServletResponse.SC_OK);
	resp.setContentType("text/plain");
        PrintWriter pw = resp.getWriter();
        ContactCounter.Record[] contacts = null;
        if (kind.equals("inboundContacts")) {
            if (inboundContacts != null) {
                contacts = inboundContacts.getSnapshot();
                pw.println("# unique contacts: " + contacts.length + 
                           ", active connections: " + getActive(contacts));
                
                pw.println("# format: <contact attempts> <successful> <active connections> <address>");
                pw.println("#");
            }
            else {
                pw.println("# Inbound contacts are not being logged.");
                pw.println("# To enable logging set:");
                pw.println("#   logInboundContacts=true");
                pw.println("# in your freenet.conf / freenet.ini file.");
                resp.flushBuffer();
                return;
            }
        }
        else if (kind.equals("outboundContacts")) {
            if (outboundContacts != null) {
                contacts = outboundContacts.getSnapshot();
                pw.println("# unique contacts: " + contacts.length + 
                           ", live connections: " + getActive(contacts));
                
                pw.println("# format: <contact attempts> <successful> <live connections> <address>");
                pw.println("#");
            } 
            else {
                pw.println("# Outbound contacts are not being logged.");
                pw.println("# To enable logging set:");
                pw.println("#   logOutboundContacts=true");
                pw.println("# in your freenet.conf / freenet.ini file.");
                resp.flushBuffer();
                return;
            }
        }
        else if (kind.equals("outboundRequests")) {
            if (outboundRequests != null) {
                contacts = outboundRequests.getSnapshot();
                pw.println("# unique hosts: " + contacts.length); 
                pw.println("# format: <requests> <address>");
                pw.println("#");
            }
            else {
                pw.println("# Outbound requests are not being logged.");
                pw.println("# To enable logging set:");
                pw.println("#   logOutboundRequests=true");
                pw.println("# in your freenet.conf / freenet.ini file.");
                resp.flushBuffer();
                return;
            }
        }
        else {
            if (inboundRequests != null) {
                contacts = inboundRequests.getSnapshot();
                pw.println("# unique hosts: " + contacts.length); 
                pw.println("# format: <requests> <requests accepted> <address>");
                pw.println("#");
            }
            else {
                pw.println("# Inbound requests are not being logged.");
                pw.println("# To enable logging set:");
                pw.println("#   logInboundRequests=true");
                pw.println("# in your freenet.conf / freenet.ini file.");
                resp.flushBuffer();
                return;
            }
        }

        if (contacts != null) {
            // Sort by count.
            HeapSorter.heapSort(new ArraySorter(contacts));
            
            for (int i = 0; i < contacts.length; i++) {
                if (kind.equals("inboundRequests")) {
                    pw.println( contacts[i].totalContacts + "\t" +
                                contacts[i].successes + "\t" +
                                contacts[i].addr);
                   
                }
                else if (kind.equals("outboundRequests")) {
                    pw.println( contacts[i].totalContacts + "\t" +
                                contacts[i].addr);
                   
                }
                else {
                    pw.println( contacts[i].totalContacts + "\t" +
                                contacts[i].successes + "\t" +
                                contacts[i].activeContacts + "\t"+
                                contacts[i].addr );
                }
            }
        }

        resp.flushBuffer();
    }

    ////////////////////////////////////////////////////////////
    // Hooks to provide nicer formatting for RoutingTable
    // implementations that we know about.
    private final Long MINUS_ONE = new Long(-1);
   
    Object[] formatRef(String rtType, Object[] refValues) {
        if (rtType == null) {
            return refValues;
        }
        
        if (rtType.equals("freenet.node.rt.CPAlgoRoutingTable")) {
            int attempts = ((Integer)refValues[3]).intValue();
            if (attempts > 0) {
                // percent successful
                refValues[4] = refValues[4].toString() + " &nbsp &nbsp ("
                    + (100 * ((Integer)refValues[4]).intValue() / (float)attempts) +
                    "%)";
            }
            
            if (refValues[5].equals(MINUS_ONE)) {
                refValues[5] = "no";
            }
            else {
                refValues[1] = " <font color=\"blue\"> " + refValues[1] + "</font> ";
                refValues[5] = refValues[5].toString() + " secs. longer";
            }

            if (refValues[6].equals(MINUS_ONE)) {
                refValues[6] = "never";
            }
            else {
                refValues[6] = refValues[6].toString() + " secs. ago";
            }
            
            // clean up version
            String verString = (String)refValues[8];
            int pos = verString.lastIndexOf(",");
            if (pos > -1 && pos < verString.length() - 1) {
                refValues[8] = verString.substring(pos + 1);
            }
        }
        
        return refValues;
    }
    
    ////////////////////////////////////////////////////////////
    Node node = null;
    RoutingTable rt = null;
    Diagnostics diagnostics = null;
    ContactCounter inboundContacts = null;
    ContactCounter outboundContacts = null;
    ContactCounter inboundRequests = null;
    ContactCounter outboundRequests = null;
    LoadStats loadStats = null;
}






