package freenet.fs.acct;

import freenet.support.Fields;
import freenet.support.Comparator;
import java.io.*;

/**
 * A Fragment represents some continuous range in the datastore.
 * A file-system element, or file, for example, consists of one or
 * more fragments of data.  The directory's free-list consists of a number
 * of fragments of free space.
 *
 * @author Scott
 * @author tavin
 */
public class Fragment {

    public static final class ComparatorByPos implements Comparator {
        public final int compare(Object o1, Object o2) {
            return compare((Fragment) o1, (Fragment) o2);
        }
        public static final int compare(Fragment f1, Fragment f2) {
            return f1.lo == f2.lo
                   ? 0 : (f1.lo > f2.lo ? 1 : -1);
        }
    }

    public static final class ComparatorBySize implements Comparator {
        public final int compare(Object o1, Object o2) {
            return compare((Fragment) o1, (Fragment) o2);
        }
        public static final int compare(Fragment f1, Fragment f2) {
            return f1.size() == f2.size()
                   ? 0 : (f1.size() > f2.size() ? 1 : -1);
        }
    }
    

    /**
     * Utility method to compute the byte size of a series of Fragments.
     * @param ranges an array of Fragments
     * @return the total byte length of the Fragments
     */
    public static final long rangeLength(Fragment[] ranges) {
        long total = 0;
        for (int i=0; i<ranges.length; ++i) 
            total += ranges[i].size();
        return total;
    }

    /**
     * Utility method.  Takes a series of Fragments and stringifies
     * them as a comma-separated list.
     * @param ranges  an array of Fragments
     * @param radix   numeric base to use for the string representation
     * @return String consisting of comma-separated
     *         low and high Fragment boundaries
     */
    public static String rangeList(Fragment[] ranges, int radix) {
        String[] fr = new String[2*ranges.length];
        for (int i=0; i<ranges.length; ++i) {
            fr[2*i]   = Long.toString(ranges[i].lo, radix);
            fr[2*i+1] = Long.toString(ranges[i].hi, radix);
        }
        return Fields.commaList(fr);
    }

    /**
     * Defaults to hex (radix 16).
     * @see rangeList(Fragment[], int)
     */
    public static final String rangeList(Fragment[] ranges) {
        return rangeList(ranges, 16);
    }

    /**
     * Parses the output of rangeList() and
     * reconstructs the Fragment array.
     * @param r      the String output of rangeList()
     * @param radix  the numeric base that was used
     * @return  the array of Fragments
     */
    public static Fragment[] parseRangeList(String r, int radix) {
        String[] fr = Fields.commaList(r);
        if (!(fr.length > 0 && fr.length % 2 == 0))
            throw new IllegalArgumentException("ranges list is invalid");
        Fragment[] ranges = new Fragment[fr.length >> 1];
        for (int i=0; i<ranges.length; ++i) {
            ranges[i] = new Fragment( Long.parseLong(fr[2*i],   radix),
                                      Long.parseLong(fr[2*i+1], radix) );
        }
        return ranges;
    }

    /**
     * Defaults to hex (radix 16).
     * @see parseRangeList(String, int)
     */
    public static final Fragment[] parseRangeList(String r) {
        return parseRangeList(r, 16);
    }


    public static final void parseRangeList(Fragment[] ranges, DataInput din)
                                                        throws IOException {
        for (int i=0; i<ranges.length; ++i)
            ranges[i] = new Fragment(din.readLong(), din.readLong());
    }

    public static final void writeRangeList(Fragment[] ranges, DataOutput out)
                                                        throws IOException {
        for (int i=0; i<ranges.length; ++i) {
            out.writeLong(ranges[i].getLowerBound());
            out.writeLong(ranges[i].getUpperBound());
        }
    }


    public static final Fragment parseRange(DataInput din)
                                        throws IOException {
        return new Fragment(din.readLong(), din.readLong());
    }

    public static final void writeRange(Fragment f, DataOutput out)
                                                throws IOException {
        out.writeLong(f.getLowerBound());
        out.writeLong(f.getUpperBound());
    }

    
    /** lower bound */
    private long lo;

    /** upper bound */
    private long hi;

    
    /**
     * @param lo  lower bound of the range
     * @param hi  upper bound of the range
     */
    public Fragment(long lo, long hi) {
        this.lo = lo;
        this.hi = hi;
    }

    /**
     * Creates a Fragment fixed at 0.
     * @param size  size of fragment (determines upper bound)
     */
    public Fragment(long size) {
        this.lo = 0;
        this.hi = size-1;
    }

    /**
     * Copies an existing Fragment.
     */
    public Fragment(Fragment f) {
        this.lo = f.lo;
        this.hi = f.hi;
    }

    public final long getLowerBound() {
        return lo;
    }

    public final long getUpperBound() {
        return hi;
    }

    public final int hashCode() {
        return ((int) (lo & 0xffffffffL)) ^ ((int) (lo >> 32))
             ^ ((int) (hi & 0xffffffffL)) ^ ((int) (hi >> 32));
    }

    public final boolean equals(Object other) {
        return other instanceof Fragment && equals((Fragment) other);
    }

    public final boolean equals(Fragment other) {
        return lo == other.lo && hi == other.hi;
    }

    public final long size() {
        return hi + 1 - lo;
    }

    public final boolean overlaps(Fragment other) {
        return other.lo <= hi
            && other.hi >= lo;
    }

    public final boolean borders(Fragment other) {
        return lo == other.hi+1 || hi == other.lo-1;
    }

    public final boolean contains(Fragment other) {
        return lo <= other.lo && hi >= other.hi;
    }

    public final boolean subsume(Fragment other) {
        if (lo == other.hi+1) {
            lo = other.lo;
            return true;
        }
        if (hi == other.lo-1) {
            hi = other.hi;
            return true;
        }
        return false;
    }

    public final String toString() {
        return Fields.longToHex(lo)+"->"+Fields.longToHex(hi);
    }
}


