package freenet.fs.dir;

import java.io.*;

/**
 * A Buffer for tunneling data through a limited storage area.
 * After the writer starts writing to the output stream, readers
 * may pick up input streams until the first time the buffer laps.
 * @author tavin
 */
public class CircularBuffer implements Buffer {

    protected final Buffer lapbuf;
    protected final long vlength;

    protected int lap = -1;
    protected boolean aborted = false;
    

    /**
     * @param lapbuf   buffer to do laps over
     * @param vlength  virtual length -- total length of tunneled data
     */
    public CircularBuffer(Buffer lapbuf, long vlength) {
        this.lapbuf = lapbuf;
        this.vlength = vlength;
    }

    
    public final Ticket ticket() {
        return lapbuf.ticket();
    }


    public final void touch() {
        //throw new DirectoryException("now what's the point of that?");
    }

    public final void commit() {
        throw new DirectoryException("you can't commit a circular buffer");
    }
    
    public final void release() {
        lapbuf.release();
    }
    

    public final boolean failed() {
        return aborted || lapbuf.failed();
    }

    public final long length() {
        return vlength;
    }

    /**
     * This will grant input streams only until the buffer has lapped.
     * It will block until the output stream has been obtained.
     */
    public InputStream getInputStream() throws IOException,
                                               BufferException {
        return new CircularInputStream();
    }

    private class CircularInputStream extends FilterInputStream {

        private long rlim;
        private int rlap = -1;
     
        private CircularInputStream() throws IOException, BufferException {
            super(null);
            synchronized (CircularBuffer.this) {
                if (lap > 0)
                    throw new BufferException("buffer already lapped");
                nextLap();
            }
        }

        public int read() throws IOException {
            if (aborted) throw new BufferException("buffer write aborted");
            if (rlim <= 0 && !nextLap())
                return -1;
            int rv = in.read();
            if (rv != -1) --rlim;
            return rv;
        }

        public int read(byte[] buf, int off, int len) throws IOException {
            if (aborted) throw new BufferException("buffer write aborted");
            if (rlim <= 0 && !nextLap())
                return -1;
            int rv = in.read(buf, off, (int) Math.min(len, rlim));
            if (rv != -1) rlim -= rv;
            return rv;
        }

        public int available() throws IOException {
            if (aborted) throw new BufferException("buffer write aborted");
            return in.available();
        }
        
        public long skip(long n) throws IOException {
            if (aborted) throw new BufferException("buffer write aborted");
            return in.skip(n);
        }

        private boolean nextLap() throws IOException {
            rlim = Math.min(lapbuf.length(),
                            vlength - (1+rlap) * lapbuf.length());
            if (rlim <= 0) {
                return false;
            }
            synchronized (CircularBuffer.this) {
                ++rlap;
                while (!failed() && rlap > lap) {
                    try { CircularBuffer.this.wait(); }
                    catch (InterruptedException e) {}
                }
                if (aborted)
                    throw new BufferException("buffer write aborted");
                in = lapbuf.getInputStream();
            }
            return true;
        }
    }

    /**
     * This will grant one and only one output stream.  When the stream
     * laps the buffer, the buffer will be removed from the directory.
     */
    public OutputStream getOutputStream() throws IOException,
                                                 BufferException {
        return new CircularOutputStream();
    }    

    private class CircularOutputStream extends FilterOutputStream {

        private long wlim;

        private CircularOutputStream() throws IOException, BufferException {
            super(null);
            synchronized (CircularBuffer.this) {
                if (lap != -1)
                    throw new BufferException("buffer already written");
                nextLap();
            }
        }

        public void write(int b) throws IOException {
            if (wlim <= 0) nextLap();
            out.write(b);
            --wlim;
        }

        public void write(byte[] buf, int off, int len) throws IOException {
            while (len > 0) {
                if (wlim <= 0) nextLap();
                int n = (int) Math.min(len, wlim);
                out.write(buf, off, n);
                off += n;
                len -= n;
                wlim -= n;
            }
        }
        
        private void nextLap() throws IOException {
            wlim = Math.min(lapbuf.length(),
                            vlength - (1+lap) * lapbuf.length());
            if (wlim <= 0) {
                throw new EOFException();
            }
            synchronized (CircularBuffer.this) {
                ++lap;
                CircularBuffer.this.notifyAll();
                out = lapbuf.getOutputStream();
            }
        }

        public void close() throws IOException {
            try {
                out.close();
            }
            finally {
                kick();
            }
        }

        private void kick() {
            if (wlim + lap * lapbuf.length() < vlength) {
                synchronized (CircularBuffer.this) {
                    aborted = true;
                    CircularBuffer.this.notifyAll();
                }
            }
        }

        protected void finalize() throws Throwable {
            kick();
        }
    }
}



