package freenet.fs.dir;

import freenet.fs.acct.*;
import freenet.fs.acct.sys.*;
import freenet.support.*;
import freenet.support.BinaryTree.Node;

/**
 * An AccountingTree where the elements are Fragment instances.
 * Keeps track of the total size and number of the Fragments in the tree.
 * @author tavin
 */
abstract class FragmentMap extends AccountingTree {

    private long count = 0;  // number of Fragments
    
    private long total = 0;  // total byte size
    
    
    FragmentMap(AccountingProcess proc, AccountingTreeMarshal marshal, Cache cache) {
        super(proc, marshal, cache);
    }


    /**
     * @return  the sum of the sizes of the Fragments in the tree
     */
    final long total() {
        return total;
    }

    /**
     * @return  the number of Fragments in the tree
     */
    final long count() {
        return count;
    }


    /**
     * @return  a Walk of FragmentRecord objects in arbitrary order
     */
    final Walk fragments() {
        return new FragmentRecordWalk(treeWalk(true));
    }

    private static final class FragmentRecordWalk implements Walk {
        private final Walk walk;
        FragmentRecordWalk(Walk walk) {
            this.walk = walk;
        }
        public final Object getNext() {
            Node n = (Node) walk.getNext();
            return n == null ? null : n.getObject();
        }
    }
    

    
    public final Node treeInsert(Node n, boolean replace) {
        Node old = super.treeInsert(n, replace);
        if (old != null && replace) {
            --count;
            total -= ((Fragment) old.getObject()).size();
        }
        if (old == null || replace) {
            ++count;
            total += ((Fragment) n.getObject()).size();
        }
        return old;
    }

    
    public final boolean treeRemove(AccountingTreeNode n) {
        if (super.treeRemove(n)) {
            --count;
            total -= ((Fragment) n.getObject()).size();
            return true;
        }
        else return false;
    }

    
    protected final void found(BinaryTree subTree) {
        Walk walk = subTree.treeWalk(true);
        Node node;
        while (null != (node = (Node) walk.getNext())) {
            ++count;
            total += ((Fragment) node.getObject()).size();
        }
    }
}



