package freenet.fs.dir;

import freenet.support.*;
import java.io.IOException;
import java.util.Enumeration;

/**
 * The LossyDirectory attempts to automatically free space to
 * make room for flush and store operations.  It can do this
 * by deleting its own files, or the files of another shared
 * directory.
 * @author tavin
 */
public class LossyDirectory extends SharedDirectory {

    private final FilePattern lossyPattern;


    /**
     * A lone directory that deletes from itself when it fills.
     */
    public LossyDirectory(Directory dir) {
        this(0, dir);
    }

    /**
     * A shared directory that deletes from itself when it fills.
     */
    public LossyDirectory(int dirID, Directory dir) {
        this(dirID, dir, dirID);
    }

    /**
     * A directory that deletes from another when it fills.
     */
    public LossyDirectory(int dirID, Directory dir, int lossy) {
        this(dirID, dir, new int[] {lossy});
    }
    
    /**
     * A directory that deletes from several others when it fills.
     */
    public LossyDirectory(int dirID, Directory dir, int[] lossy) {
        super(dirID, dir);
        lossyPattern = new DirectoryIDFilePattern(lossy, true, true);
    }


    
    /**
     * Attempts a flush and deletes files to make room for it,
     * if necessary.  Succeeds unless it exits with an exception.
     * @throws IOException  for a disk I/O error, or if not enough
     *                      space could be freed 
     */
    public final void forceFlush() throws IOException {
        
        if (!dirty())
            return;

        synchronized (semaphore()) {
            if (flush() > 0) {
                Walk walk = FileNumber.filter(lossyPattern,
                                              new EnumerationWalk(dir.lruKeys(true)));
                FileNumber fn;
                while (null != (fn = (FileNumber) walk.getNext())) {
                    dir.delete(fn);
                    if (flush() == 0)
                        return;
                }
                throw new IOException("insufficient storage");
            }
        }
    }

    /**
     * Deletes files if necessary to allocate a buffer.
     * Succeeds unless it exits with an exception.
     * @throws IOException  if not enough space could be freed
     */
    public final Buffer forceStore(long size, FileNumber fnew) throws IOException {
        synchronized (semaphore()) {
            while (available() < size) {
                try {
                    Walk walk = FileNumber.filter(lossyPattern,
                                                  new EnumerationWalk(dir.lruKeys(true)));
                    do {
                        FileNumber fn = (FileNumber) walk.getNext();
                        if (fn == null)
                            throw new IOException("insufficient storage");
                        dir.delete(fn);
                    }
                    while (available() < size);
                }
                finally {
                    forceFlush();
                }
            }
            
            Buffer buffer = store(size, fnew);
            if (buffer == null)
                throw new IOException("insufficient storage");
            return buffer;
        }
    }
}



