package freenet.interfaces;

import freenet.*;
import freenet.thread.*;
import freenet.support.Logger;
import java.io.*;
import java.util.Vector;
import java.util.Enumeration;

/**
 * An interface receives incoming connections from a Freenet peer
 * (or a client).  It contains a Transport layer ListeningAddress
 * to listen for connections, and a ConnectionRunner to dispatch
 * the accepted connections.
 *
 * If an exception throw occurs in the socket accept, a failure
 * counter is incremented and the thread sleeps for 2500ms before
 * trying again.  After 10 successive failures the interface
 * stops listening until explicitly reactivated with listen().
 * Restarting the interface resets the failure count to 0.
 *
 * @author oskar
 */
public abstract class Interface implements Runnable {

    /** address this interface listens on */
    private final ListeningAddress listenAddr;

    
    /** the thread that is accepting connections */
    private Thread executor;

    
    /** to decommission the interface completely */
    private boolean terminated = false;

    /** toggled to bring the interface up and down */
    private boolean listening = true;
    
    
    /** collects exception throws */
    private final Vector errors = new Vector();
    

    /**
     * @param listenAddr  The listening address this Interface should
     *                    operate on.
     */
    public Interface(ListeningAddress listenAddr) {
        this.listenAddr = listenAddr;
    }

    /**
     * @return "Interface # <port>"
     */
    public String toString() {
        return "Interface # "+listenAddr;
    }


    /**
     * @return  the Transport of the listening address
     */
    public final Transport transport() {
        return listenAddr.transport();
    }

    /**
     * @return  the ListeningAddress this was constructed with
     */
    public final ListeningAddress listeningAddress() {
        return listenAddr;
    }
    

    /**
     * @return  the number of successive socket accept failures
     */
    public final int getExceptionCount() {
        return errors.size();
    }
    
    /**
     * @return  an enumeration of the collected exception throws
     */
    public final Enumeration getExceptions() {
        return errors.elements();
    }
    
    
    /**
     * @return  true, if the interface is accepting connections
     */
    public final boolean isListening() {
        return listening;
    }

    /**
     * @return  true, if the interface has been stopped
     */
    public final boolean isTerminated() {
        return terminated;
    }

    
    /**
     * Bring the interface up or down.  This merely pauses the thread.
     * @param listen  true for up, false for down
     */
    public synchronized final void listen(boolean listen) {
        if (listen) {
            listening = true;
            errors.removeAllElements();
            this.notify();
        }
        else {
            listening = false;
            if (executor != null)
                executor.interrupt();
        }
    }

    /**
     * Decommission the interface entirely.  The run() method will return.
     */
    public synchronized final void terminate() {
        terminated = true;
        this.notify();
        if (executor != null)
            executor.interrupt();
    }


    /**
     * Accepts connections and feeds them through the layers of this Interface.
     */
    public void run() {
        Core.logger.log(this, "Starting interface: "+this, Logger.MINOR);
        try {
            executor = Thread.currentThread();
            while (!terminated) {
                if (!listening) {
                    synchronized (this) {
                        while (!listening && !terminated) {
                            try {
                                this.wait();
                            }
                            catch (InterruptedException e) {}
                        }
                    }
                    continue;
                }
                try {
                    acceptConnections();
                }
                catch (Throwable e) {
                    Core.logger.log(this,
                                    "Unhandled throw accepting connections: "+this,
                                    e, Logger.ERROR);
                    errors.addElement(e);
                    listening = false;
                }
            }
        }
        finally {
            Core.logger.log(this, "Interface terminating: "+this, Logger.MINOR);
        }
    }
                    

    private void acceptConnections() {
        Listener listener = null;
        try {
            while (listening && !terminated) {
                try {
                    if (listener == null) {
                        listener = listenAddr.getListener();
                    }
                    try {
                        listener.setTimeout(2500);
                    }
                    catch (IOException e) {}
                    
                    Connection conn = listener.accept();
                    
                    try {
                        dispatch(conn);
                        Core.logger.log(this, "Accepted connection: "+conn,
                                        Logger.MINOR);
                    }
                    catch (RejectedConnectionException e) {
                        Core.logger.log(this, "Rejected connection: "+e.getMessage(),
                                        Logger.MINOR);
                        conn.close();
                    }
                    catch (Throwable e) {
                        Core.logger.log(this,
                                        "Unhandled throwable when dispatching connection",
                                        e, Logger.NORMAL);
                        conn.close();
                    }
                    
                    errors.removeAllElements();
                }
                catch (InterruptedIOException e) {
                    // nop
                }
                catch (IOException e) {
                    Core.logger.log(this,
                                    "I/O error accepting connections on "+this
                                    +": "+e, Logger.MINOR);
                }
                catch (ListenException e) {
                    Core.logger.log(this,
                                    "Cannot open listener: "+this,
                                    e, Logger.ERROR);
                    errors.addElement(e);
                    try {
                        Thread.sleep(5000);
                    }
                    catch (InterruptedException ie) {}
                }
                catch (Throwable e) {
                    Core.logger.log(this,
                                    "Unhandled throw accepting connections: "+this,
                                    e, Logger.ERROR);
                    errors.addElement(e);
                    try {
                        Thread.sleep(5000);
                    }
                    catch (InterruptedException ie) {}
                }
                finally {
                    if (errors.size() >= 6) {
                        Core.logger.log(this,
                                        "Stopping interface due to errors: "+this,
                                        Logger.ERROR);
                        listening = false;
                    }
                }
            }
        }
        finally {
            try {
                if (listener != null)
                    listener.close();
            }
            catch (IOException e) {
                errors.addElement(e);
            }
        }
    }
    

    /**
     * Should decide whether to accept the connection and start
     * a thread to handle it, then return quickly.
     */
    protected abstract void dispatch(Connection conn)
                    throws RejectedConnectionException;
}


