package freenet.interfaces.servlet;

import freenet.*;
import freenet.node.Node;
import freenet.interfaces.Service;
import freenet.support.Logger;
import freenet.support.servlet.*;
import freenet.client.ClientFactory;
import javax.servlet.*;
import java.io.IOException;

/**
 * Generic interface for parsing servlet requests from a connection
 * and dispatching them.
 *
 * @author tavin
 */
public abstract class ServletContainer implements Service {

    protected final Node node;
    protected final Logger logger;
    protected final ClientFactory factory;

    /**
     * Constructor for servlet containers running within the node.
     */
    protected ServletContainer(Node node) {
        this.node = node;
        this.logger = node.logger;
        this.factory = node.client;
    }

    /**
     * Constructor for servlet containers running independently of the node.
     */
    protected ServletContainer(Logger logger, ClientFactory factory) {
        this.node = null;
        this.logger = logger;
        this.factory = factory;
    }

    /**
     * Reads a ServletRequest from the connection, then obtains
     * a ServletResponse and a Servlet instance and calls the
     * service() method of the Servlet.
     *
     * If the implementation throws a BadRequestException it will
     * result in the connection simply being closed.  Therefore,
     * the implementation may instead wish to return an alternate
     * Servlet that issues an error response to the client.
     */
    public void handle(Connection conn) {
        try {
            ServletRequest request = null;
            ServletResponse response = null;
            Servlet servlet;
            
            while (null != (request = getNextRequest(request, response, conn))) {
                response = getResponseFor(request);
                servlet  = getServletFor(request);
                try {
                    servlet.service(request, response);
                    response.flushBuffer();
                }
                finally {
                    returnServlet(request, servlet);
                }
            }
        }
        catch (BadRequestException e) {
            Core.logger.log(this, "Bad servlet request", e, Logger.NORMAL);
        }
        catch (UnavailableException e) {
            Core.logger.log(this, "Servlet unavailable for connection: "+conn,
                            e, Logger.ERROR);
        }
        catch (ServletException e) {
            Core.logger.log(this, "Servlet failure: "+e.getMessage(),
                            e, Logger.NORMAL);
        }
        catch (IOException e) {
            Core.logger.log(this, "I/O error in servlet", e, Logger.NORMAL);
        }
        finally {
            conn.close();
        }
    }

    
    /**
     * @return  the filesystem path for the given virtual path
     * The subclass should override this if it wants path translation
     * to work.
     */
    public String getRealPath(String path) {
        return null;
    }
    
    
    /**
     * @return  the RequestDispatcher for the named resource
     */
    public RequestDispatcher getRequestDispatcher(String name) {
        return null;  // FIXME
    }

    
    /**
     * @return  a line of some descriptive nonsense
     */
    public abstract String getServerInfo();
    
    
    /**
     * @return  the context for the uri path (some containers
     *          might ignore the path and just use one context)
     */
    public abstract ServletContext getContext(String uripath);



    // thinking about turning this stuff into a system of Factories


    protected ServletContext createServletContext(FieldSet contextParams) {
        // NOTE:  We shouldn't expose the logger like this.
        //        Servlets automatically inherit some log() methods
        //        which they should use instead.  That is why the
        //        logger and log-level are constructor parameters
        //        of the ServletContextImpl.
        //
        //if (lg != null) {
        //    context.setAttribute("Freenet.support.Logger", lg);
        //}

        ServletContext context = new ServletContextImpl(this, contextParams,
                                                        logger, Logger.MINOR);
        if (node != null)
            context.setAttribute("freenet.node.Node", node);
        context.setAttribute("freenet.client.ClientFactory", factory);
        
        return context;
    }

    protected ServletConfig createServletConfig(ServletContext context,
                                                String servletName,
                                                FieldSet servletParams) {
        return new ServletConfigImpl(context, servletName, servletParams);
    }
    

    /**
     * Should keep returning ServletRequests until the connection
     * is ready to be closed, at which time it should return null.
     * @param lastRequest   the last ServletRequest produced
     *                      (null on the first call)
     * @param lastResponse  the last ServletResponse produced
     *                      (null on the first call)
     */
    protected abstract ServletRequest getNextRequest(ServletRequest lastRequest,
                                                     ServletResponse lastResponse,
                                                     Connection conn)
                                throws IOException, BadRequestException;

    /**
     * Create the ServletResponse object for the given ServletRequest.
     */
    protected abstract ServletResponse getResponseFor(ServletRequest req)
                                throws IOException;

    /**
     * Get a Servlet instance to handle the request.
     */
    protected abstract Servlet getServletFor(ServletRequest req)
                                throws ServletException, UnavailableException;

    /**
     * Return the Servlet instance to the pool.
     */
    protected abstract void returnServlet(ServletRequest req, Servlet servlet);
}




