package freenet.message;
import freenet.*;
import freenet.node.Node;
import freenet.support.Fields;
import freenet.crypt.EntropySource;
import java.io.OutputStream;

/**
 * Superclass for messages that are sent between nodes (basically those
 * that have a UniqueID field).
 *
 * @author oskar
 */
public abstract class NodeMessage extends Message {

    private static EntropySource uniqueIdEntropy = new EntropySource();

    /**
     * Creates a new message.
     */
    protected NodeMessage(long id) {
        super(id);
    }
    
    /**
     * Creates a new message.
     * @param  idnum        The message's Unique ID, should be a random long.
     * @param  otherFields  The remaining set of fields which are not directly
     *                      related to routing the message.
     */
    protected NodeMessage(long id, FieldSet otherFields) {
        super(id, otherFields);
    }
    
    /**
     * Creates a message from a RawMessage, as read from a presentation
     * @param raw The RawMessage. Please note that the RawMessage is 
     *            consumed after this. Both the normal and trailing
     *            fields are corrupted. Use toRawMessage() to restore it.
     **/   
    protected NodeMessage(ConnectionHandler source, RawMessage raw) 
        throws InvalidMessageException {

        super(source, 0, raw);
        id = readId(raw.fs);
        if (((id>>32) & 0xffffffffL) == (id & 0xffffffffL)) {
            System.err.println("YOUR NODE CAUSED THE HEISENBUG - " + 
                               "PLEASE REPORT SETUP TO FREENET DEV MAILING " +
                               " LIST!");
            Core.logger.log(this, "YOUR NODE CAUSED THE HEISENBUG - " + 
                            "PLEASE REPORT SETUP TO FREENET DEV MAILING " +
                            " LIST!", Core.logger.ERROR);
            throw new InvalidMessageException("Created heisenbug UniqueID: "
                                              + Fields.longToHex(id) + 
                                              ". killed chain");
        }
    }

    private static long readId(FieldSet fs) throws InvalidMessageException {
        try {
            String ids = fs.get("UniqueID");
            if (ids == null)
                throw new InvalidMessageException("no UniqueID field");
            long id = Fields.stringToLong(ids);
            if (((id>>32) & 0xffffffffL) == (id & 0xffffffffL)) {
                throw new InvalidMessageException("Read heisenbug UniqueID: "
                                + Fields.longToHex(id) + ". killed chain");
            }
            Core.randSource.acceptEntropy(uniqueIdEntropy, id, 64);
            fs.remove("UniqueID");
            return id;
        } catch (NumberFormatException e) {
            throw new InvalidMessageException("Broken id number: " 
                                              + e.getMessage());
        }
    }
    
    /** Converts this message to something that can be sent over the wire,
      * using the given Presentation.
      */
    public RawMessage toRawMessage(Presentation t) {
        RawMessage r = super.toRawMessage(t);
        r.fs.add("UniqueID", Fields.longToHex(id));
        // do heisenbug testing
        long test = Fields.stringToLong(Fields.longToHex(id));
        if(((id >> 32) & 0xffffffffL) == (id & 0xffffffffL)) {
            String s = "PANIC! Just recreated heisenbug! ID: "
                                        + Fields.longToHex(id);
            System.err.println(s);
//          (new Exception()).printStackTrace();
            Core.logger.log(NodeMessage.class, s, Core.logger.ERROR);
        }
        return r;
    }

    /** Part of the MessageObject implementation.
      * @return the unique ID of this message
      */
    public long id() {
        return id;
    }

    /**
     * Send this message to the given Peer.  "Back" is, in truth, optional..
     */
    //public OutputStream sendBack(Node n, Peer p) throws SendFailedException {
    //    try {
    //        return n.makeConnection(p).sendMessage(this);
    //    }
    //    catch (ConnectFailedException e) {
    //        throw new SendFailedException(e);
    //    }
    //}
}


