package freenet.message;
import freenet.*;
import freenet.node.states.FNP.*;
import freenet.node.*;
import java.io.InputStream;
import java.io.IOException;
import freenet.support.io.ZeroInputStream;

/**
 * Void messages are empty and meaningless.
 */

public class VoidMessage extends NodeMessage implements TrailingFieldMessage {

    public static final String messageName = "Void";

    private InputStream data;
    private long dataLength;
    private boolean close;
    private boolean sustain;

    public VoidMessage(long id, boolean close, FieldSet otherFields) {
        super(id, otherFields);
        this.close = close;
        this.sustain = false;
    }
    
    public VoidMessage(long id, boolean close, boolean sustain, 
                       FieldSet otherFields) {

        super(id, otherFields);
        this.close = close;
        this.sustain = sustain;
    }

    /**
     * Assigns a NullInputStream of dataLength to the DataStream.
     */
    public VoidMessage(long id, boolean close, boolean sustain, 
                       long dataLength, 
                       FieldSet otherFields) {
        super(id, otherFields);
        this.dataLength = dataLength;
        this.close = close;
        this.sustain = sustain;
        this.data = new ZeroInputStream(0, Core.blockSize);
    }

    public VoidMessage(ConnectionHandler source, 
                       RawMessage raw) throws InvalidMessageException {
        super(source, raw);
        if (raw.trailingFieldLength > Core.maxPadding)
            throw new InvalidMessageException("Void messages can only have " +
                                              Core.maxPadding +" long data.");
        dataLength = raw.trailingFieldLength;
        if (dataLength > 0) {
            data = raw.trailingFieldStream;
        }
    }

    public RawMessage toRawMessage(Presentation p) {
        RawMessage r = super.toRawMessage(p);
        r.messageType = messageName;
        r.close = close;
        r.sustain = sustain;
        if (dataLength > 0) {
            r.trailingFieldLength = dataLength;
            r.trailingFieldName   = "Padding";
        }
        return r;
    }

    public String getMessageName() {
        return messageName;
    }

    public State getInitialState() throws BadStateException {
        return dataLength == 0 ? null : new NewVoid(id);
    }

    public InputStream getDataStream() {
        return data;
    }

    public long length() {
        return dataLength;
    }

    public void length(long l) {
        this.dataLength = l;
    }

    public void drop(Node n) {
        try {
            if (data != null)
                data.close();
        } catch (IOException e) {
            n.logger.log(this, "Closing dropped DataSend threw exception :" 
                         + e, n.logger.DEBUGGING);
        }
    }
}



