package freenet.node;

import freenet.*;
import freenet.config.*;
import freenet.crypt.*;
import freenet.fs.FileSystem;
import freenet.fs.*;
import freenet.fs.dir.*;
import freenet.fs.acct.*;
import freenet.fs.acct.sys.*;
import freenet.fs.acct.fsck.*;
import freenet.node.ds.FSDataStore;
import freenet.support.io.*;
import freenet.support.Fields;
import java.io.*;
import java.util.Enumeration;

/**
 * A tool for manipulating the node's store while it is offline.
 * @author tavin
 */
public class FSTool {

    private static final Config switches = new Config();
    static {
        switches.addOption("hex",           1, false,        10);
        switches.addOption("major",         1, 0,            11);
        switches.addOption("keys",          1, String.class, 20);
        switches.addOption("store",         1, String.class, 21);
        switches.addOption("cipher",        1, "Twofish",    30);
        switches.addOption("cipherWidth",   1, 128,          31);
        switches.addOption("digest",        1, "SHA1",       40);
        switches.addOption("rootBlockSize", 1, 10000,        50);
        switches.addOption("acctBlockSize", 1, 10000,        51);

        switches.argDesc   ("hex", "yes|no");
        switches.shortDesc ("hex", "read input ranges in hex? (default no)");

        switches.argDesc   ("major", "<integer>");
        switches.shortDesc ("major", "force repair of N major faults");

        switches.argDesc   ("keys", "<file>");
        switches.shortDesc ("keys", "path to node keys file");

        switches.argDesc   ("store", "<file>");
        switches.shortDesc ("store", "path(s) to store file(s)");

        switches.argDesc   ("cipher", "<name>");
        switches.shortDesc ("cipher", "Twofish, Rijndael, or none (default Twofish)");

        switches.argDesc   ("cipherWidth", "<integer>");
        switches.shortDesc ("cipherWidth", "bit width of cipher key (default 128)");
        
        switches.argDesc   ("digest", "<name>");
        switches.shortDesc ("digest", "SHA1 or other digest name (default SHA1)");
        
        switches.argDesc   ("rootBlockSize", "<integer>");
        switches.shortDesc ("rootBlockSize", "size of root blocks (default 10000)");
        
        switches.argDesc   ("acctBlockSize", "<integer>");
        switches.shortDesc ("acctBlockSize", "size of accounting blocks (default 10000)");
    }

    
    public static void main(String[] args) throws Exception {

        Params params = new Params(switches.getOptions());
        params.readArgs(args);

        if (params.getNumArgs() == 0)
            usage();

        String keysFile = params.getString("keys");
        String[] storeFiles = params.getList("store");
        
        if (keysFile == null || keysFile.equals(""))
            keysFile = locateFile("node_");
        
        if (storeFiles.length == 0)
            storeFiles = new String[] {locateFile("store_")};
        
        int cipherWidth = params.getInt("cipherWidth");
        String cipherName = params.getString("cipher");
        if (cipherName.trim().equals("none")
            || cipherName.trim().equals("null")
            || cipherName.trim().equals("void")) cipherName = null;
        
        BlockCipher cipher = (cipherName == null
                              ? null
                              : Util.getCipherByName(cipherName, cipherWidth));
        
        Digest ctx = Util.getDigestByName(params.getString("digest"));
        
        
        long fsLength = 0;
        
        File[] fsFiles = new File[storeFiles.length];
        for (int i=0; i<fsFiles.length; ++i) {
            fsFiles[i] = new File(storeFiles[i]);
            fsLength += fsFiles[i].length();
        }

        Storage storage = new RAFStorage(fsFiles, fsLength);

            
        FileSystem fs;
        if (cipher == null) {
            fs = new FileSystem(storage);
        }
        else {
            FieldSet keys = readKeys(keysFile);
            byte[] cipherKey = Fields.hexToBytes(keys.get("cipherKey"));
            byte[] baseIV = Fields.hexToBytes(keys.get("baseIV"));
            cipher.initialize(cipherKey);
            fs = new EncryptedFileSystem(storage, cipher, baseIV);
        }

        
        String cmd = params.getArg(0);
        if (cmd.equals("reset")) {
            int rootBlockSize = params.getInt("rootBlockSize");
            int digestSize = ctx.digestSize() >> 3;
            byte[] zeroes = new byte[20];
            OutputStream out;
            
            out = WriteLock.getOutputStream(fs, 0,
                                            digestSize - 1);
            out.write(zeroes);
            out.close();
            
            out = WriteLock.getOutputStream(fs, rootBlockSize,
                                            rootBlockSize + digestSize - 1);
            out.write(zeroes);
            out.close();
        }
        else if (cmd.equals("check")) {
            FSDirectoryRoot root = new FSDirectoryRoot(fs, ctx,
                                                       params.getInt("rootBlockSize"));
            AccountingTable acct = new AccountingTable(fs, root.getRanges(), ctx,
                                                       params.getInt("acctBlockSize"));
            String check = params.getArg(1);
            int rc = 0;
            if (check == null || check.equals("all")) {
                //rc |= checkProc(acct);
                rc |= checkSyntax(acct);
                rc |= checkDir(acct);
            }
            else if (check.equals("proc")) {
                rc = checkProc(acct);
            }
            else if (check.equals("syntax")) {
                rc = checkSyntax(acct);
            }
            else if (check.equals("dir")) {
                rc = checkDir(acct);
            }
            else {
                usage();
            }
            System.exit(rc);
        }
        else if (cmd.equals("repair")) {
            FSDirectoryRoot root = new FSDirectoryRoot(fs, ctx,
                                                       params.getInt("rootBlockSize"));
            AccountingTable acct = new AccountingTable(fs, root.getRanges(), ctx,
                                                       params.getInt("acctBlockSize"));
            FaultAnalysis[] checks = FSDirectoryCheck.checkSemantics(acct);
            int major = params.getInt("major");
            for (int i=0; i<checks.length; ++i) {
                if (checks[i].hasFaults()) {
                    boolean needsFlush = false;
                    System.out.println("Repairing: "+checks[i].getDescription());
                    Fault f;
                    try {
                        while (null != (f = checks[i].getNextFault())) {
                            if (f.getSeverity() == Fault.FATAL) {
                                System.err.println("Cannot repair FSDirectory;  " +
                                                   "contains FATAL faults");
                                System.exit(1);
                            }
                            if (f.getSeverity() == Fault.MAJOR && major-- <= 0) {
                                System.out.print("Skipping: (");
                                System.out.print(getSeverityName(f.getSeverity()));
                                System.out.print(") ");
                                System.out.println(f.getDescription());
                            }
                            else {
                                System.out.print("Fixing: (");
                                System.out.print(getSeverityName(f.getSeverity()));
                                System.out.print(") ");
                                System.out.println(f.getDescription());
                                f.fix();
                                needsFlush = true;
                            }
                        }
                    }
                    finally {
                        if (needsFlush) {
                            System.out.println("Flushing: "+checks[i].getDescription());
                            checks[i].commitFixes();
                        }
                    }
                }
            }
        }
        else if (cmd.equals("range")) {
            long lo, hi;
            if (params.getBoolean("hex")) {
                lo = Fields.hexToLong(params.getArg(1));
                hi = Fields.hexToLong(params.getArg(2));
            }
            else {
                lo = Long.parseLong(params.getArg(1));
                hi = Long.parseLong(params.getArg(2));
            }
            InputStream in = ReadLock.getInputStream(fs, lo, hi);
            copyStream(ReadLock.getInputStream(fs, lo, hi), System.out, hi+1-lo);
        }
        else if (cmd.equals("rawblock")) {
            int bnum = Integer.parseInt(params.getArg(1));
            FSDirectoryRoot root = new FSDirectoryRoot(fs, ctx,
                                                       params.getInt("rootBlockSize"));
            AccountingTable acct = new AccountingTable(fs, root.getRanges(), ctx,
                                                       params.getInt("acctBlockSize"));
            byte[] block = acct.getBlock(bnum);
            if (block == null)
                throw new IllegalArgumentException("block number out of bounds");
            System.out.write(block);
            System.out.flush();
        }
        else if (cmd.equals("fsdump")) {
            FSDirectory dir = new FSDirectory(fs, ctx,
                                              params.getInt("rootBlockSize"),
                                              params.getInt("acctBlockSize"),
                                              0, 0);
            
            PrintWriter out = new PrintWriter(System.out);
            
            String what = params.getArg(1);
            if (what == null)
                FSDirectory.dump(dir, out);
            else if (what.equals("keys"))
                FSDirectory.dumpFiles(dir, out);
            else if (what.equals("lru"))
                FSDirectory.dumpLRUFiles(dir, out);
            else 
                FSDirectory.dump(dir, out);
            
            out.flush();
        }
        else if (cmd.equals("dsdump")) {
            Directory dir = new FSDirectory(fs, ctx,
                                            params.getInt("rootBlockSize"),
                                            params.getInt("acctBlockSize"),
                                            0, 0);
            LossyDirectory dsDir = new LossyDirectory(0x0001, dir);
            FSDataStore ds = new FSDataStore(dsDir, fs.size() / 50);
            
            PrintWriter out = new PrintWriter(System.out);
            
            String what = params.getArg(1);
            if (what == null)
                FSDataStore.dump(ds, out);
            else if (what.equals("keys"))
                FSDataStore.dumpCommittedKeys(ds, out);
            else if (what.equals("lru"))
                FSDataStore.dumpLRUKeys(ds, out);
            else 
                FSDataStore.dump(ds, out);
            
            out.flush();
        }
        else if (cmd.equals("scan")) {
            FSDirectoryRoot root = new FSDirectoryRoot(fs, ctx,
                                                       params.getInt("rootBlockSize"));
            AccountingTable acct = new AccountingTable(fs, root.getRanges(), ctx,
                                                       params.getInt("acctBlockSize"));
            AccountingInitializer init = new AccountingInitializer(acct);
            PrintWriter out = new PrintWriter(System.out);
            AccountingInitializer.dump(init, out);
            out.flush();
        }
        else if (cmd.equals("block")) {
            int bnum = Integer.parseInt(params.getArg(1));
            FSDirectoryRoot root = new FSDirectoryRoot(fs, ctx,
                                                       params.getInt("rootBlockSize"));
            AccountingTable acct = new AccountingTable(fs, root.getRanges(), ctx,
                                                       params.getInt("acctBlockSize"));
            DataInput din = acct.readBlock(bnum);
            if (din == null)
                throw new IllegalArgumentException("block number out of bounds");
            if (din.readUnsignedShort() != 0)
                throw new IllegalArgumentException("not a data block");
            PrintWriter out = new PrintWriter(System.out);
            out.print("Struct: 0x");
            out.println(Integer.toHexString(din.readUnsignedShort()));
            SerialTree.dumpRecords(din, out);
            out.flush();
        }
        else if (cmd.equals("transaction")) {
            int txid = Integer.parseInt(params.getArg(1));
            FSDirectoryRoot root = new FSDirectoryRoot(fs, ctx,
                                                       params.getInt("rootBlockSize"));
            AccountingTable acct = new AccountingTable(fs, root.getRanges(), ctx,
                                                       params.getInt("acctBlockSize"));
            AccountingInitializer init = new AccountingInitializer(acct);
            Enumeration bte = init.getBlockTransactions();
            while (bte.hasMoreElements()) {
                BlockTransaction btx = (BlockTransaction) bte.nextElement();
                if (txid == btx.getTransactionID()) {
                    DataInput din = btx.readAnnotation();
                    PrintWriter out = new PrintWriter(System.out);
                    out.print("Struct: 0x");
                    out.println(Integer.toHexString(din.readUnsignedShort()));
                    SerialTree.dumpRecords(din, out);
                    out.flush();
                    return;
                }
            }
            throw new IllegalArgumentException("no such transaction ID");
        }
        else if (cmd.equals("freeze")) {
            FSDirectory dir = new FSDirectory(fs, ctx,
                                              params.getInt("rootBlockSize"),
                                              params.getInt("acctBlockSize"),
                                              0, 0);
            dir.freeze();
        }
        else usage();
    }


    private static int checkProc(AccountingTable acct) throws IOException {
        System.out.println("Sorry pal, 'check proc' is not implemented yet.");
        return 0;
    }

    private static int checkSyntax(AccountingTable acct) throws IOException {
        System.out.println("Checking syntax of serialized accounting trees...");
        FaultAnalysis[] checks = FSDirectoryCheck.checkSyntax(acct);
        int bad = 0;
        for (int i=0; i<checks.length; ++i) {
            if (checks[i].hasFaults()) {
                ++bad;
                System.out.println();
                System.out.println("Found errors in: "+checks[i].getDescription());
                System.out.println("------------------------------------------------------------");
                Fault f;
                while (null != (f = checks[i].getNextFault())) {
                    System.out.print('(');
                    System.out.print(getSeverityName(f.getSeverity()));
                    System.out.print(") ");
                    System.out.println(f.getDescription());
                }
                System.out.println("------------------------------------------------------------");
            }
            else {
                System.out.println(checks[i].getDescription() + " .. OK");
            }
        }
        return bad;
    }

    private static int checkDir(AccountingTable acct) throws IOException {
        System.out.println("Checking consistency of directory accounting data...");
        FaultAnalysis[] checks = FSDirectoryCheck.checkSemantics(acct);
        int bad = 0;
        for (int i=0; i<checks.length; ++i) {
            if (checks[i].hasFaults()) {
                ++bad;
                System.out.println();
                System.out.println("Found errors in: "+checks[i].getDescription());
                System.out.println("------------------------------------------------------------");
                Fault f;
                while (null != (f = checks[i].getNextFault())) {
                    System.out.print('(');
                    System.out.print(getSeverityName(f.getSeverity()));
                    System.out.print(") ");
                    System.out.println(f.getDescription());
                }
                System.out.println("------------------------------------------------------------");
            }
            else {
                System.out.println(checks[i].getDescription() + " .. OK");
            }
        }
        return bad;
    }
    

    private static final String getSeverityName(int severity) {
        switch (severity) {
            case Fault.MINOR:   return "MINOR";
            case Fault.MAJOR:   return "MAJOR";
            case Fault.FATAL:   return "FATAL";
            default:            return "(?)";
        }
    }
    

    private static void usage() throws IOException {
        System.out.println("Usage: java "+FSTool.class.getName()+" [options] <command>");
        System.out.println();
        System.out.println("  Options");
        System.out.println("  -------");
        switches.printUsage(System.out);
        System.out.println();
        System.out.println("By default, the node_<port> and store_<port> files are looked for");
        System.out.println("in the current directory if not specified.");
        System.out.println();
        System.out.println("  Commands");
        System.out.println("  --------");
        System.out.println("  reset               .. wipe out everything");
        System.out.println("  check proc          .. check accounting-process integrity");
        System.out.println("  check syntax        .. check that tree records are well-formed");
        System.out.println("  check dir           .. check semantic consistency of directory data");
        System.out.println("  check [all]         .. check proc, syntax, and dir");
        System.out.println("  repair              .. attempt to repair faults");
        System.out.println("  range <lo> <hi>     .. get binary contents of arbitrary fragment");
        System.out.println("  rawblock <n>        .. get binary contents of accounting block");
        System.out.println("  fsdump [keys|lru]   .. dump file-system info");
        System.out.println("  dsdump [keys|lru]   .. dump DataStore info");
        System.out.println("  scan                .. do initialization scan");
        System.out.println("  block <n>           .. read a data block");
        System.out.println("  transaction <id>    .. read a transaction");
        System.out.println("  freeze              .. freeze all transactions");
        System.out.println();
        System.out.println("All output is written to the standard output.");
        System.exit(1);
    }


    private static FieldSet readKeys(String keysFile) throws IOException {
        InputStream in = new FileInputStream(keysFile);
        try {
            return new FieldSet(new CommentedReadInputStream(in, "#"));
        }
        finally {
            in.close();
        }
    }
    

    private static void copyStream(InputStream in, OutputStream out, long length)
                                                            throws IOException {
        byte[] buf = new byte[0x1000];
        while (length > 0) {
            int n = in.read(buf, 0, (int) Math.min(length, buf.length));
            if (n == -1) throw new EOFException();
            out.write(buf, 0, n);
            length -= n;
        }
    }
    

                    
    private static String locateFile(String prefix) throws FileNotFoundException {
        File currentDir = new File(System.getProperty("user.dir"));
        String[] found = currentDir.list(new PrefixFilter(prefix));
        if (found.length == 0)
            throw new FileNotFoundException("no "+prefix+" file found");
        return found[0]; 
    }

    private static final class PrefixFilter implements FilenameFilter {
        
        private final String prefix;
        
        PrefixFilter(String prefix) {
            this.prefix = prefix;
        }
        
        public final boolean accept(File dir, String name) {
            return name.startsWith(prefix);
        }
    }
}


