package freenet.node;

import freenet.*;
import freenet.crypt.*;
import freenet.config.*;
import freenet.support.*;
import freenet.support.io.*;
import freenet.fs.dir.Directory;
import freenet.node.ds.*;
import freenet.node.rt.*;
import freenet.presentation.FreenetProtocol;
import freenet.transport.*;
import freenet.session.*;
import freenet.diagnostics.*;
import freenet.thread.ThreadFactory;
import freenet.interfaces.Interface;
import freenet.node.states.maintenance.*;
import freenet.client.ClientFactory;
import freenet.client.InternalClient;
import freenet.message.Request;
import java.util.*;
import java.net.*;
import java.io.*;

/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.
*/

/**
 * This is a Wrapper object that contains the components of a Node in the
 * Adaptive Network.  It uses a Network object to communicate with other
 * Nodes in the Network.
 *
 * @author <A HREF="mailto:I.Clarke@strs.co.uk">Ian Clarke</A>
 * @author <a href="mailto:blanu@uts.cc.utexas.edu">Brandon Wiley</a>
 */

public class Node extends Core {

    static {
        int port = 5001 + Math.abs(randSource.nextInt()) % (65536 - 5001);

        // internal defaults
        config.addOption("rtMaxRefs",           1, 100,  1300);   // 100 refs/node
        config.addOption("rtMaxNodes",          1, 100,   1301);  // to 100 nodes
        config.addOption("messageStoreSize",    1, 500,   1350);  // 500 live chains
        config.addOption("failureTableSize",    1, 1000,  1360); // 1000 failed keys
        config.addOption("failureTableTime",    1, 1800000l, 1361); // 30 min

        // network defaults
        config.addOption("routeConnectTimeout", 1, 10000, 1400);  // 10 sec
        config.addOption("maxHopsToLive",       1, 15,    1410);

        // network resource limiting options
        config.addOption("bandwidthLimit",       1, 100000, 1200);  // 100k/s
        config.addOption("inputBandwidthLimit",  1, 0,      1201);  // disabled
        config.addOption("outputBandwidthLimit", 1, 0,      1202);  // disabled
        config.addOption("maxNodeConnections",   1, 60,     1225);
        
        // data store settings
        config.addOption("nodeFile",         1, "",            1000);  // node_<port>
        config.addOption("storeFile",        1, "",            1001);  // store_<port>
        config.addOption("storeSize",        1, 200L*(1L<<20), 1010);  // 200 megs
        config.addOption("storeCipherName",  1, "Twofish",     1020);  // Twofish cipher
        config.addOption("storeCipherWidth", 1, 128,           1021);  // 128 bits
        
        // network settings
        config.addOption("ipAddress",  1, "",              100);  // must be set by user
        config.addOption("listenPort", 1, port,            101);  // random choice
        config.addOption("clientPort", 1, 8481,            110);
        config.addOption("fcpHosts",   1, "",              112);  // loopback only
        config.addOption("transient",  1, false,           300);
        config.addOption("seedFile",   1, "seednodes.ref", 320);
        
        // logging options
        config.addOption("logLevel",  1, "normal",      1250);     
        config.addOption("logFile",   1, "freenet.log", 1251);
        config.addOption("logFormat", 1, "d (c, t): m", 1252);
        config.addOption("logDate",   1, "",            1253);  // locale default
        
        // diagnostics options
        config.addOption("doDiagnostics",   1, true,    500);  // defaults to doing them
        config.addOption("diagnosticsPath", 1, "stats", 501);
        
        // announcement options
        config.addOption("doAnnounce",            1, true,          310);
        config.addOption("localAnnounceTargets",  1, String.class, 5000);
        config.addOption("announcementPeers",     1, 15,           1501);  // 15 for now
        config.addOption("announcementAttempts",  1, 10,           1502);
        config.addOption("announcementDelay",     1, 1800000,      1511);  // 30 minutes
        config.addOption("announcementDelayBase", 1, 2,            1512);  // double
        config.addOption("initialRequests",       1, 10,           1520);
        config.addOption("initialRequestHTL",     1, 15,           1521);

        // FCP admin options
        config.addOption("adminPassword", 1, String.class, 200);
        config.addOption("adminPeer",     1, String.class, 201);
        
        

        // RouteConnectTimeout
        config.setExpert ("routeConnectTimeout", true);
        config.argDesc   ("routeConnectTimeout", "<millis>");
        config.shortDesc ("routeConnectTimeout", "wait on new connection when routing.");
        config.longDesc  ("routeConnectTimeout",
            "The time to wait for connections to be established and ",
            "authenticated before passing by a node while routing out.",
            "Connections that are by passed are still finished and cached ", 
            "for the time set by <connectionTimeout> (in milliseconds)."
        );

        // maxHopsToLive
        config.setExpert ("maxHopsToLive", true);
        config.argDesc   ("maxHopsToLive", "<integer>");
        config.shortDesc ("maxHopsToLive", "max HTL allowed on routed requests");
        config.longDesc  ("maxHopsToLive",
            "When forwarding a request, the node will reduce the HTL to this value",
            "if it is found to be in excess."                                       
        );


        // nodeFile
        config.setExpert ("nodeFile", true);
        config.argDesc   ("nodeFile", "<file>");
        config.shortDesc ("nodeFile", "location of node's key file");
        config.longDesc  ("nodeFile",
            "The path to the file containing the node's private key, DSA group,",
            "cipher key, etc.  Defaults to node_<port> in the current directory."
        );

        // storeFile
        config.setExpert ("storeFile", true);
        config.argDesc   ("storeFile", "<file>[,..]");
        config.shortDesc ("storeFile", "location of data store file(s)");
        config.longDesc  ("storeFile",
            "The path to a single file, or a comma-separated list of files,",
            "containing the data store.  The size of each file is given by",
            "<storeSize>."
        );

        // storeSize
        config.argDesc   ("storeSize", "<bytes>");
        config.shortDesc ("storeSize", "size of the data store file(s)");
        config.longDesc  ("storeSize",
            "The byte size of each data store file.  If there is more than one",
            "file, the total size of the store is the product of the number of",
            "files and <storeSize>."
        );
        
        // storeCipherName
        config.setExpert ("storeCipherName", true);
        config.argDesc   ("storeCipherName", "<string>");
        config.shortDesc ("storeCipherName", "name of symmetric cipher algorithm");
        config.longDesc  ("storeCipherName",
            "The name of a symmetric cipher algorithm to encrypt the datastore",
            "contents with.  Supported algorithms are \"Twofish\", \"Rijndael\",",
            "and \"null\", \"none\", or \"void\" (for no encryption)."
        );
        
        // storeCipherWidth
        config.setExpert ("storeCipherWidth", true);
        config.argDesc   ("storeCipherWidth", "<integer>");
        config.shortDesc ("storeCipherWidth", "bit-width of cipher key");
        config.longDesc  ("storeCipherWidth",
            "The width in bits of the cipher key to use for the datastore.",
            "The allowed values for this will depend on the cipher algorithm.",
            "Twofish allows 64, 128, 192, or 256, while Rijndael allows",
            "128, 192, or 256."
        );

        // rtMaxRefs
        config.setExpert ("rtMaxRefs", true);
        config.argDesc   ("rtMaxRefs", "<integer>");
        config.shortDesc ("rtMaxRefs", "max no. of refs per node");
        config.longDesc  ("rtMaxRefs",
            "The number of references allowed per node in the routing table.",
            "This should not be set too high."
        );
        
        // rtMaxNodes
        config.setExpert ("rtMaxNodes", true);
        config.argDesc   ("rtMaxNodes", "<integer>");
        config.shortDesc ("rtMaxNodes", "max no. unique nodes in routing table");
        config.longDesc  ("rtMaxNodes",
            "The number of unique nodes that can be contained in the routing table."
        );
        
        // messageStoreSize
        config.setExpert ("messageStoreSize", true);
        config.argDesc   ("messageStoreSize", "<integer>");
        config.shortDesc ("messageStoreSize", "max no. of simultaneous requests.");
        config.longDesc  ("messageStoreSize",
            "The number of outstanding message replies the node will",
            "wait for before it starts to abandon them."              
        );

        // failureTableSize
        config.setExpert ("failureTableSize", true);
        config.argDesc   ("failureTableSize", "<integer>");
        config.shortDesc ("failureTableSize", "max. no. cached failed keys.");
        config.longDesc  ("failureTableSize",
             "The number keys that failed to be retrieved the node should key track of."
        );

        // failureTableTime
        config.setExpert ("failureTableTime", true);
        config.argDesc   ("failureTableTime", "<milliseconds>");
        config.shortDesc ("failureTableTime", "max. time to fail keys.");
        config.longDesc  ("failureTableTime",
             "The amount of time to keep keys cache keys that could not be found and",
             "automatically fail requests for them.");

        // bandwidthLimit
        config.argDesc   ("bandwidthLimit", "<bytes/sec>");
        config.shortDesc ("bandwidthLimit", "total bandwidth limit, in & out");
        config.longDesc  ("bandwidthLimit",
            "The maximum number of bytes per second to transmit, totaled between",
            "incoming and outgoing connections.  Ignored if either inputBandwidthLimit",
            "or outputBandwidthLimit is nonzero."
        );
        
        // inputBandwidthLimit
        config.argDesc   ("inputBandwidthLimit", "<bytes/sec>");
        config.shortDesc ("inputBandwidthLimit", "incoming bandwidth limit");
        config.longDesc  ("inputBandwidthLimit",
            "If nonzero, specifies an independent limit for incoming data only.",
            "(overrides bandwidthLimit if nonzero)"
        );
        
        // outputBandwidthLimit
        config.argDesc   ("outputBandwidthLimit", "<bytes/sec>");
        config.shortDesc ("outputBandwidthLimit", "outgoing bandwidth limit");
        config.longDesc  ("outputBandwidthLimit",
            "If nonzero, specifies an independent limit for outgoing data only.",
            "(overrides bandwidthLimit if nonzero)"
        );

        // maxNodeConnections
        config.setExpert ("maxNodeConnections", true);
        config.argDesc   ("maxNodeConnections", "<int>");
        config.shortDesc ("maxNodeConnections", "Max. no. of connections to other nodes.");
        config.longDesc  ("maxNodeConnections", 
                          "The maximum number of incoming and outgoing connections to allow ", 
                          "at the same time.");
        
        // ipAddress
        config.argDesc   ("ipAddress", "xxx.xxx.xxx.xxx");
        config.shortDesc ("ipAddress", "your IP as seen by the public internet");
        config.longDesc  ("ipAddress",
            "The IP address of this node as seen by the public internet.  This is",
            "needed in order for the node to determine its own node reference.",
            "If you have a dynamic IP address, you may enter a host name in this",
            "field (assuming you have a dynamic DNS service).  If this is a",
            "transient node, you can leave this blank."
        );
        
        // listenPort
        config.argDesc   ("listenPort", "<port no.>");
        config.shortDesc ("listenPort", "incoming FNP port");
        config.longDesc  ("listenPort",
            "The port to listen for incoming FNP (Freenet Node Protocol) connections on."
        );
        
        // clientPort
        config.argDesc   ("clientPort", "<port no.>");
        config.shortDesc ("clientPort", "incoming FCP port");
        config.longDesc  ("clientPort",
            "The port to listen for local FCP (Freenet Client Protocol) connections on."
        );

        // fcpHosts
        config.argDesc   ("fcpHosts", "<host list>");
        config.shortDesc ("fcpHosts", "hosts allowed to connect with FCP");
        config.longDesc  ("fcpHosts",
            "A comma-separated list of hosts that may connect to the FCP port",
            "(clientPort).  If left blank, only the localhost will be allowed.",
            "May be given as IP addresses or host names."
        );
        
        // logLevel
        config.argDesc   ("logLevel", "<word>");
        config.shortDesc ("logLevel", "error, normal, minor, or debug");
        config.longDesc  ("logLevel",
            "The error reporting threshold, one of:",
            "  Error:   Errors only",
            "  Normal:  Report significant events",
            "  Minor:   Report minor events",
            "  Debug:   Report events only of relevance when debugging"
        );
        
        // logFile
        config.setExpert ("logFile", true);
        config.argDesc   ("logFile", "<filename>|NO");
        config.shortDesc ("logFile", "path to the log file, or NO for STDERR");
        config.longDesc  ("logFile",
            "The name of the log file (`NO' to log to standard out)"
        );
        
        // logFormat
        config.setExpert ("logFormat", true);
        config.argDesc   ("logFormat", "<tmpl.>");
        config.shortDesc ("logFormat", "template, like d:c:h:t:p:m");
        config.longDesc  ("logFormat",
            "A template string for log messages.  All non-alphabet characters are",
            "reproduced verbatim.  Alphabet characters are substituted as follows:",
            "d = date (timestamp), c = class name of the source object,",
            "h = hashcode of the object, t = thread name, p = priority,",
            "m = the actual log message"
        );

        // logDate
        config.setExpert ("logDate", true);
        config.argDesc   ("logDate", "<tmpl.>");
        config.shortDesc ("logDate", "java style date/time template");
        config.longDesc  ("logDate",
            "A template for formatting the timestamp in log messages.  Defaults to",
            "the locale specific fully specified date format.  The template string",
            "is an ordinary Java date/time template - see:",
            "http://java.sun.com/products/jdk/1.1/docs/api/java.text.SimpleDateFormat.html"
        );
        
        // seedFile
        config.argDesc   ("seedFile", "<file>");
        config.shortDesc ("seedFile", "initial node ref(s), for announcing");
        config.longDesc  ("seedFile",
            "A file containing one or more node references which will be incorporated",
            "into the node's routing table on startup.  A reference is only added if",
            "there is no previously existing reference to that node.  When this node",
            "announces, it will announce to the nodes listed in this file."
        );

        // doAnnounce
        config.argDesc   ("doAnnounce", "yes|no");
        config.shortDesc ("doAnnounce", "whether to automatically announce");
        config.longDesc  ("doAnnounce",
            "If this is true, the node will automatically announce to all nodes in",
            "the <seedFile> file, as specified by <announcementDelay>, etc."
        );

        // localAnnounceTargets
        config.setExpert ("localAnnounceTargets", true);
        config.argDesc   ("localAnnounceTargets", "<list>");
        config.shortDesc ("localAnnounceTargets", "debugging option");
        config.longDesc  ("localAnnounceTargets", "Used only in debugging.");

        // announcementPeers
        config.setExpert ("announcementPeers", true);
        config.argDesc   ("announcementPeers", "<integer>");
        config.shortDesc ("announcementPeers", "no. of nodes announcement goes to");
        config.longDesc  ("announcementPeers",
            "An announcement must visit announcementPeers nodes to be successful.",
            "If there is more than one seed node, announcements will be sent to",
            "them all, and the HTL of each announcement will be given by:",
            "HTL = announcementPeers / (no. of seed nodes)"
        );
        
        // announcementAttempts
        config.setExpert ("announcementAttempts", true);
        config.argDesc   ("announcementAttempts", "<integer>");
        config.shortDesc ("announcementAttempts", "number of attempts to announce");
        config.longDesc  ("announcementAttempts",
            "The number of attempts to make at announcing this node per",
            "initial peer. Zero means the node will not announce itself");

        // announcementDelay
        config.setExpert ("announcementDelay",true);
        config.argDesc   ("announcementDelay","<integer>");
        config.shortDesc ("announcementDelay","The delay before announcing");
        config.longDesc  ("announcementDelay",
                          "The amount of time to wait before initially announcing the node,",
                          "and to base the time the time between retries on. In milliseconds.");

        // announcementDelayBase
        config.setExpert ("announcementDelayBase",true);
        config.argDesc   ("announcementDelayBase","<integer>");
        config.shortDesc ("announcementDelayBase",
                          "Multiple for delay time after failure");
        config.longDesc  ("announcementDelayBase",
                          "The value to mutliply the last delay time with for each retry.",
                          "That is, for try N, we weight <announcementDelay>*<announcementDelay>^N",
                          "before starting.");

        // initialRequests
        config.setExpert ("initialRequests", true);
        config.argDesc   ("initialRequests","<integer>");
        config.shortDesc ("initialRequests", "number of initial requests");
        config.longDesc  ("initialRequests", 
             "The number of keys to request from the returned close values",
             "after an Announcement (this is per announcement made)."
        );

        // initialRequestHTL
        config.setExpert ("initialRequestHTL", true);
        config.argDesc   ("initialRequestHTL", "<integer>");
        config.shortDesc ("initialRequestHTL",
                          "HopsToLive on initial requests");
        config.longDesc  ("initialRequestHTL",
                          "The hops that initial requests should make.");

        // adminPassword
        config.argDesc   ("adminPassword", "<string>");
        config.shortDesc ("adminPassword", 
                          "allows remote admin using password.");
        config.longDesc  ("adminPassword",
               "If this is set then users that can provide the password can",
               "can have administrative access. It is recommended that",
               "you do not use this without also using adminPeer below",
               "in which case both are required.");

        // adminPeer
        config.setExpert ("adminPeer", true);
        config.argDesc   ("adminPeer","<Identity FieldSet>");
        config.shortDesc ("adminPeer",
                          "allows remote admin using PKI");
        config.longDesc  ("adminPeer",
              "If this is set, then users that are authenticated owners",
              "of the given PK identity can have administrative access.",
              "If adminPassword is also set both are required.");

        // transient
        config.argDesc   ("transient", "yes|no");
        config.shortDesc ("transient", "a transient node is non-permanent");
        config.longDesc  ("transient",
            "Transient nodes do not give out references to themselves, and should",
            "therefore not receive any requests.  Set this to yes if you cannot",
            "receive incoming connections, or cannot keep the computer continuously",
            "online."
        );

        // doDiagnostics
        config.argDesc   ("doDiagnostics", "yes|no");
        config.shortDesc ("doDiagnostics", "whether to collect statistics");
        config.longDesc  ("doDiagnostics",
            "The diagnostics module receives and aggregates statistics about",
            "Freenet's performance.  This will eat some gratuitous memory and",
            "CPU time but may let you provide valuable data to the project."
        );

        // diagnosticsPath
        config.setExpert ("diagnosticsPath", true);
        config.argDesc   ("diagnosticsPath", "<dir>");
        config.shortDesc ("diagnosticsPath", "directory to save statistics in");
        config.longDesc  ("diagnosticsPath",
            "The directory in which to save diagnostics data.  Defaults to",
            "<storePath>/stats if left blank."
        );

        // Options for NodeStatusServlet.
        config.addOption("nodestatus.class", 1, "freenet.client.http.NodeStatusServlet", 2040);
        config.addOption("nodestatus.port", 1, 8889, 2041);

        // Options for fproxy service.
        config.addOption("fproxy.class", 1, "freenet.client.http.FproxyServlet", 2000);
        config.addOption("fproxy.port", 1, 8888, 2001);
        config.addOption("fproxy.params.insertHtl", 1, 5, 2002);
        config.addOption("fproxy.params.requestHtl", 1, 15, 2003);
        config.addOption("fproxy.params.passThroughMimeTypes", 1, 
                         "text/plain,image/jpeg,image/gif,image/png", 2004);
        config.addOption("fproxy.params.filter", 1, true, 2005);
        // SplitFile specific options
        config.addOption("fproxy.params.splitFileThreads", 1, 5, 2006);
        config.addOption("fproxy.params.splitFileRetries", 1, 0, 2007);
        config.addOption("fproxy.params.splitFileRetryHtlIncrement", 1, 5, 2008);
        config.addOption("fproxy.params.pollForDroppedConnection", 1, true, 2009);
	config.addOption("fproxy.params.showNewBuildWarning", 1, true, 2010);

        // Turn on fproxy, nodeinfo and NodeStatusServlet by default.
        config.addOption("services", 1, "fproxy,nodestatus,nodeinfo", 2011);

        // fproxy Servlet class
        config.setExpert ("fproxy.class", true);
        config.argDesc   ("fproxy.class", "<class name>");
        config.shortDesc ("fproxy.class", "fproxy Servlet class");
        config.longDesc  ("fproxy.class",
            "The Java class for the fproxy HttpServlet. You shouldn't need to change this."
        );
        
        // fproxy port
        config.setExpert ("fproxy.port", true);
        config.argDesc   ("fproxy.port", "<port number>");
        config.shortDesc ("fproxy.port", "fproxy listen port");
        config.longDesc  ("fproxy.port",
            "The port that fproxy listens for HTTP requests on."
        );

        config.setExpert ("fproxy.params.insertHtl", true);
        config.argDesc   ("fproxy.params.insertHtl", "n");
        config.shortDesc ("fproxy.params.insertHtl", "fproxy insert htl");
        config.longDesc  ("fproxy.params.insertHtl",
                          "The default htl used when inserting via fproxy.");

        config.setExpert ("fproxy.params.requestHtl", true);
        config.argDesc   ("fproxy.params.requestHtl", "n");
        config.shortDesc ("fproxy.params.requestHtl", "fproxy request htl");
        config.longDesc  ("fproxy.params.requestHtl",
                          "The default htl used when requesting via fproxy.");


        config.setExpert ("fproxy.params.passThroughMimeTypes", true);
        config.argDesc   ("fproxy.params.passThroughMimeTypes", "<list>");
        config.shortDesc ("fproxy.params.passThroughMimeTypes", "Fproxy allowed mime types");
        config.longDesc  ("fproxy.params.passThroughMimeTypes",
                          "A comma delimited list of mime types that are allowed by fproxy's" +
                          " anonymity filter code. ");

        config.setExpert ("fproxy.params.filter", true);
        config.argDesc   ("fproxy.params.filter", "true/false");
        config.shortDesc ("fproxy.params.filter", "fproxy anonymity filter.");
        config.longDesc  ("fproxy.params.filter",
                          "Set true to enable fproxy's anonymity protection filtering.");

        config.setExpert ("fproxy.params.splitFileThreads", true);
        config.argDesc   ("fproxy.params.splitFileThreads", "n");
        config.shortDesc ("fproxy.params.splitFileThreads", "max SplitFile threads");
        config.longDesc  ("fproxy.params.splitFileThreads",
                          "Maximum number of threads per SplitFile request.");

        config.setExpert ("fproxy.params.splitFileRetries", true);
        config.argDesc   ("fproxy.params.splitFileRetries", "n");
        config.shortDesc ("fproxy.params.splitFileRetries", "SplitFile retry attempts");
        config.longDesc  ("fproxy.params.splitFileRetries",
                          "Number of times to retry failed SplitFile block requests.");

        config.setExpert ("fproxy.params.splitFileRetryHtlIncrement", true);
        config.argDesc   ("fproxy.params.splitFileRetryHtlIncrement", "n");
        config.shortDesc ("fproxy.params.splitFileRetryHtlIncrement", "SplitFile htl retry increment.");
        config.longDesc  ("fproxy.params.splitFileRetryHtlIncrement",
                          "Amount to increment SplitFile request htl each retry.");

        // "fproxy.pollForDroppedConnection" undocumented on purpose.
        config.setExpert ("fproxy.params.pollForDroppedConnection", true);
        config.argDesc   ("fproxy.params.pollForDroppedConnection", "true/false");
        config.shortDesc ("fproxy.params.pollForDroppedConnection", "poll for dropped SplitFile requests");
        config.longDesc  ("fproxy.params.pollForDroppedConnection",
                          "Set true to allow polling for dropped requests during SplitFile download.");

        // FProxy new build warning
        config.setExpert ("fproxy.params.showNewBuildWarning", true);
        config.argDesc   ("fproxy.params.showNewBuildWarning", "true/false");
        config.shortDesc ("fproxy.params.showNewBuildWarning", "Show a warning when a new build is available");
        config.longDesc  ("fproxy.params.showNewBuildWarning",
                          "Set to 'yes' to show a warning when a new build may be available.");

        // NodeStatusServlet class
        config.setExpert ("nodestatus.class", true);
        config.argDesc   ("nodestatus.class", "<class name>");
        config.shortDesc ("nodestatus.class", "NodeStatusServlet class");
        config.longDesc  ("nodestatus.class",
            "The Java class for the NodeStatusServlet. You shouldn't need to change this."
        );
        
        // NodeStatusServlet port
        config.setExpert ("nodestatus.port", true);
        config.argDesc   ("nodestatus.port", "<port number>");
        config.shortDesc ("nodestatus.port", "NodeStatusServlet listen port");
        config.longDesc  ("nodestatus.port",
            "The port that the node status servlet listens for HTTP requests on."
        );

        config.setExpert ("services", false);
        config.argDesc   ("services", "service_0,service_1,...");
        config.shortDesc ("services", "services run at start up");
        // REDFLAG: fix comment when fproxy turned on by default.
        config.longDesc  ("services",
                          "A comma delimited list of services that are run when the node starts. " +
                          "Enter 'fproxy' here to start fproxy automatically."
        );
	
	// Options for nodeinfo service
	config.addOption("nodeinfo.class", 1, "freenet.node.http.NodeInfoServlet", 2042);
	config.addOption("nodeinfo.port", 1, 8890, 2043);

	// NodeInfoServlet class
	config.setExpert("nodeinfo.class", true);
	config.argDesc("nodeinfo.class","<class name>");
	config.shortDesc ("nodeinfo.class", "NodeInfoServlet class");
	config.longDesc  ("nodeinfo.class",
			  "The Java class for the NodeInfoServlet. You shouldn't need to change this.");

	// NodeStatusServlet port
        config.setExpert ("nodeinfo.port", true);
        config.argDesc   ("nodeinfo.port", "<port number>");
        config.shortDesc ("nodeinfo.port", "NodeInfoServlet listen port");
        config.longDesc  ("nodeinfo.port",
			  "The port that the node information servlet listens for HTTP requests on.");
    }



    //=== Static elements ======================================================
    
    // node specific options
    static public int routeConnectTimeout;
    static public int maxHopsToLive;

    // announcement options
    static public int announcementPeers;
    static public int announcementAttempts;
    static public int announcementDelay;
    static public int announcementDelayBase;
    static public int initialRequests;
    static public int initialRequestHTL;
    //static public int maxForwardTries;

    // node data files
    static public String nodeFile, storeFile[];

    // node limits
    static public long storeSize;
    static public int rtMaxRefs, rtMaxNodes;

    // datastore encryption
    static public String storeCipherName;
    static public int storeCipherWidth;

    // So we can keep track of how long the
    // node has been running. Approximate, but good enough.
    static final public long startupTimeMs = System.currentTimeMillis();

    // remote admin stuff
    static private String adminPassword;
    static private Identity adminPeer;

    // thread management
    static protected ThreadFactory threadFactory;

    public static boolean isAuthorized(Identity id, String p) {
        if ( adminPassword != null && !adminPassword.equals("") 
             && adminPeer != null) {
            return adminPassword.equals(p) && adminPeer.equals(id);
        } else if (adminPassword != null && !adminPassword.equals("")) {
            return adminPassword.equals(p);
        } else if (adminPeer != null) {
            return adminPeer.equals(id);
        } else { // no remote admin
            return false;
        }
    }

    
    /**
     * @throws CoreException  if initialization has already occurred
     */
    public static void init(Params params) throws CoreException {
        // Init the core settings
        Core.init(params);

        // set keytypes 
        Key.addKeyType(freenet.keys.SVK.keyNumber, freenet.keys.SVK.class);
        Key.addKeyType(freenet.keys.CHK.keyNumber, freenet.keys.CHK.class);

        String[] keyTypes = params.getList("keyTypes");
        for (int i = 1; keyTypes != null && i < keyTypes.length ; i += 2) { 
            try {
                Key.addKeyType(Integer.parseInt(keyTypes[i-1]),
                               Class.forName(keyTypes[i]));
            } catch (ClassNotFoundException e) {
                Core.logger.log(Node.class,"No such class: " + keyTypes[i] +
                                " for Key type " + keyTypes[i-1],Logger.ERROR);
            } catch (ClassCastException e) {
                Core.logger.log(Node.class, 
                                "Class " + keyTypes[i] + " is not a Key",
                                Logger.ERROR);
            }
        }

        // set network parameters
        routeConnectTimeout = params.getInt("routeConnectTimeout");
        maxHopsToLive = params.getInt("maxHopsToLive");
        announcementPeers = params.getInt("announcementPeers");
        announcementAttempts = params.getInt("announcementAttempts");
        announcementDelay = params.getInt("announcementDelay"); // 30 minutes
        announcementDelayBase = params.getInt("announcementDelayBase"); 
        initialRequests = params.getInt("initialRequests");
        initialRequestHTL = params.getInt("initialRequestHTL");
        //maxForwardTries    = Defaults.getInt("maxForwardTries", params);

        int bandwidthLimit, inputBandwidthLimit, outputBandwidthLimit;
        bandwidthLimit       = params.getInt("bandwidthLimit");
        inputBandwidthLimit  = params.getInt("inputBandwidthLimit");
        outputBandwidthLimit = params.getInt("outputBandwidthLimit");
        
        Bandwidth ibw, obw;
        if (inputBandwidthLimit == 0 && outputBandwidthLimit == 0) {
            ibw = obw = new Bandwidth(bandwidthLimit);
        }
        else {
            ibw = new Bandwidth(inputBandwidthLimit);
            obw = new Bandwidth(outputBandwidthLimit);
        }
        ThrottledInputStream.setThrottle(ibw);
        ThrottledOutputStream.setThrottle(obw);

        // set storage parameters
        storeSize = params.getLong("storeSize");
        //maxFileSize = cacheSize / params.getInt("storeCacheCount");
        
        rtMaxRefs  = params.getInt("rtMaxRefs");
        rtMaxNodes = params.getInt("rtMaxNodes");

        storeCipherName = params.getString("storeCipherName");
        if (storeCipherName.equals("none")
            || storeCipherName.equals("null")
            || storeCipherName.equals("void")) storeCipherName = null;
            
        storeCipherWidth = params.getInt("storeCipherWidth");

        // get the listening port
        int port = params.getInt("listenPort");

        // locate the data files
        nodeFile = params.getString("nodeFile");
        if (nodeFile == null || nodeFile.equals(""))
            nodeFile = "node_" + port;

        storeFile = params.getList("storeFile");
        if (storeFile.length == 0 || storeFile[0] == null || storeFile[0].equals(""))
            storeFile = new String[] { "store_"+port };
        
        // set admin permisions

        String pword = params.getString("adminPassword");
        if (!"".equals(pword))
            adminPassword = pword;
        
        FieldSet peer = params.getSet("adminPeer");
        if (peer != null) {
            adminPeer = new DSAIdentity(peer);
        }
    }

    

    /**
     * Construct this node's NodeReference from the given private key.
     */
    static NodeReference makeNodeRef(Authentity priv, Address[] addr,
                                     SessionHandler sh, PresentationHandler ph) {
        long[] sessions, presentations;
        Enumeration e;

        sessions = new long[sh.size()];
        e = sh.getLinkManagers();
        for (int i=0; i<sessions.length; ++i)
            sessions[i] = ((LinkManager) e.nextElement()).designatorNum();

        presentations = new long[ph.size()];
        e = ph.getPresentations();
        for (int i=0; i<presentations.length; ++i)
            presentations[i] = ((Presentation) e.nextElement()).designatorNum();
    
        NodeReference nr = new NodeReference(priv.getIdentity(), addr,
                                             sessions, presentations,
                                             Version.getVersionString(), null);

        // FIXME - this method should accept an uncast Authentity
        nr.addSignature((DSAAuthentity) priv);

        return nr;
    }


    /**
     * The Directory that stores all the node's data.
     */
    public final Directory dir;

    /**
     * A source of temp file buckets.
     */
    public final BucketFactory bf;

    /**
     * The nodes table of cached documents.
     */    
    public final DataStore ds;

    /**
     * The routing table for routing requests on a key.
     */
    public final RoutingTable rt;

    /**
     * The table of recently failed keys.
     */
    public final FailureTable ft;

    
    /**
     * A node reference to this node. Basically this is who the node is
     * to the network.
     */
    public final NodeReference myRef;

    public final NodeReference getNodeReference() {
        return myRef;
    }


    private final boolean isTransient;
    
    /** @return  true if this is a transient node */
    public final boolean isTransient() {
        return isTransient;
    }

    
    /**
     * Internal client access.
     */
    public final ClientFactory client;

    
    /**
     * Creates a new Node.
     * @param privKey     The node's private key.
     * @param dir         Directory of the node's storage repository.
     * @param bf          The source for temp file buckets.
     * @param ds          The store where cached data is kept.
     * @param rt          The table for routing requests based on key values.
     * @param ft          The table that keeps track recently failed keys.
     * @param myRef       The NodeReference name to send to other 
     *                    nodes.
     * @param th          A transporthandler on which the available transports
     *                    are registered.
     * @param sh          The sessionhandler to use for making connections
     * @param ph          The presentationHandler to use for making connections
     * @param isTransient whether this node is transient
     */
    public Node(Authentity privKey, NodeReference myRef,
                Directory dir, BucketFactory bf, DataStore ds, RoutingTable rt,
                FailureTable ft, TransportHandler th, SessionHandler sh,
                PresentationHandler ph, boolean isTransient) {
        
        super(privKey, myRef.getIdentity(), th, sh, ph);
        this.myRef = myRef;
        this.dir = dir;
        this.bf = bf;
        this.ds = ds;
        this.ft = ft;
        this.rt = rt;
        this.client = new InternalClient(this);
        this.isTransient = isTransient;
    }

    
    /** @return  something for the logs.. */
    public final String toString() {
        return "Freenet Node: " + Fields.bytesToHex(identity.fingerprint());
    }

    /**
     * Returns a peer object for which a connection with a node can be
     * made.
     * @param nr  The node for which a Peer object is needed.
     */
    public final Peer getPeer(NodeReference nr) {
        return nr.getPeer(transports, sessions, presentations);
    }

    /*
    public void begin(ThreadFactory tf, Ticker t,
                      OpenConnectionManager ocm, Interface[] inter,
                      boolean daemon) throws CoreException {
        
        super.begin(tf, t, ocm, inter, daemon);

        //(new StoreCheckpoint(randSource.nextLong(),
        //                     1000 * checkpointInterval)).schedule(this);
        //(new LinkCleanup(randSource.nextLong(),
        //                 1000 * 60 * 5, sessions)).schedule(this);

        schedule(15 * 1000, new DiagnosticsPeriod(randSource.nextLong()));
    }
    */


    public final LoadStats loadStats = new LoadStats(System.currentTimeMillis(), 50);

    // Number of outbound requests per minute the node will make.
    public final LimitCounter outboundRequestLimit = new LimitCounter(60000, 60);

    /**
     * @return Number of jobs running or queued for execution by
     *         the node's ThreadManager. Can return -1 if no thread manager is
     *         being used.
     **/
    public final int activeJobs() {
        return threadFactory.activeThreads();
    }

    /**
     * @return true if the Node is QueryRejecting inbound requests, false otherwise.
     **/
    public boolean rejectingRequests() {
        if (outboundRequestLimit.exceeded()) {
            return true;
        }

        return activeJobs() > (2 * maximumThreads / 3);
    }

    /**
     * Hook for rate limiting.
     */
    public boolean acceptRequest(Request req) {

        int requestThreadCutoff = 2 * maximumThreads / 3;

        String diagAddr = null;

        if (inboundRequests != null) {
            diagAddr = req.source.peerAddress().toString();
            inboundRequests.incTotal(diagAddr);
        }

        if (activeJobs() > requestThreadCutoff) {
            //System.err.println("Node.acceptRequest -- exceeded thread limit: " + 
            //                   threadManager.activeJobs());
            return false;
        }

        // I tried just using the number of threads, but
        // it seemed to overshoot too easily, causing inbound connections
        // to be rejected. --gj
        if (outboundRequestLimit.exceeded()) {
            //System.err.println("Node.acceptRequest -- rate limit exceeded.");
            return false;
        }

        if (inboundRequests != null) {
            inboundRequests.incSuccesses(diagAddr);
        }

        return true;
    }


    //=== communications methods ===============================================

    /**
     * Returns a connection that messages can be sent on. This saves a 
     * little time as getPeer is only called if a free connection isn't
     * available.
     * @param nr        The node to connect to.
     * @param timeout   The time to allow in connecting.
     */
    public ConnectionHandler makeConnection(NodeReference nr, long timeout) 
                                            throws CommunicationException {
        Peer p = getPeer(nr);
        if (p == null)
            throw new ConnectFailedException(new VoidAddress(), nr.getIdentity(),
                                             "Unusable node ref", true);
        return makeConnection(p, timeout);
    }

    public final ConnectionHandler makeConnection(NodeReference nr)
                                            throws CommunicationException {
        return makeConnection(nr, 0);
    }


    /**
     * Sends a message accross an open or new connection. This saves a 
     * little time as getPeer is only called if a free connection isn't
     * available.
     * @param nr        The node to connect to.
     * @param m         The message to send.
     * @param timeout   The time to allow in connecting.
     * @return  The trailing field stream (if there is one).
     */
    public final OutputStream sendMessage(Message m, NodeReference nr, long timeout) 
                                                    throws CommunicationException {
        return makeConnection(nr, timeout).sendMessage(m);
    }

    public final OutputStream sendMessage(Message m, NodeReference nr)
                                                    throws CommunicationException {
        return sendMessage(m, nr, 0);
    }
    

    public final OutputStream sendMessage(Message m, Peer p, long timeout) 
                                                    throws CommunicationException {
        return makeConnection(p, timeout).sendMessage(m);
    }

    public final OutputStream sendMessage(Message m, Peer p)
                                                    throws CommunicationException {
        return sendMessage(m, p, 0);
    }
}







