package freenet.node.ds;

import freenet.*;
import freenet.fs.dir.*;
import freenet.support.io.*;
import java.io.*;

/**
 * Device to manage a storage entry.
 * @author tavin
 */
final class FSDataStoreElement {

    private final FSDataStore ds;
        
    private final Key key;
    private final Buffer buffer;
        
    private final long physLen;

    private int failureCode = -1;

    private boolean newData = false;

    private int users = 0;

        
    FSDataStoreElement(FSDataStore ds, Key key, Buffer buffer, long physLen) {
        this.ds = ds;
        this.key = key;
        this.buffer = buffer;
        this.physLen = physLen;
    }

    public final String toString() {
        return "Key: "+key+" Buffer: "+buffer+" New: "+newData;
    }

    synchronized final KeyOutputStream getKeyOutputStream() throws IOException {
        newData = true;
        return new KeyOutputStreamImpl();
    }

    synchronized final KeyInputStream getKeyInputStream() throws IOException {
        return new KeyInputStreamImpl();
    }
            
    synchronized final void setFailureCode(int c) {
        failureCode = c;
        notifyAll();
    }

    synchronized final int getFailureCode() {
        if (failureCode == -1 && newData) {
            try {
                wait();
            }
            catch (InterruptedException e) {}
        }
        return failureCode;
    }

    synchronized final void release() {
        if (--users == 0)
            buffer.release();
    }

    
    private final class KeyOutputStreamImpl extends KeyOutputStream {

        final OutputStream out;

        long bytesWritten = 0;
        
        boolean closed = false;

        boolean commit = false,
                rollback = false;
        
        
        KeyOutputStreamImpl() throws IOException {
            out = buffer.getOutputStream();
            ++users;
        }

        public final void write(int b) throws IOException {
            out.write(b);
            ++bytesWritten;
        }

        public final void write(byte[] buf, int off, int len) throws IOException {
            out.write(buf, off, len);
            bytesWritten += len;
        }

        public final void flush() throws IOException {
            out.flush();
        }

        public void close() throws IOException {
            if (!closed) {
                try {
                    if (bytesWritten != buffer.length() || bytesWritten != physLen)
                        rollback = true;
                    out.close();
                    closed = true;
                }
                finally {
                    if (!closed || rollback) {
                        closed = true;
                        rollback = true;
                        release();
                    }
                }
            }
        }
        
        public final KeyInputStream getKeyInputStream() throws IOException {
            return FSDataStoreElement.this.getKeyInputStream();
        }

        public final void commit() throws IOException, KeyCollisionException {
            if (!closed) {
                throw new IllegalStateException("cannot commit before closing: "+this);
            }
            if (!rollback && !commit) {
                synchronized (FSDataStoreElement.this) {
                    try {
                        Core.logger.log(this, "committing key: "+key,
                                        Core.logger.DEBUG);
                        
                        synchronized (ds.dir.semaphore()) {
                            if (ds.dir.contains(new FileNumber(key.getVal()))) {
                                commit = true;
                                setFailureCode(Presentation.CB_CANCELLED);
                                Core.logger.log(this, "collision: "+key,
                                                Core.logger.DEBUG);
                                throw new KeyCollisionException();
                            }
                            buffer.commit();
                            ds.dir.forceFlush();
                        }
                        commit = true;
                    }
                    finally {
                        release();
                        if (!commit) {
                            setFailureCode(Presentation.CB_CACHE_FAILED);
                            Core.logger.log(this,
                                            "failed to store key: "+key,
                                            Core.logger.ERROR);
                        }
                    }
                }
            }
        }
        
        public final void rollback() {
            if (commit) {
                throw new IllegalStateException("already committed: "+this);
            }
            if (!rollback && closed) {
                release();
            }
            rollback = true;
        }

        public final void fail(int code) {
            rollback();
            setFailureCode(code);
        }

        protected final void finalize() throws Throwable {
            if (!closed) {
                Core.logger.log(this,
                                "Please close() me manually: "+this,
                                Core.logger.ERROR);
                if (!commit && !rollback)
                    rollback = true;
                close();
            }
        }

        public final String toString() {
            return FSDataStoreElement.this.toString();
        }
    }
    
    
    private final class KeyInputStreamImpl extends KeyInputStream {

        final InputStream in;

        final long length;

        final Storables storables = new Storables();
        
        private boolean closed = false;

        
        KeyInputStreamImpl() throws IOException {
            in = buffer.getInputStream();
            CountedInputStream cin = new CountedInputStream(in);
            storables.parseFields(new ReadInputStream(cin));
            length = buffer.length() - cin.count();
            ++users;
        }

        public final int read() throws IOException {
            return in.read();
        }

        public final int read(byte[] buf, int off, int len) throws IOException {
            return in.read(buf, off, len);
        }

        public final int available() throws IOException {
            return in.available();
        }

        public final void close() throws IOException {
            if (!closed) {
                closed = true;
                try {
                    in.close();
                }
                finally {
                    release();
                }
            }
        }

        public final long skip(long n) throws IOException {
            return in.skip(n);
        }
        
        public final long length() {
            return length;
        }

        public final Storables getStorables() {
            return storables;
        }

        public final int getFailureCode() {
            return newData ? FSDataStoreElement.this.getFailureCode()
                           : Presentation.CB_CACHE_FAILED;
        }

        protected final void finalize() throws Throwable {
            if (!closed) {
                Core.logger.log(this,
                                "Please close() me manually: "+this,
                                Core.logger.ERROR);
                close();
            }
        }

        public final String toString() {
            return FSDataStoreElement.this.toString();
        }
    }
}


