package freenet.node.http;

import freenet.node.Node;
import freenet.node.http.infolets.*;

import java.util.Vector;
import java.util.Enumeration;
import java.util.Hashtable;
import javax.servlet.ServletException;
import javax.servlet.http.*;
import java.io.PrintWriter;
import java.io.IOException;

/**
 * A servlet which aggregates "Infolets", pieces of code which represent
 * some aspect of node status in HTML.  It creates a nice menu from which
 * a user may conveniently find the infolet they are looking for.
 */
public class NodeInfoServlet extends HttpServlet  {

    private Node node;
    private Vector groups = new Vector();
    private Hashtable lookupTable = new Hashtable();
    private Infolet defaultInfolet = new DefaultInfolet();

    public final void init() { 

        node = (Node) getServletContext().getAttribute("freenet.node.Node");
        node.logger.log(this, "Initializing Infolets", node.logger.DEBUG);

        defaultInfolet.init(node);

        /* Construct menu
           ==============
           Create the group, register it, and then register the
           Infolets which are to be part of that group in the order you
           want to see them in the menu.  The long group name should be
           the name which will appear in the menu, the short group name
           will be the name that will appear in the URL (so it should
           basically be nothing but lower-case alphanumeric characters).
           The same is true for the long and short names of the Infolets
           themselves.

           For example:
           Group testGroup = new Group("Long Group Name", "briefgroupname");
           registerGroup(testGroup);
           registerInfolet(testGroup, new TestInfolet());
           registerInfolet(testGroup, new TestInfolet2());
           ...
           Group testGroup2 = new Group("Second Long Group Name", "briefgroupname2");
           ...
        */

        Group general = new Group("General","general");
        registerGroup(general);
        registerInfolet(general, new LoggerInfolet());
        registerInfolet(general, new TickerContents());
        registerInfolet(general, new OpenConnections());
        registerInfolet(general, new LoadStatsInfolet());
        registerInfolet(general, new FailureTableInfolet());
    }

    public void doGet(HttpServletRequest req, HttpServletResponse resp)
        throws ServletException, IOException  {
        String path=req.getPathInfo();
        PrintWriter pw = resp.getWriter();

        resp.setContentType("text/html");

        // Obtain Infolet
        Infolet i = null;
        if (path != null)  {
            try {
                i=(Infolet) lookupTable.get(path);
            }
            catch (ClassCastException e)  { }
        }
        if (i == null)  {
            i = defaultInfolet;
        }
    
        pw.println("<html><head><title>Node Information: " + i.longName() + 
                   "</title>");
        pw.println("<style type=\"text/css\">");
        pw.println("<!--");
        pw.println("BODY {color: black; font-family:Arial,sans-serif}");
        pw.println("-->");
        pw.println("</style>");
        pw.println("</head>");
        pw.println("<body>");
        pw.println("<h3>Node Information: "+i.longName()+"</h3>");
        pw.println("<table width=\"100%\" cellpadding=\"4\"><tr>" + 
                   "<td bgcolor=\"#999999\" nowrap=\"true\" valign=\"top\">");
        // Display menu
        String servletPath = req.getServletPath();
        if (!servletPath.endsWith("/"))
            servletPath = servletPath + "/";

        for(Enumeration e = groups.elements(); e.hasMoreElements();)  {
            Group g = (Group) e.nextElement();
            g.toHtml(pw, servletPath);
        }
        pw.println("<br>&nbsp;-<a href=\"" + servletPath + "\">Home</a>");
        
        pw.println("</td><td width=\"100%\">");
        // Display Infolet
        i.toHtml(pw);
        // Closing stuff
        pw.println("</td></tr></table></body></html>");
    }

    public String getServletInfo() {
        return "Information on the nodes current internals.";
    }

    public String getServletName() {
        return "Node Information";
    }


    private void registerGroup(Group g) {
        groups.addElement(g);
        lookupTable.put(g.shortName, g);
    }

    private void registerInfolet(Group g, Infolet i)  {
        g.register(i);
        lookupTable.put(g.shortName+"/"+i.shortName(), i);
        i.init(node);
    }

    private static class Group  {
        public String longName;
        public String shortName;
        Vector infolets = new Vector();
        Hashtable lookupTable;
        
        public Group(String longName,
                     String shortName)  {
            this.longName=longName;
            this.shortName=shortName;
        }
        
        public final void register(Infolet l)  {
            infolets.addElement(l);
        }
        
        public final void toHtml(PrintWriter pw, String servletPath)  {
            pw.println("<B>" + longName+ "</B><BR>");
            for (Enumeration e = infolets.elements(); e.hasMoreElements();) {
                Infolet l = (Infolet) e.nextElement();
                pw.println("&nbsp;-<a href=\"" + servletPath + shortName+"/"+
                           l.shortName()+
                           "\">"+l.longName()+"</a><br>");
            }
        }
    }
}

