package freenet.node.rt;

import freenet.support.*;
import java.io.*;

/**
 * Class to encapsulate the data used to implement the tested CP
 * algorithm that was in place before Tavin ripped out the old
 * routing code.
 * <p>
 *
 **/
class CPAlgoData implements DataObject {
    // persisted.
    float contactProbability = 1.0f;

    // not persisted.
    long lastRetryMs = System.currentTimeMillis();
    long intervalTimeoutMs = 0;
    int failureIntervals = 0;
    int failuresThisInterval = 0;

    long lastAttenuationMs = 0;

    // Diagnostic info
    int successes = 0;
    int trials = 0;

    ////////////////////////////////////////////////////////////
    static CPAlgoData getProperty(RoutingMemory mem, String name) {
        CPAlgoData ret;
        try {
            ret = (CPAlgoData) mem.getProperty(name);
        }
        catch (DataObjectUnloadedException e) {
            ret = new CPAlgoData(e);
        }
        if (ret == null) {
            ret = new CPAlgoData();
            mem.setProperty(name, ret);
        }
        return ret;
    }

    private CPAlgoData() {}
    
    
    ////////////////////////////////////////////////////////////
    // DataObject implementation
    //
    // REDFLAG: double check 
    private CPAlgoData(DataObjectPending dop) {
        if (dop.getDataLength() > 0) {
            try {
                DataInputStream din = dop.getDataInputStream();
                contactProbability = din.readFloat();
            }
            catch (IOException e) {}
            //wtf? Tavin, If this is ok there should be a comment explaining why.
            //
            // GJ, there were a few fine points I kind of glossed over..  but
            // it's not unreasonable to do this, the object just ends up being
            // reset to initial values from the application layer's point of view.
            //
            // Besides, if we get IOExceptions here there is something very wrong
            // with the data store that will probably bring down the whole node
            // anyway.
        }
        dop.resolve(this);
    }

    public final int getDataLength() {
        // 1 float, 1 * 4    4
        return 4;
    }

    public final void writeTo(DataOutputStream out) throws IOException {
        out.writeFloat(contactProbability);
    }

    ////////////////////////////////////////////////////////////
    // Some thoughts...
    //
    // Contact Probabilities (CP values) are used by the routing
    // algorithm to factor how reliably noderefs answer
    // connection attempts into routing decisions.
    //
    // Assumption:
    // The reliability of the CP value is inversely proportional
    // to the time since the last attempt was made to contact
    // the node.
    //
    // Objectives:
    // 0) Need to periodically retry even very bad nodes to see if 
    //    they have come back on line.
    // 1) Making too many connections to bad nodes isn't good.
    // 2) Must deal with, both high (well integrated nodes) and
    //    low (transient nodes) throughput cases robustly.
    //
    // Note:
    // There are several places where the code temporarily 
    // stops using a given node ref:  a) when a connection fails
    // to a node ref, it isn't used for the next 15 seconds,
    // b) if there are too many successive failures connecting
    // to a ref routing to it is disabled until the 
    // current failure interval expires.
    //
    // This behavior is probably beneficial to the health of
    // the network, but that's not why I did it.  Making connections
    // that will probably fail anyway wastes resources and makes 
    // the node run slower. 
    //
    // Connection limits are inexact because of the time lag
    // between connection attempts and the CP update.  One
    // could achieve more exact enforcement at the cost of
    // exposing connection management to the routing code.
    // It doesn't look like that's necessary.  There is
    // already a hack in the OCM that keeps too many 
    // connections to the same ref from blocking at
    // the socket layer.
    //

    private final static int BACKOFFINTERVAL_MS = 60000;
    private final static int SUCCESSHORIZON_MS = 120 * 60000;
    private final static int MAXFAILUREINTERVALS = 6;

    final static int MAXFAILURESPERINTERVAL = 3;
    final static int SHORTBACKOFFMS = 15000;
    
    private final static long timeHorizon(int intervals) {
        if (intervals == 0) {
            return SUCCESSHORIZON_MS;
        }
        
        return BACKOFFINTERVAL_MS * (1  << intervals);
    }

    // Weight the CP based on the time since the last connection
    // attempt and the number of failures.  
    final float effectiveCP(long nowMs) {
        if ((failuresThisInterval > 0) && (nowMs > intervalTimeoutMs)) {
            failuresThisInterval = 0;
        }                

        final long elapsedMs = nowMs - lastRetryMs;
        final long horizonMs = timeHorizon(failureIntervals);
        
        if (elapsedMs >  horizonMs) {
            // Completly ignore the CP once we get past the
            // time horizon.
            return (float)1.0;
        }

        final float weight = (float) elapsedMs / (float) horizonMs;
        
        return  (float)((1.0 - weight) * contactProbability) +
            (float)(weight / (float) (failuresThisInterval + 1));
    }

    float contactProbability() {
        long now = System.currentTimeMillis();

        // Drop nodes which have failed in the last 15 seconds from
        // routing.
        if ((failureIntervals > 0) && (now -  lastRetryMs < SHORTBACKOFFMS)) {
            return 0.0f;
        }
        
        // Must reset when interval expires.
        if ((failuresThisInterval > 0) && (now > intervalTimeoutMs)) {
            failuresThisInterval = 0;
        }                
        
        if (failuresThisInterval > MAXFAILURESPERINTERVAL) {
            // Mask out node if it has failed too many times
            return 0.0f;
        }                
        
        // Time weighted contact probability.
        return effectiveCP(now);
    }

    // returns true if the node should be dereferenced.
    boolean decreaseContactProbability() {
        long now = System.currentTimeMillis();
        lastRetryMs = now;
        contactProbability = (float)(((9.0 * contactProbability) /* + 0.0 */) / 10.0);
        
        failuresThisInterval++;
        
        if ((now > intervalTimeoutMs)) {
            // Time check above keeps ref from being punished
            // multiple times in the same interval.
            failureIntervals++;
            failuresThisInterval = 0;
            intervalTimeoutMs = now + timeHorizon(failureIntervals);
            //  Core.log(this, "reset time horizon to "+ 
            //                       (timeHorizon(failureIntervals) / 60000) +
            //         " minutes,  node: " + noderef, Logger.MINOR);
        }
        
        //Core.log(this, "decreasing CP to " + contactProbability
        //         +" for node: " + noderef, Logger.MINOR);
    
        // REDFLAG
        if (failureIntervals > MAXFAILUREINTERVALS) {
            return true;
        }
        return false;
    }

    void increaseContactProbability() {
        contactProbability = (float)(((4.0 * contactProbability) + 1.0) / 5.0);
        intervalTimeoutMs = 0;
        lastRetryMs = System.currentTimeMillis();
        failureIntervals = 0;
        failuresThisInterval = 0;
        
        //Core.log(this, "increasing CP to "+ contactProbability
        //         +" for node: "+ noderef, Logger.MINOR);
    }

    void backoffRouting() {
        lastAttenuationMs = System.currentTimeMillis();
    }

    boolean routingBackedOff() {
        return (lastAttenuationMs > 0) &&  
            (System.currentTimeMillis() - lastAttenuationMs < BACKOFFINTERVAL_MS); 
    }
    
}







