package freenet.node.states.FNP;

import freenet.*;
import freenet.node.*;
import freenet.message.*;
import freenet.node.states.request.FeedbackToken;
import java.io.OutputStream;
import java.net.UnknownHostException;

/** Relays results from the request states back to a remote node.
  * @author tavin
  */
final class FNPFeedbackToken implements FeedbackToken {

    // id of the associated state chain
    private final long id;
    // the node who asked us for this
    private final Peer origPeer;

    FNPFeedbackToken(long id, Peer origPeer) {
        this.id      = id;
        this.origPeer = origPeer;
    }

    public final void queryRejected(Node n, int htl, String reason, FieldSet fs,
                                    int unreachable, int restarted, int rejected)
                                                throws CommunicationException {
        n.sendMessage(new QueryRejected(id, htl, reason, fs), origPeer);
    }

    public final void restarted(Node n, long millis) throws CommunicationException {
        n.sendMessage(new QueryRestarted(id), origPeer);
    }
    
    public final void dataNotFound(Node n, long timeOfQuery) throws CommunicationException {
        n.sendMessage(new DataNotFound(id, timeOfQuery), origPeer);
    }
    
    public final OutputStream dataFound(Node n, Storables storables, long ctLength)
                                                    throws CommunicationException {
        try {
            FieldSet fs = new FieldSet();
            storables.addTo(fs);
            return n.sendMessage(new DataReply(id, fs, ctLength), origPeer);
        }
        catch (UnknownHostException e) {
            // this should be impossible
            n.logger.log(this, "This should be impossible", e, 
                         n.logger.ERROR);
            throw new SendFailedException(origPeer.getAddress(), e.getMessage());
        }
    }

    public final void insertReply(Node n, long millis) throws CommunicationException {
        n.sendMessage(new InsertReply(id), origPeer);
    }

    public final void storeData(Node n, NodeReference nr, long rate) throws CommunicationException {
        if (nr != null) {
            // experimental way of helping to keep buggy nodes out of the network
            String vers = nr.getVersion();
            if (vers != null && !Version.checkGoodVersion(vers)) {
                n.logger.log(this,
                    "Automatically resetting DataSource due to old version: "+vers,
                    n.logger.DEBUG);
                nr = null;
            }
        }
        // LATER... 
        // Use n.loadStats.resetProbability() to determine the reset
        // probability.
        if (!n.isTransient() && (nr == null || n.randSource.nextInt() % 20 == 0)) {
            nr = n.getNodeReference();
            // Send our request rate estimate too.
            rate = (long)n.loadStats.localRequestsPerHour();
        }
        n.sendMessage(new StoreData(id, nr, rate), origPeer);
    }
}



