package freenet.node.states.FNP;

import freenet.*;
import freenet.node.*;
import freenet.node.states.request.RequestAbortException;
import freenet.node.states.request.RequestDone;
import freenet.message.*;
import freenet.transport.VoidAddress;
import freenet.support.Logger;
import freenet.support.Fields;

public abstract class NewRequest extends State {

    protected Peer origRec;
    
    // Chance to decrease HTL p:
    //      p = 1 - e^(k*HTL^2)     where k == HTL_FACTOR
    //
    // a value of -1.5 gives:
    //          p
    //        ---
    // HTL 1  .78
    // HTL 2  .99
    //
    public static final double HTL_FACTOR = -1.5;
    
    protected NewRequest(long id) {
        super(id);
    }
    
    protected void genReceived(Node n, Request mo) 
                                throws RequestAbortException {

        origRec = n.getPeer(mo.getSource());        
        if (origRec == null) {
            n.logger.log(this,
                "No shared session and presentation with: "+mo.getSource(),
                Logger.MINOR);
            throw new RequestAbortException(null);
        }
            
        try {
            // enforce version aloofness
            String vers = mo.getSource().getVersion();
            if (vers != null && !Version.checkGoodVersion(vers)) {
                String reason = Version.explainBadVersion(vers);
                n.logger.log(this,
                    "Rejecting query from host of type "+vers+": "+reason,
                    Logger.MINOR);
                n.sendMessage(new QueryRejected(id, mo.hopsToLive,
                                                reason, mo.otherFields), origRec);
                throw new RequestAbortException(null);
            }

            // Any request that gets past the node version check.
            n.diagnostics.occurrenceCounting("inboundAggregateRequests", 1);
            n.loadStats.incRequestCount();

                        // decrement HTL 
            int htl = (int) mo.hopsToLive;
            if (htl < 1)
                htl = 1;
            else if (htl > Node.maxHopsToLive)
                htl = Node.maxHopsToLive;

            if (Math.exp(HTL_FACTOR*htl*htl) < 1 - n.randSource.nextDouble()){
                //n.logger.log(this, "Decrementing HTL",
                //             n.logger.DEBUG);
                --htl;
            } else {
                //n.logger.log(this, "Not decrementing HTL",
                //             n.logger.DEBUG);
            }

            // Enforce request rate limiting
            if (!n.acceptRequest(mo)) {
                String reason = "Node overloaded";
                n.logger.log(this,
                    "Rejecting query, rate limit exceeded.",
                    Logger.DEBUGGING);
                n.sendMessage(new QueryRejected(id, htl, 1, reason, 
                                                //       ^--- attenuate routing.
                                                mo.otherFields), origRec);
                throw new RequestAbortException(null);
            }

            mo.hopsToLive = htl; // <- actually, I think Tavin recreates
                                 //    the messages now anyways.

            // Any request that gets past the rate limiting.
            n.diagnostics.occurrenceCounting("inboundAggregateRequestsHandled",
                                             1);
            
            // reply with Accepted A.S.A.P.
            n.sendMessage(new Accepted(id), origRec);
            
        } catch (CommunicationException e) {
            n.logger.log(this, "Failed to send initial response to Request.", 
                         Logger.NORMAL);
            throw new RequestAbortException(null);
        }
    }

   /**
     * Requests in state new do nothing when lost.
     */
    public final void lost(Node n) {}
}



