package freenet.node.states.request;

import freenet.*;
import freenet.node.*;
import freenet.node.states.data.*;
import freenet.message.*;
import freenet.support.*;
import java.io.*;

/**
 * This is the State pertaining to Data Requests sending data from the
 * node.
 */

public class SendingReply extends RequestState {

    /** the auxiliary chain that is sending the data */
    SendData sendingData;

    SendingReply(Pending ancestor) {
        super(ancestor);
        sendingData = ancestor.sendingData;
    }

    /**
     * Returns the name.
     * @return "Sending data"
     */
    public final String getName() {
        return "Sending DataReply";
    }

    public State receivedMessage(Node n, DataSent ds) throws BadStateException {
        if (sendingData != ds.source()) {
            throw new BadStateException("Not my DataSent: "+ds);
        }
        int cb = ds.getCB();
        switch (cb) {
            case Presentation.CB_OK:
                n.logger.log(this, "Data sent successfully!", Logger.MINOR);
                try {
                    // return the noderef we would have routed to
                    // (there is still a chance of DataSource resetting)
                    // -1 means we don't know the rate for that ref.
                    ft.storeData(n, n.rt.route(searchKey).getNextRoute(), -1);
                }
                catch (CommunicationException e) {
                    n.logger.log(this,
                        "Failed to send back StoreData to peer " + e.peer,
                        e, Logger.MINOR);
                }
                return new RequestDone(this);
            
            case Presentation.CB_CACHE_FAILED:
                fail(n, "Cache failed");
                // fall through
            case Presentation.CB_SEND_CONN_DIED:
                n.logger.log(this,
                    "Dropping send on CB "+Presentation.getCBdescription(cb)
                    +", on chain "+Long.toHexString(id),
                    (cb == Presentation.CB_CACHE_FAILED ? Logger.ERROR : Logger.MINOR));
                return new RequestDone(this);
                
            // the StoreInputStream we were reading from got restarted
            default:
                scheduleRestart(n, 0);
                return new DataPending(this);
        }
    }

    /**
     * This is only needed for InsertRequests on key collisions, so really
     * it shouldn't be here - but the safety added is minimal so I'll
     * fix it another day. It just eats the data.
     */
    public State receivedMessage(Node n, DataInsert dim) throws BadStateException {
        if (!fromOrigPeer(dim)) {
            throw new BadStateException("DataInsert from the wrong peer!");
        }
        n.logger.log(this, "Eating DataInsert during SendingReply", Logger.DEBUG);
        dim.eatData(n);
        return this;
    }

    // There's not much we can do if the state is lost while sending data.
    // The MessageHandler will log it. 
    public final void lost(Node n) {
        Core.diagnostics.occurrenceCounting("lostRequestState", 1);
    }
}



