package freenet.support;
import java.io.*;
import java.util.StringTokenizer;
import java.util.Vector;
import java.util.Date;
import java.text.DateFormat;
import java.text.SimpleDateFormat;

/**
 * Converted the old StandardLogger to Ian's loggerhook interface.
 *
 * @author oskar
 */
public class FileLoggerHook implements LoggerHook {

    /** Verbosity types **/
    public static final int DATE       = 1,
                            CLASS      = 2,
                            HASHCODE   = 3,
                            THREAD     = 4,
                            PRIORITY   = 5,
                            MESSAGE    = 6,
                            UNAME      = 7;
 
    private static String uname;
    static {
        try {
            uname =
                new StringTokenizer(java.net.InetAddress.getLocalHost().getHostName(),
                                    ".").nextToken();
        } catch (java.net.UnknownHostException e) {
            uname = "unknown";
        }
    }

    private DateFormat df;
    private int threshold;
    private int[] fmt;
    private String[] str;
    
    
    /**
     * The printstream the Logger writes too.
     **/
    public PrintStream lout;


    /**
     * Create a Logger to append to the given file. If the file does not
     * exist it will be created.
     *
     * @param filename  the name of the file to log to.
     * @param verb      The verbosity setting.
     * @param fmt       log message format string
     * @param dfmt      date format string
     * @param threshhold  Lowest logged priority
     * @exception IOException if the file couldn't be opened for append.
     */
    public FileLoggerHook(String filename, String fmt, 
                          String dfmt, int threshold) throws IOException {
        this(new PrintStream(new FileOutputStream(filename, true)),
             fmt, dfmt, threshold);
    }

    /**
     * Create a Logger to send log output to the given PrintStream.
     *
     * @param stream  the PrintStream to send log output to.
     * @param fmt     log message format string
     * @param dfmt    date format string
     * @param threshhold  Lowest logged priority
     */
    public FileLoggerHook(PrintStream stream, String fmt, String dfmt,
                          int threshold) {
        lout = stream;
        this.threshold = threshold;
        if (dfmt != null && !dfmt.equals("")) {
            try {
                df = new SimpleDateFormat(dfmt);
            }
            catch (RuntimeException e) {
                df = DateFormat.getDateTimeInstance();
            }
        }
        else df = DateFormat.getDateTimeInstance();
        
        if (fmt == null || fmt.equals("")) fmt = "d:c:h:t:p:m";
        char[] f = fmt.toCharArray();
        
        Vector fmtVec = new Vector(),
               strVec = new Vector();
               
        StringBuffer sb = new StringBuffer();
        int sctr = 0;
        
        boolean comment = false;
        for (int i=0; i<f.length; ++i) {
            if (!comment && numberOf(f[i]) != 0) {
                if (sb.length() > 0) {
                    strVec.addElement(sb.toString());
                    fmtVec.addElement(new Integer(0));
                    sb.setLength(0);
                }
                fmtVec.addElement(new Integer(numberOf(f[i])));
            } else if (f[i] == '\\') {
                comment = true;
            } else {
                comment = false;
                sb.append(f[i]);
            }
        }
        if (sb.length() > 0) {
            strVec.addElement(sb.toString());
            fmtVec.addElement(new Integer(0));
            sb.setLength(0);
        }
        
        this.fmt = new int[fmtVec.size()];
        int size = fmtVec.size();
        for (int i=0; i < size; ++i)
            this.fmt[i] = ((Integer) fmtVec.elementAt(i)).intValue();

        this.str = new String[strVec.size()];
        strVec.copyInto(this.str);
    }

    public void log(Object o, Class c, String msg, Throwable e, int priority){
        if (priority < threshold) return;

        StringBuffer sb = new StringBuffer(512);
        int sctr = 0;
        
        for (int i=0; i<fmt.length; ++i) {
            switch (fmt[i]) {
            case 0:         sb.append(str[sctr++]); break;
            case DATE:      sb.append(df.format(new Date())); break;
            case CLASS:     sb.append(c == null ? "<none>" : c.getName());
                break;
            case HASHCODE:  sb.append(o == null ? "<none>"
                                      : Integer.toHexString(o.hashCode()));
            break;
            case THREAD:    sb.append(Thread.currentThread().getName()); break;
            case PRIORITY:  sb.append(Logger.priorityOf(priority)); break;
            case MESSAGE:   sb.append(msg); break;
            case UNAME:         sb.append(uname); break;
            }
        }
        
        lout.println(sb.toString());
        if (e != null && (priority >= ERROR || 
                          threshold == DEBUG)) 
            e.printStackTrace(lout);
        lout.flush();
    }
    
    public static int numberOf(char c) {
        switch (c) {
        case 'd':   return DATE;
        case 'c':   return CLASS;
        case 'h':   return HASHCODE;
        case 't':   return THREAD;
        case 'p':   return PRIORITY;
        case 'm':   return MESSAGE;
        case 'u':   return UNAME;    
        default:    return 0;
        }
    }

}
