package freenet.support;
import freenet.Core;
import java.io.PrintStream;
import java.util.Vector;
import java.util.Enumeration;
/**
 * A class that takes logging messages and distributes them to LoggerHooks.
 * It implements LoggerHook itself, so that instances can be chained (just
 * don't create loops).
 */
public class Logger implements LoggerHook {
  
    private int threshold;
    private Vector hooks;

    /**
     * Create a logger. Threshhold set to NORMAL.
     */
    public Logger() {
        this(NORMAL);
    }

    /**
     * Create a logger.
     * @param threshold   Suppress all log calls with lower priority then 
     *                     this.
     */
    public Logger(int threshold) {
        this.threshold = threshold;
        hooks = new Vector();
    }


    /**
     * Log a message.
     * @param source        The source object where this message was generated
     * @param message A clear and verbose message describing the event
     * @param priority The priority of the mesage, one of Logger.ERROR,
     *                 Logger.NORMAL, Logger.MINOR, or Logger.DEBUGGING.
     **/
    public void log(Object source, String message, int priority) {
        if (priority < threshold) return;
        log(source, source == null ? null : source.getClass(), 
            message, null, priority);
    }

    /** 
     * Log a message with an exception.
     * @param o   The source object where this message was generated.
     * @param message  A clear and verbose message describing the event.
     * @param e        Logs this exception with the message.
     * @param priority The priority of the mesage, one of Logger.ERROR,
     *                 Logger.NORMAL, Logger.MINOR, or Logger.DEBUGGING.
     * @see log(Object o, String message, int priority)
     */
    public void log(Object o, String message, Throwable e, 
                    int priority) {
        if (priority < threshold) return;
        log(o, o == null ? null : o.getClass(), message, e, priority);
    }

    /**
     * Log a message from static code.
     * @param c        The class where this message was generated.
     * @param message  A clear and verbose message describing the event
     * @param priority The priority of the mesage, one of Logger.ERROR,
     *                 Logger.NORMAL, Logger.MINOR, or Logger.DEBUGGING.
     */
    public void log(Class c, String message, int priority) {
        if (priority < threshold) return;
        log(null, c, message, null, priority);
    }

    /**
     * Log a message from static code.
     * @param c     The class where this message was generated.
     * @param message A clear and verbose message describing the event
     * @param e        Logs this exception with the message.
     * @param priority The priority of the mesage, one of Logger.ERROR,
     *                 Logger.NORMAL, Logger.MINOR, or Logger.DEBUGGING.
     */
    public void log(Class c, String message, Throwable e,
                    int priority) {
        if (priority < threshold) return;
        log(null, c, message, e, priority);
    }

    /**
     * This is the implementation of LoggerHook method, which allows
     * one logger receive events from another.
     * @implements LoggerHook.log()
     */
    public void log(Object o, Class c, String msg, Throwable e, int priority){
        for(Enumeration en = hooks.elements(); en.hasMoreElements();) {
            ((LoggerHook) en.nextElement()).log(o,c,msg,e,priority);
        }
    }

    /**
     * Add a hook which will be called every time a message is logged
     */
    public void addHook(LoggerHook lh) {
        hooks.addElement(lh);
    }

    /**
     * Remove a hook from the logger.
     */
    public void removeHook(LoggerHook lh) {
        hooks.removeElement(lh);
    }

    /**
     * Returns all the current hooks.
     */
    public LoggerHook[] getHooks() {
        LoggerHook[] r = new LoggerHook[hooks.size()];
        hooks.copyInto(r);
        return r;
    }

    /**
     * Changes the priority threshold.
     * @param thresh   The new threshhold
     */
    public void setThreshold(int thresh) {
        this.threshold = thresh;
    }

    /**
     * Returns whether this priority will be logged.
     * @param priority   The priority
     */
    public boolean shouldLog(int priority) {
        return priority >= threshold;
    }

    /**
     * Returns the priority level matching the string. If no priority
     * matches, Logger.NORMAL is returned.
     * @param s  A string matching one of the logging priorities, case
     *           insensitive.
     **/
    public static int priorityOf(String s) {
        if (s.equalsIgnoreCase("error"))
            return Logger.ERROR;
        else if (s.equalsIgnoreCase("normal"))
            return Logger.NORMAL;
        else if (s.equalsIgnoreCase("minor"))
            return Logger.MINOR;
        else if (s.equalsIgnoreCase("debugging"))
            return Logger.DEBUGGING;
        else if (s.equalsIgnoreCase("debug"))
            return Logger.DEBUGGING;
        else 
            return Logger.NORMAL;
    }

    /**
     * Returns the name of the priority matching a number, null if none do
     * @param priority  The priority
     */
    public static String priorityOf(int priority) {
        switch (priority) {
            case ERROR:     return "ERROR";
            case NORMAL:    return "NORMAL";
            case MINOR:     return "MINOR";
            case DEBUG:     return "DEBUG";
            default:        return null;
        }
    }

}










