package freenet.transport;

import freenet.*;
import freenet.support.io.ThrottledOutputStream;
import freenet.support.Logger;
import java.io.*;
import java.net.Socket;
import java.net.InetAddress;

final class tcpConnection extends Connection {

    private Socket sock;
    private InputStream in;
    private OutputStream out;

    private final tcpTransport t;

    private tcpConnection(tcpTransport t) {
        super(t);
        this.t = t;
    }
    
    /**
     * Used to create an outbound connection.
     */
    tcpConnection(tcpTransport t, tcpAddress addr) throws ConnectFailedException {
        this(t);
        try {
            long time = System.currentTimeMillis();
            sock = t.getSocketFactory().createSocket(addr.getHost(), addr.getPort());
            Core.diagnostics.occurrenceContinuous("socketTime",
                                                 System.currentTimeMillis() - time);
            in = sock.getInputStream();        
            if (sock.getLocalAddress().equals(sock.getInetAddress())) {
                Core.logger.log(this,
                                "not throttling local connection", Logger.DEBUG);
                out = sock.getOutputStream();
            } else {
               out = ThrottledOutputStream.throttledStream(sock.getOutputStream());
            }
        } catch (IOException e) {
            throw new ConnectFailedException(addr);
        }
    }

    /**
     * Used to accept an incoming connection.
     */
    tcpConnection(tcpTransport t, Socket sock, int designator) throws IOException {
        this(t);

        this.sock = sock;
        //      sock.setSoLinger(true, 0);
        if (designator < 0) {
            in=sock.getInputStream();
        } else {
            byte[] b = {(byte) ((designator >> 8) & 0xff) , 
                        (byte) (designator & 0xff) };
            in = new SequenceInputStream(new ByteArrayInputStream(b),
                                         sock.getInputStream());
        }
        if (sock.getLocalAddress().equals(sock.getInetAddress())) {
            Core.logger.log(this,
                            "not throttling local connection", Logger.DEBUG);
          out = sock.getOutputStream();
        } else {
            out = 
                ThrottledOutputStream.throttledStream(sock.getOutputStream());
        }
    }

    public final void close() {
        try { 
            in.close();
        } catch(Exception e) {}
        in=null;
        
        try {
            out.close();
        } catch(Exception e) {}
        out=null;

        try {
            sock.close(); 
        } catch(IOException e) {
            // It may have been closed remotely in which case
            // sock.close will throw an exception.  We really don't
            // care though.
        }
    }

    public final InputStream getIn() {
        return in;
    }

    public final OutputStream getOut() {
        return out;
    }


    public final void setSoTimeout(int timeout) throws IOException {
        if (sock != null)
            sock.setSoTimeout(timeout);
        else
            throw new IOException();
    }
    
    public final int getSoTimeout() throws IOException {
        return sock.getSoTimeout();
    }

    public final Address getMyAddress(ListeningAddress lstaddr) {
        if (sock == null) return null;
        try {
            return new tcpAddress(t, sock.getLocalAddress(),
                                  ((tcpListeningAddress) lstaddr).getPort());
        }
        catch (BadAddressException e) {  // shouldn't really be possible
            return null;
        }
    }

    public final Address getMyAddress() {
        if (sock == null) return null;
        try {
            return new tcpAddress(t, sock.getLocalAddress(), sock.getLocalPort());
        }
        catch (BadAddressException e) {  // shouldn't really be possible
            return null;
        }
    }

    public final Address getPeerAddress(ListeningAddress lstaddr) {
        if (sock == null) return null;
        try {
            return new tcpAddress(t, sock.getInetAddress(),
                                  ((tcpListeningAddress) lstaddr).getPort());
        }
        catch (BadAddressException e) {  // shouldn't really be possible
            return null;
        }
    }

    public final Address getPeerAddress() {
        if (sock == null) return null;
        try {
            return new tcpAddress(t, sock.getInetAddress(), sock.getPort());
        }
        catch (BadAddressException e) {  // shouldn't really be possible
            return null;
        }
    }

    public final String toString() {
        return getTransport().getName()+" connection: " + sock;
    }
}


