
/*
    FUNIONFS: UNIONFS over FUSE Usermode filesystem
    Copyright (C) 2005-2006  Stephane APIOU <stephane.apiou@free.fr>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <fuse.h>
#include <ctype.h>
#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <dirent.h>
#include <sys/mman.h>
#include <utime.h>
#include <signal.h>

#include "funionfs.h"

#if FUSE_VERSION < 25
#include "compat/fuse_opt.h"
#endif

#ifdef HAVE_SETXATTR
#include <sys/xattr.h>
#endif

/*!
* \brief main file of the funionfs program.
* This file contains all the callbacks to implement funionfs into fuse
* This file contains also the command line parser for funionfs
*
* Last release: $Id$
*
* \author Stephane APIOU
* \date   mer jui 12 00:35:26 CEST 2006
* \todo  Do the TODO file
*
* A faire: pour les liens symboliques, la creation de lien doit tenir compte des chemins relatifs
* de la version en lecture seule /  l'arborescence monte pour des liens pointant en dehors du
* systme unionfs et elle doit tenir compte des pointages des liens des arborescences en lecture seule
* et en lecture/ecriture. 
* Cette fonctionnalit doit tre optionnelle
*/

f_opt_t f_opt = { "", "", "", 0, 0, 0, "", "", "", DELETED_STRING, 0, "" };

extern struct unionfs_desc *unionfs_list;	//!< list of filesystem to overlay

char initial_working_dir[PATH_MAX + 1];

FILE *log_fd;

/*!
* \brief
*
* \param[int] num
*
* \return nothing
*/

void
signal_handler(int num)
{
	DEBUG_PRINTF("Signal %d caught.\n", num);
	switch (num)
	{
	case SIGQUIT:
	case SIGTERM:
		DEBUG_PRINTF("Unmounting %s.\n", f_opt.mountpoint);
		if (*f_opt.mountpoint == '/')
			fuse_unmount(f_opt.mountpoint);
		else
			fuse_unmount(rel2abspath
				     (f_opt.mountpoint, initial_working_dir));
		exit(0);
		break;
	default:
		exit(32 + num);
	}
}

static int
funionfs_getattr(const char *path, struct stat *stbuf)
{
	int res = 0;
	struct unionfs_desc *pdesc_tab[3];

	char *getpath, *setpath, *newpath;
	struct stat statfile;

	res = funionfs_realpath(path, &getpath, &setpath, &newpath, pdesc_tab);
	if (res < 0)
		return res;

	DEBUG_PRINTF("GETATTR(%s,%s,%s)\n", getpath, setpath, newpath);

	if (res & FILE_EXIST)
	{
		res = lstat(getpath, &statfile);
	}
	else
	{
		res = -1;
		errno = ENOENT;
	}

	free(getpath);
	free(setpath);
	free(newpath);

	if (res < 0)
		return -errno;

	memcpy(stbuf, &statfile, sizeof(struct stat));

	return 0;
}

static int
funionfs_readlink(const char *path, char *buf, size_t size)
{
	char *getpath, *setpath, *newpath;
	int res;
	struct unionfs_desc *pdesc_tab[3];

	res = funionfs_realpath(path, &getpath, &setpath, &newpath, pdesc_tab);
	if (res < 0)
		return res;

	DEBUG_PRINTF("GETATTR(%s,%s,%s): %d \n", getpath, setpath, newpath,
		     size);

	if (res & FILE_EXIST)
		res = readlink((const char *) getpath, buf, size - 1);
	else
	{
		res = -1;
		errno = ENOENT;
	}

// ATTENTION, il faut corriger le chemin afin de prendre en compte les chemins relalifs de faon correcte !
// MAJ: A voir, peut etre que la reecriture doit tre optionnelle

	free(getpath);
	free(setpath);
	free(newpath);

	if (res < 0)
		return -errno;

	buf[res] = '\0';
	return 0;
}


static int
funionfs_mknod(const char *path, mode_t mode, dev_t dev)
{
	char *getpath, *setpath, *newpath;
	int res, status;
	int len;
	struct unionfs_desc *pdesc_tab[3];

	status = funionfs_realpath(path, &getpath, &setpath, &newpath,
				   pdesc_tab);
	if (status < 0)
		return status;

	DEBUG_PRINTF("MKNOD(%s,%s,%s): %8.8x %8.8x\n", getpath, setpath,
		     newpath, (unsigned int) mode, (unsigned int) dev);

	res = -1;

	do
	{
		if (status & FILE_EXIST)
		{
			errno = EEXIST;
			break;
		}
		if (!(status & FILE_CREATABLE))
		{
			errno = EROFS;
			break;
		}
		if (status & FILE_DELETED)
		{
			len = strlen(newpath);
			strcat(newpath, f_opt.del_string);
			res = unlink(newpath);
			if (res == -1)
			{
				break;
			}
			newpath[len] = '\0';
		}
		else
		{
			res = funionfs_pathtowrite(path);
			if (res < 0)
				break;
		}

		if (S_ISFIFO(mode))
			res = mkfifo(newpath, mode);
		else
			res = mknod(newpath, mode, dev);
		if ((res == 0) && (geteuid() == 0))
		{
			res = chown(newpath, fuse_get_context()->uid,
				    fuse_get_context()->gid);
		}

	}
	while (0);

	free(getpath);
	free(setpath);
	free(newpath);

	if (res == -1)
		return -errno;
	return 0;
}

static int
funionfs_mkdir(const char *path, mode_t mode)
{
	char *getpath, *setpath, *newpath;
	char *delpath;
	int res, status;
	int len;
	struct unionfs_desc *pdesc_tab[3];

	status = funionfs_realpath(path, &getpath, &setpath, &newpath,
				   pdesc_tab);
	if (status < 0)
		return status;

	DEBUG_PRINTF("MKDIR(%s,%s,%s): %8.8x \n", getpath, setpath, newpath,
		     mode);

	res = -1;

	do
	{
		if (!(status & FILE_CREATABLE))
		{
			errno = EROFS;
			break;
		}
		if (status & FILE_EXIST)
		{
			errno = EEXIST;
			break;
		}
		if (status & FILE_DELETED)
		{
			len = strlen(newpath);
			delpath = (char *)malloc( len + f_opt.del_len);
			strcpy(delpath, newpath);
			strcat(delpath, f_opt.del_string);
			res = rename(delpath, newpath);
			free(delpath);
			if ((!res) && (!geteuid()))
			{
				res = chown(newpath, fuse_get_context()->uid,
					    fuse_get_context()->gid);
			}
			if (!res) {
				res = chmod(setpath, mode);
			}
			if (res == -1)
			{
				break;
			}
		}
		else if (status & FILE_EXIST)
		{
			errno = EEXIST;
			break;
		}
		else
		{
			res = funionfs_pathtowrite(path);
			if (res < 0)
				res = mkdir(newpath, mode);
			if ((!res) && (!geteuid()))
			{
				res = chown(newpath, fuse_get_context()->uid,
					    fuse_get_context()->gid);
			}
			break;
		}
	}
	while (0);

	free(getpath);
	free(setpath);
	free(newpath);

	if (res < 0)
		return -errno;
	return 0;
}

static int
funionfs_unlink(const char *path)
{
	char *getpath, *setpath, *newpath;
	int status, res;
	struct unionfs_desc *pdesc_tab[3];

	status = funionfs_realpath(path, &getpath, &setpath, &newpath,
				   pdesc_tab);
	if (status < 0)
		return status;

	DEBUG_PRINTF("UNLINK(%s,%s,%s) \n", getpath, setpath, newpath);

	res = -1;
	do
	{

		if (!(status & FILE_CREATABLE))
		{
			errno = EROFS;
			break;
		}
		if ((!(status & FILE_EXIST)) || (status & FILE_DELETED))
		{
			errno = ENOENT;
			break;
		}
		if (status & FILE_WRITEABLE)
		{
			res = unlink(setpath);
			if (res < 0)
				break;
			if (status & FILE_TODELETE)
			{
				strcat(newpath, f_opt.del_string);
				res = funionfs_pathtowrite(path);
				if (res < 0)
					break;
				res = mknod(newpath, 0777, 0);
			}
		}
		else
		{
			strcat(newpath, f_opt.del_string);
			res = funionfs_pathtowrite(path);
			if (res < 0)
				break;
			res = mknod(newpath, 0777, 0);
		}
	}
	while (0);

	free(getpath);
	free(setpath);
	free(newpath);

	if (res < 0)
		return -errno;
	return 0;

}

static int
funionfs_rmdir(const char *path)
{
	char *getpath, *setpath, *newpath;
	int res, status;
	struct unionfs_desc *pdesc_tab[3];

	status = funionfs_realpath(path, &getpath, &setpath, &newpath,
				   pdesc_tab);
	if (status < 0)
		return status;

	DEBUG_PRINTF("RMDIR(%s,%s,%s) \n", getpath, setpath, newpath);

	res = -1;
	do
	{
		if (!(status & FILE_CREATABLE))
		{
			errno = EROFS;
			break;
		}
		if ((!(status & FILE_EXIST)) || (status & FILE_DELETED))
		{
			errno = ENOENT;
			break;
		}
		if (status & FILE_WRITEABLE)
		{
			if (status & FILE_TODELETE)
			{
				strcat(newpath, f_opt.del_string);
				res = rename(setpath, newpath);
			}
			else
			{
				res = rmdir(setpath);
				if (res < 0)
					break;
			}
		}
		else
		{
			strcat(newpath, f_opt.del_string);
			res = mknod(newpath, 0777, 0);
		}
	}
	while (0);

	free(getpath);
	free(setpath);
	free(newpath);

	if (res < 0)
		return -errno;
	return 0;
}

static int
funionfs_symlink(const char *from, const char *to)
{
	char *getpath, *setpath, *newpath;
	int res, status;
	struct unionfs_desc *pdesc_tab[3];

	status = funionfs_realpath(to, &getpath, &setpath, &newpath, pdesc_tab);
	if (status < 0)
		return status;

	DEBUG_PRINTF("SYMLINK(%s,%s,%s): %s \n", getpath, setpath, newpath,
		     from);

	res = -1;

	do
	{
		if (!(status & FILE_CREATABLE))
		{
			errno = EROFS;
			break;
		}
		if (status & FILE_DELETED)
		{
			strcat(newpath, f_opt.del_string);
			res = unlink(newpath);
		}
		else if (status & FILE_EXIST)
		{
			errno = EEXIST;
			break;
		}
		else
		{
			res = funionfs_pathtowrite(to);
			if (res < 0)
				break;
		}

		res = symlink(from, newpath);
		if (res == -1)
		{
			break;
		}
		if ((!res) && (!geteuid()))
		{
			res = lchown(newpath, fuse_get_context()->uid,
				     fuse_get_context()->gid);
		}

	}
	while (0);

	free(getpath);
	free(setpath);
	free(newpath);

	if (res == -1)
		return -errno;
	return 0;

}

static int
funionfs_rename(const char *from, const char *to)
{
	char *to_getpath, *to_setpath, *to_newpath;
	char *from_getpath, *from_setpath, *from_newpath;
	struct unionfs_desc *pdesc_tab_to[3];
	struct unionfs_desc *pdesc_tab_from[3];
	int res;
	int status1, status2;

	status1 =
		funionfs_realpath(to, &to_getpath, &to_setpath, &to_newpath,
				  pdesc_tab_to);
	if (status1 < 0)
		return status1;

	status2 =
		funionfs_realpath(from, &from_getpath, &from_setpath,
				  &from_newpath, pdesc_tab_from);
	if (status2 < 0)
	{
		free(to_getpath);
		free(to_setpath);
		free(to_newpath);
		return status2;
	}

	DEBUG_PRINTF("RENAME(%s,%s,%s): (%s,%s,%s) \n", to_getpath, to_setpath,
		     to_newpath, from_getpath, from_setpath, from_newpath);
	do
	{
		/* test if destination directory exists and create if not */
		/* destination file is overwritten if exist */
		if (!(status1 & FILE_CREATABLE))
		{
			errno = EROFS;
			break;
		}
		if (status1 & FILE_DELETED)
		{
			strcat(to_newpath, f_opt.del_string);
			res = unlink(to_newpath);
		}
		else
		{
			res = funionfs_pathtowrite(to);
			if (res < 0)
				break;
		}
		if (status2 & FILE_WRITEABLE)
		{
			res = rename(from_setpath, to_newpath);
			if (res < 0)
				break;
			if (status2 & FILE_TODELETE)
			{
				strcat(from_newpath, f_opt.del_string);
				res = mknod(from_newpath, 0777, 0);
			}
		}
		else
		{
			res = funionfs_copytowrite(from_getpath, from_newpath);
			if (res < 0)
				break;
			res = rename(from_newpath, to_newpath);
			if (res < 0)
				break;
			strcat(from_newpath, f_opt.del_string);
			res = mknod(from_newpath, 0777, 0);
		}

	}
	while (0);

	free(to_getpath);
	free(to_setpath);
	free(to_newpath);
	free(from_getpath);
	free(from_setpath);
	free(from_newpath);

	if (res == -1)
		return -errno;
	return 0;

}

static int
funionfs_link(const char *from, const char *to)
{
	char *to_getpath, *to_setpath, *to_newpath;
	char *from_getpath, *from_setpath, *from_newpath;
	int res, status1, status2;
	struct unionfs_desc *pdesc_tab_to[3];
	struct unionfs_desc *pdesc_tab_from[3];

	status1 =
		funionfs_realpath(to, &to_getpath, &to_setpath, &to_newpath,
				  pdesc_tab_to);
	if (status1 < 0)
		return status1;

	status2 =
		funionfs_realpath(from, &from_getpath, &from_setpath,
				  &from_newpath, pdesc_tab_from);
	if (status2 < 0)
	{
		free(to_getpath);
		free(to_setpath);
		free(to_newpath);
		return status2;
	}

	DEBUG_PRINTF("LINK(%s,%s,%s): (%s,%s,%s) \n", to_getpath, to_setpath,
		     to_newpath, from_getpath, from_setpath, from_newpath);

	res = -1;

	do
	{
		if (status1 & FILE_EXIST)
		{
			errno = EEXIST;
			break;
		}
		if (!(status1 & FILE_CREATABLE))
		{
			errno = EROFS;
			break;
		}
		if (status1 & FILE_DELETED)
		{
			strcat(to_newpath, f_opt.del_string);
			res = unlink(to_newpath);
		}
		else
		{
			res = funionfs_pathtowrite(to);
			if (res < 0)
				break;
		}

		res = link(from_getpath, to_newpath);
		if (res == -1)
		{
			res = funionfs_copytowrite(from_getpath, from_newpath);
			if (res < 0)
				break;
			res = link(from_newpath, to_newpath);
			if (res == -1)
				break;
		}

	}
	while (0);

	free(to_getpath);
	free(to_setpath);
	free(to_newpath);
	free(from_getpath);
	free(from_setpath);
	free(from_newpath);

	if (res == -1)
		return -errno;
	return 0;
}


static int
funionfs_chmod(const char *path, mode_t mode)
{
	int res;
	char *getpath, *setpath, *newpath;
	struct unionfs_desc *pdesc_tab[3];
	int status;

	status = funionfs_realpath(path, &getpath, &setpath, &newpath,
				   pdesc_tab);
	if (status < 0)
		return status;

	DEBUG_PRINTF("CHMOD(%s,%s,%s): %x \n", getpath, setpath, newpath, mode);

	res = -1;

	do
	{
		if (status & FILE_EXIST)
		{
			errno = EEXIST;
			break;
		}
		if (!(status & FILE_CREATABLE))
		{
			errno = EROFS;
			break;
		}
		if (status & FILE_WRITEABLE)
			res = chmod(setpath, mode);
		else
		{
			res = funionfs_copytowrite(getpath, newpath);
			if (res < 0)
				break;
			res = chmod(newpath, mode);
		}

	}
	while (0);

	free(getpath);
	free(setpath);
	free(newpath);

	if (res == -1)
		return -errno;

	return 0;
}

static int
funionfs_chown(const char *path, uid_t uid, gid_t gid)
{
	int res;
	char *getpath, *setpath, *newpath;
	struct unionfs_desc *pdesc_tab[3];
	int status;

	status = funionfs_realpath(path, &getpath, &setpath, &newpath,
				   pdesc_tab);
	if (status < 0)
		return status;

	DEBUG_PRINTF("CHOWN(%s,%s,%s): %d %d \n", getpath, setpath, newpath,
		     uid, gid);

	res = -1;

	do
	{
		if (status & FILE_EXIST)
		{
			errno = EEXIST;
			break;
		}
		if (!(status & FILE_CREATABLE))
		{
			errno = EROFS;
			break;
		}
		if (status & FILE_WRITEABLE)
			res = lchown(setpath, uid, gid);
		else
		{
			res = funionfs_copytowrite(getpath, newpath);
			if (res < 0)
				break;
			res = lchown(newpath, uid, gid);
		}

	}
	while (0);

	free(getpath);
	free(setpath);
	free(newpath);

	if (res == -1)
		return -errno;

	return 0;
}

static int
funionfs_truncate(const char *path, off_t size)
{
	int res;
	char *getpath, *setpath, *newpath;
	struct unionfs_desc *pdesc_tab[3];
	int status;

	status = funionfs_realpath(path, &getpath, &setpath, &newpath,
				   pdesc_tab);
	if (status < 0)
		return status;

	DEBUG_PRINTF("TRUNCATE(%s,%s,%s): %x \n", getpath, setpath, newpath,
		     (unsigned int) size);

	res = -1;

	do
	{
		if (status & FILE_EXIST)
		{
			errno = EEXIST;
			break;
		}
		if (!(status & FILE_CREATABLE))
		{
			errno = EROFS;
			break;
		}
		if (status & FILE_WRITEABLE)
			res = truncate(setpath, size);
		else
		{
			res = funionfs_copytowrite(getpath, newpath);
			if (res < 0)
				break;
			res = truncate(newpath, size);
		}

	}
	while (0);

	free(getpath);
	free(setpath);
	free(newpath);

	if (res == -1)
		return -errno;

	return 0;
}

static int
funionfs_utime(const char *path, struct utimbuf *buf)
{
	int res;
	char *getpath, *setpath, *newpath;
	struct unionfs_desc *pdesc_tab[3];
	int status;

	status = funionfs_realpath(path, &getpath, &setpath, &newpath,
				   pdesc_tab);
	if (status < 0)
		return status;

	DEBUG_PRINTF("UTIME(%s,%s,%s) \n", getpath, setpath, newpath);

	res = -1;

	do
	{
		if (status & FILE_EXIST)
		{
			errno = EEXIST;
			break;
		}
		if (!(status & FILE_CREATABLE))
		{
			errno = EROFS;
			break;
		}
		if (status & FILE_WRITEABLE)
			res = utime(setpath, buf);
		else
		{
			res = funionfs_copytowrite(getpath, newpath);
			if (res < 0)
				break;
			res = utime(newpath, buf);
		}

	}
	while (0);

	free(getpath);
	free(setpath);
	free(newpath);

	if (res == -1)
		return -errno;

	return 0;
}

static int
funionfs_open(const char *path, struct fuse_file_info *info)
{
	int res, status;

	char *getpath, *setpath, *newpath;
	struct unionfs_desc *pdesc_tab[3];

	status = funionfs_realpath(path, &getpath, &setpath, &newpath,
				   pdesc_tab);
	if (status < 0)
		return status;

	DEBUG_PRINTF("OPEN(%s,%s,%s)\n", getpath, setpath, newpath);

	res = -1;

	do
	{
		if ((!(status & FILE_EXIST)) || (status & FILE_DELETED))
		{
			errno = ENOENT;
			break;
		}
		res = access(getpath, F_OK);
		if (res < 0)
			break;
	}
	while (0);

	free(getpath);
	free(setpath);
	free(newpath);

	if (res < 0)
		return -errno;

	return 0;

}

static int
funionfs_read(const char *path, char *buf, size_t size, off_t offset,
	      struct fuse_file_info *info)
{
	int fd;
	int res, status;
	off_t res_seek;

	char *getpath, *setpath, *newpath;
	struct unionfs_desc *pdesc_tab[3];

	(void) info;

	status = funionfs_realpath(path, &getpath, &setpath, &newpath,
				   pdesc_tab);
	if (status < 0)
		return status;

	DEBUG_PRINTF("READ(%s,%s,%s): %x %x\n", getpath, setpath, newpath,
		     (unsigned int) offset, (unsigned int) size);

	res = -1;

	do
	{
		if ((!(status & FILE_EXIST)) || (status & FILE_DELETED))
		{
			errno = ENOENT;
			break;
		}
		fd = open(getpath, O_RDONLY);
		res = 0;
	}
	while (0);

	free(getpath);
	free(setpath);
	free(newpath);

	if (res < 0)
		return -errno;
	if (fd < 0)
		return -errno;

	res_seek = lseek(fd, offset, SEEK_SET);
	if (res_seek  == (off_t)-1)
		res = -errno;
	else
	{
		res = read(fd, buf, size);
		if (res < 0)
			res = -errno;
	}

	close(fd);

	if (res < 0)
		return -errno;

	return res;
}

/* liste un repertoire
 * Cet algorithme est simpliste et mauvais. les performances sont dplorables
 * sur des rpertoires gigantesques.  revoir avec un algorithme de hashing
 * ou avec une recherche logarithmique.
 */

static int
funionfs_readdir(const char *path, void *buf, fuse_fill_dir_t filler,
		 off_t offset, struct fuse_file_info *fi)
{
	struct unionfs_dname
	{
		char *name;
		struct unionfs_dname *pnext;
	};

	DIR *dir;
	struct dirent *curdir;
	char *getpath;
	char *curcpy;
	struct unionfs_walk *pwalk;
	int len;
	long type;

	struct unionfs_dname *pfirst, *pcur, *pprev;

	(void) offset;
	(void) fi;

	pwalk = funionfs_openpath(path);
	if (pwalk == NULL)
		return -errno;

	DEBUG_PRINTF("READDIR(%s)\n", path);

	pfirst = NULL;

	while (1)
	{
		getpath = funionfs_readpath(pwalk, &type);
		if (getpath == NULL)
			break;

		dir = opendir(getpath);

		if (dir == NULL)
			return -errno;

		while (1)
		{
			curdir = readdir(dir);
			if (curdir == NULL)
				break;

			if (getrightstr(curdir->d_name, f_opt.del_string) != 0)
			{

				len = strlen(curdir->d_name) - f_opt.del_len;
				curcpy = (char *) malloc(len + 1);
				if (curcpy == NULL)
				{
					closedir(dir);
					funionfs_closepath(pwalk);
					/* free de la liste des reps */
					pcur = pfirst;
					while (pcur != NULL)
					{
						pprev = pcur;
						pcur = pcur->pnext;
						free(pprev->name);
						free(pprev);
					}
					return -errno;
				}
				strncpy(curcpy, curdir->d_name, len);
				curcpy[len] = '\0';

				pcur = pfirst;
				pprev = NULL;
				/* chercher le repertoire et l'effacer de la liste si prsent */
				while (pcur != NULL)
				{
					if (strcmp(curcpy, pcur->name) == 0)
					{
						free(pcur->name);
						if (pprev == NULL)
							pfirst = pcur->pnext;
						else
							pprev->pnext =
								pcur->pnext;
						free(pcur);
						break;
					}
					pprev = pcur;
					pcur = pcur->pnext;
				}
				free(curcpy);
			}
			else
			{

				pcur = pfirst;
				/* chercher le repertoire ne l'insrer que si absent */
				while (pcur != NULL)
				{
					if (strcmp(curdir->d_name, pcur->name)
					    == 0)
					{
						break;
					}
					pcur = pcur->pnext;
				}
				if (pcur == NULL)
				{
					pcur = (struct unionfs_dname *)
						malloc(sizeof
						       (struct unionfs_dname));
					if (pcur == NULL)
					{
						closedir(dir);
						funionfs_closepath(pwalk);
						/* free de la liste des reps */
						pcur = pfirst;
						while (pcur != NULL)
						{
							pprev = pcur;
							pcur = pcur->pnext;
							free(pprev->name);
							free(pprev);
						}
						return -errno;
					}
					pcur->name = strdup(curdir->d_name);
					if (pcur->name == NULL)
					{
						free(pcur);
						closedir(dir);
						funionfs_closepath(pwalk);
						/* free de la liste des reps */
						pcur = pfirst;
						while (pcur != NULL)
						{
							pprev = pcur;
							pcur = pcur->pnext;
							free(pprev->name);
							free(pprev);
						}
						return -errno;
					}
					pcur->pnext = pfirst;
					pfirst = pcur;
				}
			}
		}
		closedir(dir);
	}
	funionfs_closepath(pwalk);

	pcur = pfirst;
	while (pcur != NULL)
	{
		filler(buf, pcur->name, NULL, 0);
		pprev = pcur;
		pcur = pcur->pnext;
		free(pprev->name);
		free(pprev);
	}

	return 0;
}

static int
funionfs_release(const char *path, struct fuse_file_info *info)
{
	(void) info;

	if ((*path == '/') && !strcmp(path + 1, CONTROL_FILE))
		return funionfs_control_write("\n", 2);
	return 0;
}

static int
funionfs_write(const char *path, const char *buf, size_t size, off_t offset,
	       struct fuse_file_info *info)
{
	int res;
	char *getpath, *setpath, *newpath;
	struct unionfs_desc *pdesc_tab[3];
	int status;
	int fd;
	off_t res_seek;

	(void) info;

	DEBUG_PRINTF("write to %s\n", path);

	if ((*path == '/') && !strcmp(path + 1, CONTROL_FILE))
		return funionfs_control_write(buf, size);

	status = funionfs_realpath(path, &getpath, &setpath, &newpath,
				   pdesc_tab);
	if (status < 0)
		return status;

	DEBUG_PRINTF("WRITE(%s,%s,%s): %x %x\n", getpath, setpath, newpath,
		     (unsigned int) offset, (unsigned int) size);

	res = -1;

	do
	{
		if ((!(status & FILE_EXIST)) || (status & FILE_DELETED))
		{
			errno = ENOENT;
			break;
		}
		if (!(status & FILE_CREATABLE))
		{
			errno = EROFS;
			break;
		}
		if (status & FILE_WRITEABLE)
		{
			res = 0;
			fd = open(setpath, O_WRONLY);
		}
		else
		{
			res = funionfs_copytowrite(getpath, newpath);
			if (res < 0)
				break;
			fd = open(newpath, O_WRONLY);
		}

	}
	while (0);

	free(getpath);
	free(setpath);
	free(newpath);

	if (res < 0)
		return -errno;
	if (fd < 0)
		return -errno;

	res_seek = lseek(fd, offset, SEEK_SET);
	if (res_seek  == (off_t)-1)
		res = -errno;
	else
	{
		res = write(fd, buf, size);
		if (res < 0)
			res = -errno;
	}

	close(fd);
	return res;
}

#if FUSE_USE_VERSION == 25
static int
funionfs_statfs(const char *path, struct statvfs *fst)
{
	int rv;
	char *getpath, *setpath, *newpath;
	int status;
	struct unionfs_desc *pdesc_tab[3];

	status = funionfs_realpath(path, &getpath, &setpath, &newpath,
				   pdesc_tab);
	if (status < 0)
		return status;

	DEBUG_PRINTF("STATFS(%s,%s,%s) \n", getpath, setpath, newpath);

	rv = statvfs(newpath, fst);

	return rv;
}

#else

static int
funionfs_statfs(const char *path, struct statfs *fst)
{
	int rv;
	char *getpath, *setpath, *newpath;
	int status;
	struct unionfs_desc *pdesc_tab[3];

	status = funionfs_realpath(path, &getpath, &setpath, &newpath,
				   pdesc_tab);
	if (status < 0)
		return status;

	DEBUG_PRINTF("STATFS(%s,%s,%s) \n", getpath, setpath, newpath);

	rv = statfs(newpath, fst);

	return rv;
}

#endif

#ifdef HAVE_SETXATTR

static int
funionfs_setxattr(const char *path, const char *name, const char *value,
		  size_t size, int flags)
{
	int res;
	char *getpath, *setpath, *newpath;
	struct unionfs_desc *pdesc_tab[3];
	int status;

	status = funionfs_realpath(path, &getpath, &setpath, &newpath,
				   pdesc_tab);
	if (status < 0)
		return status;

	DEBUG_PRINTF("SETXATTR(%s,%s,%s): %s, %s, %d, %d\n", getpath, setpath,
		     newpath, name, value, size, flags);

	res = -1;

	do
	{
		if (status & FILE_EXIST)
		{
			errno = EEXIST;
			break;
		}
		if (!(status & FILE_CREATABLE))
		{
			errno = EROFS;
			break;
		}
		if (status & FILE_WRITEABLE)
			res = lsetxattr(setpath, name, value, size, flags);
		else
		{
			res = funionfs_copytowrite(getpath, newpath);
			if (res < 0)
				break;
			res = lsetxattr(newpath, name, value, size, flags);
		}

	}
	while (0);

	free(getpath);
	free(setpath);
	free(newpath);

	if (res == -1)
		return -errno;

	return 0;
}

static int
funionfs_getxattr(const char *path, const char *name, char *value, size_t size)
{
	int res = 0;
	struct unionfs_desc *pdesc_tab[3];

	char *getpath, *setpath, *newpath;

	res = funionfs_realpath(path, &getpath, &setpath, &newpath, pdesc_tab);
	if (res < 0)
		return res;

	DEBUG_PRINTF("GETXATTR(%s,%s,%s): %s, %d\n", getpath, setpath, newpath,
		     name, size);

	if (res & FILE_EXIST)
	{
		res = lgetxattr(getpath, name, value, size);
	}
	else
	{
		res = -1;
		errno = ENOENT;
	}

	free(getpath);
	free(setpath);
	free(newpath);

	if (res < 0)
		return -errno;

	return res;
}

static int
funionfs_listxattr(const char *path, char *list, size_t size)
{
	int res = 0;
	struct unionfs_desc *pdesc_tab[3];

	char *getpath, *setpath, *newpath;

	res = funionfs_realpath(path, &getpath, &setpath, &newpath, pdesc_tab);
	if (res < 0)
		return res;

	DEBUG_PRINTF("LISTXATTR(%s,%s,%s): %d\n", getpath, setpath, newpath,
		     size);

	if (res & FILE_EXIST)
	{
		res = llistxattr(getpath, list, size);
	}
	else
	{
		res = -1;
		errno = ENOENT;
	}

	free(getpath);
	free(setpath);
	free(newpath);

	if (res < 0)
		return -errno;

	return res;
}

static int
funionfs_removexattr(const char *path, const char *name)
{
	int res;
	char *getpath, *setpath, *newpath;
	struct unionfs_desc *pdesc_tab[3];
	int status;

	status = funionfs_realpath(path, &getpath, &setpath, &newpath,
				   pdesc_tab);
	if (status < 0)
		return status;

	DEBUG_PRINTF("SETXATTR(%s,%s,%s): %s\n", getpath, setpath, newpath,
		     name);

	res = -1;

	do
	{
		if (status & FILE_EXIST)
		{
			errno = EEXIST;
			break;
		}
		if (!(status & FILE_CREATABLE))
		{
			errno = EROFS;
			break;
		}
		if (status & FILE_WRITEABLE)
			res = lremovexattr(setpath, name);
		else
		{
			res = funionfs_copytowrite(getpath, newpath);
			if (res < 0)
				break;
			res = lremovexattr(newpath, name);
		}

	}
	while (0);

	free(getpath);
	free(setpath);
	free(newpath);

	if (res == -1)
		return -errno;

	return 0;

}
#endif /* HAVE_SETXATTR */

#if FUSE_USE_VERSION >= 25

static int
funionfs_create(const char *path, mode_t mode, struct fuse_file_info *fi)
{
	int res, status, len;
	char *getpath, *setpath, *newpath;
	struct unionfs_desc *pdesc_tab[3];

	status = funionfs_realpath(path, &getpath, &setpath, &newpath,
				   pdesc_tab);
	if (status < 0)
		return status;

	DEBUG_PRINTF("CREATE(%s,%s,%s)\n", getpath, setpath, newpath);

	res = -1;

	do
	{
		if (status & FILE_EXIST)
		{
			errno = EEXIST;
			break;
		}
		if (!(status & FILE_CREATABLE))
		{
			errno = EROFS;
			break;
		}
		if (status & FILE_DELETED)
		{
			len = strlen(newpath);
			strcat(newpath, f_opt.del_string);
			res = unlink(newpath);
			if (res == -1)
			{
				break;
			}
			newpath[len] = '\0';
		}
		else
		{
			res = funionfs_pathtowrite(path);
			if (res < 0)
				break;
		}

		res = open(newpath, fi->flags, mode);
		if (res != -1)
			close(res);
		if ((res != -1) && (geteuid() == 0))
		{
			res = chown(newpath, fuse_get_context()->uid,
				    fuse_get_context()->gid);
		}

	}
	while (0);

	free(getpath);
	free(setpath);
	free(newpath);

	if (res < 0)
		return -errno;

	return 0;
}

static int
funionfs_access(const char *path, int mask)
{
	int res, status;
	char *getpath, *setpath, *newpath;
	struct unionfs_desc *pdesc_tab[3];

	status = funionfs_realpath(path, &getpath, &setpath, &newpath,
				   pdesc_tab);
	if (status < 0)
		return status;

	DEBUG_PRINTF("ACCESS(%s,%s,%s)\n", getpath, setpath, newpath);

	res = -1;

	do
	{
		if ((!(status & FILE_EXIST)) || (status & FILE_DELETED))
		{
			errno = ENOENT;
			break;
		}
		res = access(getpath, mask);
		if (res < 0)
			break;
	}
	while (0);

	free(getpath);
	free(setpath);
	free(newpath);

	if (res < 0)
		return -errno;

	return 0;
}

#endif

int curdir_fd; //!<  contains the current directory file descriptor of directory

void *
funionfs_init(void)
{
	if (curdir_fd != 0) {
		if (fchdir(curdir_fd) < 0) {
			fprintf(stderr, "unable to chdir to OLD mount directory");
			perror("fchdir: ");
		}
		close(curdir_fd);
	}
}


void
funionfs_destroy(void *nouse)
{
	struct unionfs_desc *pdesc, *pnext;

	(void) nouse;

	for (pdesc = unionfs_list; pdesc;)
	{
		free(pdesc->path);
		pnext = pdesc->pnext;
		free(pdesc);
		pdesc = pnext;
	}
	unionfs_list = NULL;
	if (log_fd)
		fclose(log_fd);
}

static struct fuse_operations funionfs_oper = {
	.getattr = funionfs_getattr,
	.readlink = funionfs_readlink,
	.mknod = funionfs_mknod,
	.mkdir = funionfs_mkdir,
	.unlink = funionfs_unlink,
	.rmdir = funionfs_rmdir,
	.symlink = funionfs_symlink,
	.rename = funionfs_rename,
	.link = funionfs_link,
	.chmod = funionfs_chmod,
	.chown = funionfs_chown,
	.truncate = funionfs_truncate,
	.utime = funionfs_utime,
	.open = funionfs_open,
	.read = funionfs_read,
	.readdir = funionfs_readdir,
	.write = funionfs_write,
	.statfs = funionfs_statfs,
#ifdef HAVE_SETXATTR
	.setxattr = funionfs_setxattr,
	.getxattr = funionfs_getxattr,
	.listxattr = funionfs_listxattr,
	.removexattr = funionfs_removexattr,
#endif
#if FUSE_USE_VERSION >= 25
	.create = funionfs_create,
	.access = funionfs_access,
#endif
	.init           = funionfs_init,
	.destroy = funionfs_destroy,
	.release = funionfs_release,
};

#define FUNIONFS_OPT(t, p, v) { t, offsetof(f_opt_t, p), v }

enum
{
	KEY_VERSION,
	KEY_HELP
};

static struct fuse_opt funionfs_opts[] = {
	FUNIONFS_OPT("dirs=%s", dirlist, 0),
	FUNIONFS_OPT("copyupuid=%s", uid, 0),
	FUNIONFS_OPT("copyupgid=%s", gid, 0),
	FUNIONFS_OPT("copyupmode=%s", mode, 0),
	FUNIONFS_OPT("del_string=%s", del_string, 0),
	FUNIONFS_OPT("delete=all", delete, 0),
	FUNIONFS_OPT("delete=whiteout", delete, 1),
	FUNIONFS_OPT("copyup=preserve", copyup, 0),
	FUNIONFS_OPT("copyup=currentuser", copyup, 1),
	FUNIONFS_OPT("copyup=mounter", copyup, 2),
	FUNIONFS_OPT("log=%s", log, 0),

	FUSE_OPT_KEY("-V", KEY_VERSION),
	FUSE_OPT_KEY("--version", KEY_VERSION),
	FUSE_OPT_KEY("-h", KEY_HELP),
	FUSE_OPT_KEY("--help", KEY_HELP),
	FUSE_OPT_END
};

static void
usage(const char *progname)
{
	fprintf(stderr,
		"usage: %s [ /upper_rwdir ] mountpoint [options]\n" "\n"
		"general options:\n"
		"    -o opt,[opt...]        mount options\n"
		"    -h   --help            print help\n"
		"    -V   --version         print version\n" "\n"
		"FUNIONFS options:\n"
		"    -o funionfs_debug      print some debugging information\n"
		"    -o dirs=LIST           A colon separated list of directories\n"
		"             /branch1=rw      a directory in a read-write mode\n"
		"             /branch1         if no permission is specified, the branch\n"
		"                              is in a read-write mode\n"
		"             /branch2=ro      a directory in a read-only mode\n"
		"    -o delete=TYPE         specifies how funionfs deletes and renames\n"
		"             all              delete all objects in read-write branch (default)\n"
		"                              and add a whiteout file if objects exists in\n"
		"                              read-only branches.\n"
		"             whiteout         only add a whiteout file in upper (rw) branch\n"
		"                              or delete file in upper branch.\n"
		"    -o copyup=TYPE         specifies how to manage attributes\n"
		"             preserve         preserve uid, gid and mode of object (default)\n"
		"             currentuser      the object owns to the user doing the operation\n"
		"             mounter          the object owns to the mounter\n"
		"    -o copyupuid=N         uid of the copied-up files (no default)\n"
		"    -o copyupgid=N         gid of the copied-up files (no default)\n"
		"    -o copyupmode=N        mode of the copied-up files (no default)\n"
		"    -o del_string=N        string to mark a deleted file(default: %s )\n"
		"\n" "You could use the mount command instead :\n"
		"   mount -t fuse funionfs#/upper_rwdir mountpoint [ options ]\n"
		"\n"
		"There is no implementation of -o delete and -o copyup for now.\n"
		"\n", progname, DELETED_STRING);
}

static int
funionfs_opt_proc(void *data, const char *arg, int key,
		  struct fuse_args *outargs)
{
	(void) data;

	switch (key)
	{
	case FUSE_OPT_KEY_NONOPT:
		if (*f_opt.firstdir == '\0')
		{
			f_opt.firstdir = strdup(arg);
			return 0;
		}
		else if (*f_opt.mountpoint == '\0')
			f_opt.mountpoint = strdup(arg);
		return 1;
	case FUSE_OPT_KEY_OPT:
		if (!strcmp(arg, "-d"))
		{
			f_opt.debug = 1;
		}
		return 1;
	case KEY_HELP:
		usage(outargs->argv[0]);
		fuse_opt_add_arg(outargs, "-ho");
		fuse_main(outargs->argc, outargs->argv, &funionfs_oper);
		exit(1);

	case KEY_VERSION:

		fprintf(stderr, "FUNIONFS version %s\n", VERSION);
#if FUSE_VERSION >= 25
		fuse_opt_add_arg(outargs, "--version");
		fuse_main(outargs->argc, outargs->argv, &funionfs_oper);
#else
		printf("FUSE VERSION: %d.%d\n", FUSE_MAJOR_VERSION,
		       FUSE_MINOR_VERSION);
#endif
		exit(0);

	default:
		fprintf(stderr, "internal error: %d\n", key);
		abort();
	}
}

int
treat_options()
{
	int ret;

	if (!f_opt.del_string)
		return -1;
	f_opt.del_len = strlen(f_opt.del_string);

	if (*f_opt.log)
	{
		log_fd = fopen(f_opt.log, "a");
		if (log_fd == NULL)
		{
			fprintf(stderr, "unable to open log file: %s\n",
				f_opt.log);
			return -1;
		}
	}
	else
		log_fd = NULL;

	if (*f_opt.mountpoint != '\0') {
		if (*f_opt.mountpoint == '/')
			curdir_fd = open(f_opt.mountpoint, O_RDONLY);
		else
			curdir_fd = open(rel2abspath (f_opt.mountpoint, initial_working_dir), O_RDONLY);
		if (curdir_fd < 0) {
			fprintf(stderr, "Unable to open : %s\n", f_opt.mountpoint);
			perror("open ");
		}
	}
	else
		curdir_fd = 0;
		

	return 0;
}


int
main(int argc, char *argv[])
{
	int res;
	struct fuse_args args = FUSE_ARGS_INIT(argc, argv);

	getcwd(initial_working_dir, PATH_MAX);

	signal(2, &signal_handler);
	signal(15, &signal_handler);

	if (fuse_opt_parse(&args, &f_opt, funionfs_opts, funionfs_opt_proc) ==
	    -1)
		exit(1);
	if (parse_and_add_dirs() < 0)
		exit(1);
	if (funionfs_relist() < 0)
		exit(1);
	if (treat_options() < 0)
		exit(1);
	res = fuse_main(args.argc, args.argv, &funionfs_oper);
	fuse_opt_free_args(&args);

	return res;
}
