/* 

                          Firewall Builder

                 Copyright (C) 2000 Vadim Kurland

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: fwcompiler.c,v 1.58 2001/12/06 03:55:25 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Puplic License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
 

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <assert.h>

#include "fwcompiler.h"

#include <glib.h>


extern int optind,opterr,optopt;
extern char *optarg;

static char *fwbuilder_file="objects.xml";

int   debug=0;
int   verbose=0;
int   turn_off_optimisation=0;

xmlDocPtr  doc;
xmlNodePtr root;

GSList     *mem_allocations=NULL;
void       *mem_allocations_mark=NULL;

void* fwb_malloc(int size)
{
    void* chunk=malloc(size);
    if (chunk==NULL) {
	fprintf(stderr,"Out of memory\n");
	exit(1);
    }
    mem_allocations=g_slist_append(mem_allocations , chunk);

    if (debug>1)
	fprintf(stderr,"allocated %d bytes: %p\n",size,chunk);
    
    return chunk;
}

void* fwb_realloc(void *ptr,int size)
{
    GSList *mem_ptr;
    void   *chunk;

    mem_ptr=g_slist_find(mem_allocations,ptr);
    chunk=realloc(ptr,size);
    if (chunk==NULL) {
	fprintf(stderr,"Out of memory\n");
	exit(1);
    }
    mem_allocations=g_slist_remove(mem_allocations, mem_ptr->data);
    mem_allocations=g_slist_append(mem_allocations , chunk);

    return chunk;
}

void fwb_mem_set_mark()
{
    GSList *m=g_slist_last(mem_allocations);

    if (debug>1)
	fprintf(stderr,"Set mark ...\n");

    if (m==NULL) return;    
    mem_allocations_mark=m->data;
}

void fwb_free_mem()
{
    GSList   *mem_ptr, *mem_ptr2;

    mem_ptr=(mem_allocations_mark==NULL)?mem_allocations:
	g_slist_find(mem_allocations,mem_allocations_mark);
    mem_ptr=g_slist_next(mem_ptr);

    while (mem_ptr!=NULL) {
	mem_ptr2=g_slist_next(mem_ptr);

	if (debug>1)
	    fprintf(stderr,"Freeing %p\n",mem_ptr->data);

	free(mem_ptr->data);

	mem_allocations=g_slist_remove(mem_allocations, mem_ptr->data);
	mem_ptr->data=NULL;

	mem_ptr=mem_ptr2;
    }

    mem_allocations_mark=NULL;
}

char* fwb_strdup(const char* str)
{
    char* c;

    c=fwb_malloc(strlen(str)+1);
    strcpy(c,str);
    return c;
}

xmlNodePtr getFirstChildXmlNode(xmlNodePtr node,
				const char* child_element_name)
{
    xmlNodePtr cur;
    for(cur=node->xmlChildrenNode; cur; cur=cur->next) {
        if(cur && !xmlIsBlankNode(cur))  {
	    if (strcmp( child_element_name , getXmlElementName(cur))==0)
		return cur;
        }
    }
    return NULL;
}

const char*  getId(xmlNodePtr node)      { return getStr(node,"id");   }
const char*  getName(xmlNodePtr node)    { return getStr(node,"name"); }
const char*  getComment(xmlNodePtr node) { return getStr(node,"comment"); }

const char*  getRef(xmlNodePtr node)     { return getStr(node,"ref"); }

int    getBool(xmlNodePtr node, const char* attribute)
{
    const char* s;
    s=getStr(node,attribute);
    if (s!=NULL) {
	if (strcasecmp(s,"false")==0) return 0;
	if (strcasecmp(s,"true")==0)  return 1;
    }
    return 0;
}

int    getInt(xmlNodePtr node, const char* attribute)
{
    const char* s;
    s=getStr(node,attribute);
    if (s!=NULL) return atoi(s);
    return(0);
}

const char*  getStr(xmlNodePtr node, const char* attribute)
{
    return FROMXMLCAST(xmlGetProp(node,TOXMLCAST(attribute)));
}

void  setStr(xmlNodePtr node,const char* attribute,const char* val)
{
    xmlSetProp(node,TOXMLCAST(attribute),TOXMLCAST(val));
}


int getOptionInt(xmlNodePtr options_object,const char* option_name)
{
    const char* c;

    c=getOptionStr(options_object,option_name);
    if (c) return atoi(c);
    return 0;
}

int getOptionBool(xmlNodePtr options_object,const char* option_name)
{
    const char* c;

    c=getOptionStr(options_object,option_name);
    if (c && strcasecmp(c,"false")==0) return(0);
    if (c && strcasecmp(c,"true")==0)  return(1);
    if (c && strcmp(c,"0")==0)  return(0);
    if (c && strcmp(c,"1")==0)  return(1);
    return 0;
}

const char* getOptionStr(xmlNodePtr options_object,const char* option_name)
{
    xmlNodePtr cur;
    const char* oname;
    for(cur=options_object->xmlChildrenNode; cur; cur=cur->next) {
	if ( xmlIsBlankNode(cur) ) continue;
	oname=getStr(cur,"name");
	if (oname && strcmp(oname,option_name)==0) {
	    return  FROMXMLCAST( xmlNodeGetContent(cur) );
	}
    }
    return NULL;
}

xmlNodePtr resolveReference(xmlNodePtr node)
{
    const char      *ref;
    xmlNodePtr res;

    if ( isReference(node) ) {
	ref=getStr(node,"ref");
	if (ref!=NULL) {
	    res=getById(ref);
	    if (res==NULL) {
		fprintf(stderr,
"Data format error: reference to non-existing object. Referenced id=%s\n",ref);
		exit(1);
	    }
	    return res;
	}
    }
    return NULL;
}

/*
 *  compare objects. We consider objects equal if their IDs are equal
 */
int cmpObjects(xmlNodePtr obj1,xmlNodePtr obj2)
{
    const char* id1=getId(obj1);
    const char* id2=getId(obj2);
    if (id1==NULL || id2==NULL) return 1;
    return ( strcmp(id1,id2) );
}


int cmpPolicyRule(elementaryRule *r1,elementaryRule *r2)
{
    if (r1->p_direction!=r2->p_direction) {
	if (r1->p_direction!=NULL && r2->p_direction!=NULL) {
	    if ( strcmp(r1->p_direction,r2->p_direction) ) return 1;
	} else
	    return 1;
    }
    
    if ( r1->p_iface!=r2->p_iface ) {
	if (r1->p_iface!=NULL && r2->p_iface!=NULL){
	    if (strcmp(r1->p_iface,r2->p_iface) )  return 1;
	} else
	    return 1;
    }
    
    if ( r1->src!=r2->src ) {
	if (!xmlIsBlankNode(r1->src) && !xmlIsBlankNode(r2->src)){
	    if (cmpObjects(r1->src,r2->src)!=SAME)  return 1;
	} else
	    return 1;
    }

/*    if (r1->src_neg!=r2->src_neg) return 0; */
    
    if ( r1->dst!=r2->dst ) {
	if (!xmlIsBlankNode(r1->dst) && !xmlIsBlankNode(r2->dst)){
	    if (cmpObjects(r1->dst,r2->dst)!=SAME)
		return 1;
	} else
	    return 1;
    }

/*    if (r1->dst_neg!=r2->dst_neg) return 0; */
    
    if ( r1->srv!=r2->srv ) {
	if (!xmlIsBlankNode(r1->srv) && !xmlIsBlankNode(r2->srv)){
	    if (cmpObjects(r1->srv,r2->srv)!=SAME)
		return 1;
	} else
	    return 1;
    }
    
/*    if (r1->srv_neg!=r2->srv_neg) return 0; */
    
    if ( r1->time!=r2->time ) {
	if (!xmlIsBlankNode(r1->time) && !xmlIsBlankNode(r2->time)){
	    if (cmpObjects(r1->time,r2->time)!=SAME)  return 1;
	} else
	    return 1;
    }

    return(SAME);
}

int cmpNatRule(elementaryNatRule *r1,elementaryNatRule *r2)
{

    if ( r1->osrc!=r2->osrc ) {
	if (!xmlIsBlankNode(r1->osrc) && !xmlIsBlankNode(r2->osrc)){
	    if (cmpObjects(r1->osrc,r2->osrc)!=SAME)  return 1;
	} else
	    return 1;
    }

    if ( r1->odst!=r2->odst ) {
	if (!xmlIsBlankNode(r1->odst) && !xmlIsBlankNode(r2->odst)){
	    if (cmpObjects(r1->odst,r2->odst)!=SAME)
		return 1;
	} else
	    return 1;
    }
    
    if ( r1->osrv!=r2->osrv ) {
	if (!xmlIsBlankNode(r1->osrv) && !xmlIsBlankNode(r2->osrv)){
	    if (cmpObjects(r1->osrv,r2->osrv)!=SAME)
		return 1;
	} else
	    return 1;
    }
    
    if ( r1->tsrc!=r2->tsrc ) {
	if (!xmlIsBlankNode(r1->tsrc) && !xmlIsBlankNode(r2->tsrc)){
	    if (cmpObjects(r1->tsrc,r2->tsrc)!=SAME)  return 1;
	} else
	    return 1;
    }

    if ( r1->tdst!=r2->tdst ) {
	if (!xmlIsBlankNode(r1->tdst) && !xmlIsBlankNode(r2->tdst)){
	    if (cmpObjects(r1->tdst,r2->tdst)!=SAME)
		return 1;
	} else
	    return 1;
    }
    
    if ( r1->tsrv!=r2->tsrv ) {
	if (!xmlIsBlankNode(r1->tsrv) && !xmlIsBlankNode(r2->tsrv)){
	    if (cmpObjects(r1->tsrv,r2->tsrv)!=SAME)
		return 1;
	} else
	    return 1;
    }
    
    return(SAME);
}

int  cmpParameter(const char* p1,const char* p2)
{
    if (p1!=p2) {
	if (p1 && p2) {
	    return strcmp(p1,p2);
/*	    if ( strcmp(p1,p2)!=SAME) return !SAME; */
 	} else
		return !SAME;
    }
    return SAME;
}

int  isTripletEmpty(triplet *t)
{
    return (t->p_proto==NULL &&     
	    t->p_src==NULL &&
	    t->p_sprt==NULL &&
	    t->p_dst==NULL &&
	    t->p_mac_src==NULL &&
	    t->p_dprt==NULL &&
	    t->p_icmp_type==NULL &&
	    t->p_icmp_code==NULL );
}

int cmpTriplet(triplet *t1,triplet *t2)
{

    if ( cmpParameter(t1->p_src,t2->p_src)!=SAME) return !SAME;
    if ( cmpParameter(t1->p_dst,t2->p_dst)!=SAME)  return !SAME;
    if ( cmpParameter(t1->p_sprt,t2->p_sprt)!=SAME) return !SAME;
    if ( cmpParameter(t1->p_dprt,t2->p_dprt)!=SAME) return !SAME;
    if ( cmpParameter(t1->p_mac_src,t2->p_mac_src)!=SAME) return !SAME;
    if ( cmpParameter(t1->p_proto,t2->p_proto)!=SAME) return !SAME;
    if ( cmpParameter(t1->p_options,t2->p_options)!=SAME) return !SAME;
    if ( cmpParameter(t1->p_icmp_type,t2->p_icmp_type)!=SAME) return !SAME;
    if ( cmpParameter(t1->p_icmp_code,t2->p_icmp_code)!=SAME) return !SAME;

    if (t1->state_new!=t2->state_new) return !SAME;
    
    return(SAME);
}

int getNetmaskLength(xmlNodePtr node)
{
    int   i;
    int   a1,a2,a3,a4;
    unsigned long nm;
    int   nmlength=0;

    const char* mask;


    if ( isElement(node,"Host") )
	return 32;

    if ( isElement(node,"Network") || isElement(node,"Interface") ) {
	mask=getStr(node,"netmask");

	i=sscanf(mask,"%3d.%3d.%3d.%3d",&a1,&a2,&a3,&a4);
	if (i!=4) {
	    i=sscanf(mask,"%d",&nmlength);
	    if (i==1) return nmlength;
	    else {
		printf("Illegal netmask in object %s\n",getStr(node,"name"));
		exit(1);
	    }
	} else {
	    nm=a4;
	    nm |= a3<<8;
	    nm |= a2<<16;
	    nm |= a1<<24;

	    if (nm==0) return 0;

	    nm = ~ nm;
	    nm++;
	
/* at this point nm should consit of exactly one '1' in binary form */
	    nmlength=0;
	    while ( (nm & 1)==0 ) {
		nm = nm>>1;
		nmlength++;
	    }
	    nmlength=32-nmlength;
	}

	return nmlength;
    }
    return 0;
}

int _convertAddress(const char* addr)
{
    int   i,a1,a2,a3,a4,nm;

    i=sscanf(addr,"%d/%d",&a1,&a2);
    if (i==2 && a1==0 && a2==0) return 0;
    
    i=sscanf(addr,"%3d.%3d.%3d.%3d",&a1,&a2,&a3,&a4);
    if (i!=4) {
	fprintf(stderr,"Illegal IP address %s\n",addr);
	exit(1);
    }

    nm=a4;
    nm |= a3<<8;
    nm |= a2<<16;
    nm |= a1<<24;

    return nm;
}
 
int convertAddress(xmlNodePtr node)
{
    const char* addr  =getStr(node,"address");
    if (addr)
	return _convertAddress(addr);
    else
	return 0;
}
 
int convertNetmask(xmlNodePtr node)
{
    int nmlength=getNetmaskLength(node);
    int i,s,nm;

    nm=0;
    for (s=0x80000000,i=0; i<nmlength; i++) { nm |= s; s=s>>1; }

    return nm;
}

int getBroadcastAddress(xmlNodePtr iface)
{
    int a=convertAddress(iface);
    int n=convertNetmask(iface);

    a= a | (~n);
    return a;
}


/*
 *  examples:
 *
 *    addr1                       addr2                        return value
 *  192.168.1.0/255.255.255.0    192.168.1.0/255.255.255.0     SAME
 *  192.168.1.0/255.255.255.0    192.168.1.1/255.255.255.255   SAME
 *  192.168.1.1/255.255.255.255  192.168.20.0/255.255.255.0    -1
 *
 */
int isSameNetwork(xmlNodePtr node1,xmlNodePtr node2)
{
    int   addr1,addr2;
    int   nm1,nm2,nm;
    
    addr1=convertAddress(node1);
    nm1  =convertNetmask(node1);

    addr2=convertAddress(node2);
    nm2  =convertNetmask(node2);

    nm=(nm1<nm2)?nm1:nm2;
    if ( (addr1 & nm) == (addr2 & nm) ) return 0;
    return -1;
}

/* 
 *  this function checks if address "addr" belongs to the network interface
 *  "iface" is on
 */
int  belongsToNet(xmlNodePtr iface, const char* addr)
{
    int addr1, nm1, addr2;
    
    addr1 = convertAddress(iface);
    nm1   = convertNetmask(iface);

    addr2 = _convertAddress(addr);

    return ( ( addr1 & nm1) == (addr2 & nm1) );
}


/*
 *  examples:
 *
 *    addr1                       addr2                        return value
 *  192.168.1.0/255.255.255.0    192.168.1.0/255.255.255.0     0
 *  192.168.1.0/255.255.255.0    192.168.1.1/255.255.255.255   1
 *  192.168.1.1/255.255.255.255  192.168.1.0/255.255.255.0     -1
 *  192.168.1.1/255.255.255.255  192.168.20.0/255.255.255.0    -2
 *
 */
int cmpAddress(xmlNodePtr node1,xmlNodePtr node2)
{
    int   addr1,addr2;
    int   nm,nm1,nm2;
    int   res;


    res=-2;
    
    addr1=convertAddress(node1);
    nm1  =convertNetmask(node1);

    addr2=convertAddress(node2);
    nm2  =convertNetmask(node2);

    nm=(nm1<nm2)?nm1:nm2;
    if ( (addr1 & nm) == (addr2 & nm) ) {
	if (nm2>nm1)  res=1;
	if (nm2==nm1) res=0;
	if (nm2<nm1)  res=-1;
    }
    return res;
}


int  cmpPPolicyRules(elementaryRule *r1,elementaryRule *r2)
{
    if (cmpTriplet( r1->t,r2->t )!=SAME ) return !SAME;
    
    if ( cmpParameter(r1->p_iface,r2->p_iface)!=SAME) return !SAME;
    if ( cmpParameter(r1->p_group,r2->p_group)!=SAME) return !SAME;
    if ( cmpParameter(r1->p_action,r2->p_action)!=SAME) return !SAME;
    if ( cmpParameter(r1->p_direction,r2->p_direction)!=SAME) return !SAME;
    if ( cmpParameter(r1->p_log,r2->p_log)!=SAME) return !SAME;

    return  SAME;
}

int  cmpPNatRules(elementaryNatRule *r1,elementaryNatRule *r2)
{
    
    if (cmpTriplet( r1->o,r2->o )!=SAME ||
	cmpTriplet( r1->t,r2->t )!=SAME ) return !SAME;

    if ( cmpParameter(r1->p_iface,r2->p_iface)!=SAME) return !SAME;
    if ( cmpParameter(r1->p_group,r2->p_group)!=SAME) return !SAME;
    if ( cmpParameter(r1->p_action,r2->p_action)!=SAME) return !SAME;

    return SAME;
}

xmlNodePtr _getById(xmlNodePtr r,const char *id)
{
    xmlNodePtr  cur, c2;
    const char* myid;

    if (id) {
	myid=getId(r);
	if (myid!=NULL && strcmp(myid,id)==0) return r;
	for(cur=r->xmlChildrenNode; cur; cur=cur->next) {
	    if ( xmlIsBlankNode(cur) ) continue;
	    c2=_getById(cur,id);
	    if (c2 && ! xmlIsBlankNode(c2)) return c2;
	}
    }
    return NULL;
}

xmlNodePtr getById(const char *id)
{
    return _getById(root,id);
}

const char*      getXmlElementName(xmlNodePtr node)
{
    return FROMXMLCAST(node->name);
}

int   isElement(xmlNodePtr node,const char* element_name)
{
    return (strcmp( getXmlElementName(node) , element_name)==0 );
}

int isReference(xmlNodePtr node)
{
    return ( isElement(node, "ObjectRef")  ||
	     isElement(node, "ServiceRef") ||
	     isElement(node, "IntervalRef") );
}

int isGroup(xmlNodePtr node)
{
    return ( isElement(node, "ObjectGroup") ||
	     isElement(node, "ServiceGroup") ||
	     isElement(node, "IntervalGroup") );
}

int isAny(xmlNodePtr node)
{
    return ( isElement(node, "AnyNetwork") ||
	     isElement(node, "AnyIPService") ||
	     isElement(node, "AnyInterval") );
}

int isAnyNetwork(xmlNodePtr node)
{
    return ( isElement(node, "AnyNetwork") );
}

int isAnyService(xmlNodePtr node)
{
    return ( isElement(node, "AnyIPService") );
}

int isAnyInterval(xmlNodePtr node)
{
    return ( isElement(node, "AnyInterval") );
}

int isCustomService(xmlNodePtr node)
{
    return ( isElement(node, "CustomService") );
}

int isDynamicInterface(xmlNodePtr iface)
{
    const char *dyn;
    if (isElement(iface,"Interface")) {
	dyn=getStr(iface,"dyn");
	if (dyn && strcasecmp(dyn,"true")==SAME) return 1;
    }
    return 0;
}

int isExternalInterface(xmlNodePtr iface)
{
    int zone;
    if (isElement(iface,"Interface")) {
	zone=getInt(iface,"security_level");
	if (zone==0) return 1;
    }
    return 0;
}

/*
 *   Expand all groups in the tree
 *
 */
void crawlAndExpandGroups(xmlNodePtr root,xmlNodePtr subtree)
{

    xmlNodePtr c, o, g, no, oldnode;
    
 again:    
    for(c=subtree->xmlChildrenNode; c; c=c->next) {
	if ( xmlIsBlankNode(c) ) continue;

	oldnode=c;
	
	g=c;
	if ( isReference(c) ) {
	    g=resolveReference(c);
	    if (g==NULL) continue;
	}
	if ( isGroup(g) ) {
	    for (o=g->xmlChildrenNode; o; o=o->next) {
		no=xmlCopyNode(o,1);
		xmlAddChild(subtree,no);
	    }
	    xmlUnlinkNode( oldnode );
	    xmlFreeNode( oldnode );
	    goto again;	    
	} 
	crawlAndExpandGroups(root,c);
    }
}

void dumpNATRule(FILE* f,xmlDocPtr doc,elementaryNatRule *nrptr)
{

    fprintf(f,"NAT Rule #%d  :\n",nrptr->num);
    fprintf(f,"----------------\n");
    fprintf(f,"** OSrc:\n");  xmlElemDump(f,doc,nrptr->osrc); fprintf(f,"\n");
    fprintf(f,"** ODst:\n");  xmlElemDump(f,doc,nrptr->odst); fprintf(f,"\n");
    fprintf(f,"** OSrv:\n");  xmlElemDump(f,doc,nrptr->osrv); fprintf(f,"\n");
    fprintf(f,"** TSrc:\n");  xmlElemDump(f,doc,nrptr->tsrc); fprintf(f,"\n");
    fprintf(f,"** TDst:\n");  xmlElemDump(f,doc,nrptr->tdst); fprintf(f,"\n");
    fprintf(f,"** TSrv:\n");  xmlElemDump(f,doc,nrptr->tsrv); fprintf(f,"\n");
    
}

void dumpPolicyRule(FILE* f,xmlDocPtr doc,xmlNodePtr rule)
{
    xmlNodePtr  src,dst,srv;
    int n=getInt(rule,"position");
    
    src=getFirstChildXmlNode(rule,"Src");
    dst=getFirstChildXmlNode(rule,"Dst");
    srv=getFirstChildXmlNode(rule,"Srv");

    fprintf(f,"Policy Rule #%d  :\n",n);
    fprintf(f,"----------------\n");
    fprintf(f,"** Src:\n");   xmlElemDump(f,doc,src); fprintf(f,"\n");
    fprintf(f,"** Dst:\n");   xmlElemDump(f,doc,dst); fprintf(f,"\n");
    fprintf(f,"** Srv:\n");   xmlElemDump(f,doc,srv); fprintf(f,"\n");
    
}

void dumpElementaryPolicyRule(FILE* f,xmlDocPtr doc,elementaryRule *er)
{
    fprintf(f,"Policy Rule #%d  :\n",er->num);
    fprintf(f,"----------------\n");
    fprintf(f,"** Src: %s\n",er->t->p_src);  fprintf(f,"\n");
    fprintf(f,"** Dst: %s\n",er->t->p_dst);  fprintf(f,"\n");
/*    fprintf(f,"** Srv: %s\n",er->p_srv);  fprintf(f,"\n"); */
    
}

elementaryRule* createElementaryPolicyRule()
{
    elementaryRule* prptr;
    
    prptr=fwb_malloc(sizeof(elementaryRule));

    prptr->src=NULL;
    prptr->dst=NULL;
    prptr->srv=NULL;
    prptr->time=NULL;
    
    prptr->src_neg=0;
    prptr->dst_neg=0;
    prptr->srv_neg=0;
    
    prptr->num       = 0;
    prptr->subrule_no= 0;
    prptr->final     = 0;
    
    prptr->master_group=NULL;
    prptr->p_group=NULL;
    prptr->p_action=NULL;
    prptr->p_direction=NULL;
    prptr->p_iface=NULL;
    prptr->p_time=NULL;
    
    prptr->t=fwb_malloc(sizeof(triplet));
    
    prptr->t->p_proto=NULL;

    prptr->t->p_src=NULL;
    prptr->t->p_sprt=NULL;
    prptr->t->p_dst=NULL;
    prptr->t->p_dprt=NULL;

    prptr->t->p_mac_src=NULL;
    
    prptr->t->p_icmp_type=NULL;
    prptr->t->p_icmp_code=NULL;

    prptr->p_action=NULL;
    prptr->p_log=NULL;
    prptr->t->p_options=NULL;

    prptr->t->state_new=1;

    prptr->multiport=MULTIPORT_NONE;
    
    return prptr;
}

elementaryNatRule* createElementaryNatRule()
{

    elementaryNatRule* nrptr;
    
    nrptr=fwb_malloc(sizeof(elementaryNatRule));

    nrptr->osrc=NULL;
    nrptr->odst=NULL;
    nrptr->osrv=NULL;

    nrptr->osrc_neg=0;
    nrptr->odst_neg=0;
    nrptr->osrv_neg=0;
    
    nrptr->tsrc=NULL;
    nrptr->tdst=NULL;
    nrptr->tsrv=NULL;

    nrptr->tsrc_neg=0;
    nrptr->tdst_neg=0;
    nrptr->tsrv_neg=0;
    
    nrptr->num       = 0;
    nrptr->subrule_no= 0;

    nrptr->p_iface=NULL;
    nrptr->p_group=NULL;
    nrptr->p_action=NULL;

    nrptr->o=fwb_malloc(sizeof(triplet));
    nrptr->t=fwb_malloc(sizeof(triplet));
    
    nrptr->o->p_proto=NULL;
    nrptr->o->p_src=NULL;
    nrptr->o->p_sprt=NULL;
    nrptr->o->p_dst=NULL;
    nrptr->o->p_mac_src=NULL;
    nrptr->o->p_dprt=NULL;
    nrptr->o->p_icmp_type=NULL;
    nrptr->o->p_icmp_code=NULL;
    nrptr->o->p_options=NULL;
    nrptr->o->state_new=0;
    
    nrptr->t->p_proto=NULL;
    nrptr->t->p_src=NULL;
    nrptr->t->p_sprt=NULL;
    nrptr->t->p_dst=NULL;
    nrptr->t->p_mac_src=NULL;
    nrptr->t->p_dprt=NULL;
    nrptr->t->p_icmp_type=NULL;
    nrptr->t->p_icmp_code=NULL;
    nrptr->t->p_options=NULL;
    nrptr->t->state_new=0;
    
    return nrptr;
}

char* copyStr(const char* str)
{
    return (str)?fwb_strdup(str):NULL;
}

void copyTriplet(triplet *t1,triplet *t2)
{
    t1->p_proto     = copyStr( t2->p_proto  )   ;
                                      
    t1->p_src       = copyStr( t2->p_src    )   ;
    t1->p_sprt      = copyStr( t2->p_sprt   )   ;
    t1->p_dst       = copyStr( t2->p_dst    )   ;
    t1->p_dprt      = copyStr( t2->p_dprt   )   ;

    t1->p_mac_src   = copyStr( t2->p_mac_src)   ;
    
    t1->p_icmp_type = copyStr( t2->p_icmp_type );
    t1->p_icmp_code = copyStr( t2->p_icmp_code );
    
    t1->p_options   = copyStr( t2->p_options )  ;
    
    t1->state_new   = t2->state_new        ;
}


void copyElementaryPolicyRule(elementaryRule* r1,elementaryRule* r2)
{

    r1->src         =     r2->src         ;
    r1->dst         =     r2->dst         ;
    r1->srv         =     r2->srv         ;
    r1->time        =     r2->time        ;
                                          
    r1->src_neg     =     r2->src_neg     ;
    r1->dst_neg     =     r2->dst_neg     ;
    r1->srv_neg     =     r2->srv_neg     ;
                                          
    r1->num         =     r2->num         ;
    r1->subrule_no  =     r2->subrule_no  ;
    r1->final       =     r2->final       ;
    
    r1->master_group  = copyStr( r2->master_group)  ;
    r1->p_group       = copyStr( r2->p_group     )  ;
    r1->p_direction   = copyStr( r2->p_direction )  ;
    r1->p_iface       = copyStr( r2->p_iface     )  ;
    r1->p_time        = copyStr( r2->p_time     )   ;

    copyTriplet( r1->t , r2->t );
                                           
    r1->p_action      = copyStr( r2->p_action    )  ;
    r1->p_log         = copyStr( r2->p_log       )  ;
}

void copyElementaryNatRule(elementaryNatRule* r1,elementaryNatRule* r2)
{
    r1->num         =     r2->num         ;
    r1->subrule_no  =     r2->subrule_no  ;

    r1->osrc         =     r2->osrc         ;
    r1->odst         =     r2->odst         ;
    r1->osrv         =     r2->osrv         ;
                                          
    r1->osrc_neg     =     r2->osrc_neg     ;
    r1->odst_neg     =     r2->odst_neg     ;
    r1->osrv_neg     =     r2->osrv_neg     ;
                                          
    r1->tsrc         =     r2->tsrc         ;
    r1->tdst         =     r2->tdst         ;
    r1->tsrv         =     r2->tsrv         ;
                                          
    r1->tsrc_neg     =     r2->tsrc_neg     ;
    r1->tdst_neg     =     r2->tdst_neg     ;
    r1->tsrv_neg     =     r2->tsrv_neg     ;
                                          
    r1->p_group       = copyStr( r2->p_group )    ;
    r1->p_iface       = copyStr( r2->p_iface )    ;
    r1->p_action      = copyStr( r2->p_action)    ;

    copyTriplet( r1->o , r2->o );
    copyTriplet( r1->t , r2->t );
    
}


void         addOptionCode(triplet *tr,const char* opt)
{
    if (tr->p_options==NULL) tr->p_options=fwb_strdup(opt);
    else {
	tr->p_options=fwb_realloc(tr->p_options ,
			     strlen(tr->p_options)+strlen(opt)+1 );
	strcat(	tr->p_options , opt );
    }
}

xmlNodePtr initFWCompiler(xmlNodePtr root,const char* fw_name)
{
    xmlNodePtr firewalls,c,fw,policy,nat;

    firewalls=getById(FirewallsId);
    
    if (firewalls==NULL || xmlIsBlankNode(firewalls) ) {
	fprintf(stderr,"Could not find group 'Firewalls' \n");
	exit(1);
    }

    fw=NULL;
    for(c=firewalls->xmlChildrenNode; c; c=c->next) {
	if ( xmlIsBlankNode(c) ) continue;
	if ( strcmp( getName(c) , fw_name)==0 ) {
	    fw=c;
	    break;
	}
    }

    if (fw==NULL) {
	fprintf(stderr,"Firewall object %s not found\n",fw_name);
	exit(1);
    }
    
    policy=getFirstChildXmlNode(fw,"Policy");
    nat   =getFirstChildXmlNode(fw,"NAT");

    if (policy==NULL || nat==NULL) {
	fprintf(stderr,
		"Data format error: no policy or Nat rules in firewall %s\n",
		fw_name);
	exit(1);
    }


    return(fw);
}

  
void scan_InterfacePolicy(xmlNodePtr fw,xmlNodePtr iface,xmlNodePtr policy)
{
    int         num, subrules_n;
    xmlNodePtr  rule;
    xmlNodePtr src,dst,srv;
    xmlNodePtr when;
    
    GSList       *subrules;
    GSList       *unique_subrules;
    GSList       *list_ptr1, *list_ptr2;

    xmlNodePtr c1,c2,c3,c4;
    xmlNodePtr o1,o2,o3,o4;

    elementaryRule  *prptr, *prptr1, *prptr2;
    

    
    num=-1;
    for(rule=policy->xmlChildrenNode; rule; rule=rule->next) {
	if ( xmlIsBlankNode(rule) ) continue;
	num++;
	
	if (getBool(rule,"disabled")) continue;

	if (verbose)
	    printf("Processing rule #%d\n",num);

	
	src=getFirstChildXmlNode(rule,"Src");
	dst=getFirstChildXmlNode(rule,"Dst");
	srv=getFirstChildXmlNode(rule,"Srv");

	when=getFirstChildXmlNode(rule,"When");

/*	
	if (debug)  {
	    xmlElemDump(stdout,doc,rule);
	    fprintf(stderr,"SRC neg=%s\n",xmlGetProp(src,"neg"));
	}
*/
/*
 *   We need to pick every combination of source, destination and service.
 *   There may be duplicates (because we could have used groups with
 *   the same elements or something like that)
 */

	subrules=NULL;
	unique_subrules=NULL;

	for (c1=src->xmlChildrenNode; c1; c1=c1->next) {
	    if ( xmlIsBlankNode(c1) ) continue;

	    o1=resolveReference(c1);
	    if (o1==NULL) continue;
	    
	    for (c2=dst->xmlChildrenNode; c2; c2=c2->next) {
		if ( xmlIsBlankNode(c2) ) continue;

		o2=resolveReference(c2);
		if (o2==NULL) continue;

		for (c3=srv->xmlChildrenNode; c3; c3=c3->next) {
		    if ( xmlIsBlankNode(c3) ) continue;

		    o3=resolveReference(c3);
		    if (o3==NULL) continue;

		    if (when) {
		    
			for (c4=when->xmlChildrenNode; c4; c4=c4->next) {
			    if ( xmlIsBlankNode(c4) ) continue;

			    o4=resolveReference(c4);
			    if (o4==NULL) continue;

			    prptr=createElementaryPolicyRule();

			    prptr->p_iface=getName(iface);
			    prptr->p_direction=getStr(rule,"direction");
			    prptr->p_log=getStr(rule,"log");
			    
			    prptr->src=o1;
			    prptr->dst=o2;
			    prptr->srv=o3;

			    prptr->time=o4;
		    
			    prptr->src_neg=getBool(src,"neg");
			    prptr->dst_neg=getBool(dst,"neg");
			    prptr->srv_neg=getBool(srv,"neg");

			    prptr->num       = num;
			    
			    subrules=g_slist_append(subrules,prptr);
			}
		    } else {
			prptr=createElementaryPolicyRule();

			prptr->p_iface=getName(iface);
			prptr->p_direction=getStr(rule,"direction");
			prptr->p_log=getStr(rule,"log");
			    
			prptr->src=o1;
			prptr->dst=o2;
			prptr->srv=o3;
			
			prptr->time=NULL;
			
			prptr->src_neg=getBool(src,"neg");
			prptr->dst_neg=getBool(dst,"neg");
			prptr->srv_neg=getBool(srv,"neg");
			
			prptr->num       = num;
			
			subrules=g_slist_append(subrules,prptr);
		    }
		}
	    }
	}

/*
 *  Now eliminate duplicates. Only unique subrules will be copied
 *  to the list   unique_subrules
 */
	for (list_ptr1=subrules, subrules_n=0;
	     list_ptr1!=NULL;
	     list_ptr1=g_slist_next(list_ptr1)) {

	    prptr1=(elementaryRule*)(list_ptr1->data);
	    if (prptr1->num==-1) continue;

	    for (list_ptr2=g_slist_next(list_ptr1);
		 list_ptr2!=NULL;
		 list_ptr2=g_slist_next(list_ptr2)) {

		prptr2=(elementaryRule*)(list_ptr2->data);
		
		if (cmpPolicyRule( prptr1,prptr2 )==SAME)
 		    prptr2->num=-1;   /* mark it so we know to skip it later */
	    }
	    unique_subrules=g_slist_append(unique_subrules,prptr1);
	}

	fwb_mem_set_mark();
	
	processPolicyRule(fw,
			  iface,
			  rule,
			  getComment(rule),
			  unique_subrules );
		
	fwb_free_mem();

	list_ptr1=unique_subrules;
	while( list_ptr1!=NULL ) {
 	    prptr=(elementaryRule*)(list_ptr1->data);
	    unique_subrules=g_slist_remove( unique_subrules , prptr );
	    list_ptr1=unique_subrules;
	}

	list_ptr1=subrules;
	while( list_ptr1!=NULL ) {
 	    prptr=(elementaryRule*)(list_ptr1->data);
	    subrules=g_slist_remove( subrules , prptr );
	    list_ptr1=subrules;
	}


	unique_subrules=NULL;
	subrules=NULL;

    }


}



void scan_Policy(xmlNodePtr fw,xmlNodePtr policy)
{
    int         num;
    xmlNodePtr  rule;
    const char       *iface, *direction;
    
    num=-1;
    for(rule=policy->xmlChildrenNode; rule; rule=rule->next) {
	if ( xmlIsBlankNode(rule) ) continue;
	num++;
	
	if (getBool(rule,"disabled")) continue;
	
	iface=getStr(rule,"interface");

	if (iface!=NULL && strlen(iface)>0) {
	    
	    printf(
"Error: rule #%d: Unsupported feature: Interface specified in the general policy rule\n\n",getInt(rule,"position"));
	    printf(
"Firewall Builder does not support interface specification 
in general firewall policy rules. Rules which require
interface specification must be placed in the policy associated
with corresponding firewall interface.\n ");

	    exit(1);
	}

	direction=getStr(rule,"direction");
	
	if (direction!=NULL && strlen(direction)>0) {
	    
	    printf(
"Error: rule #%d: Unsupported feature: Direction specified in the general policy rule\n\n",getInt(rule,"position"));
	    printf(
"Firewall Builder does not support direction specification 
in general firewall policy rules. Rules which require
direction specification must be placed in the policy associated
with  firewall interface.\n ");

	    exit(1);
	}
    }

    scan_InterfacePolicy(fw,NULL,policy);
    
}

void scan_NAT(xmlNodePtr fw,xmlNodePtr nat)
{
    int         num, k;
    xmlNodePtr  rule;
    
    const char  *comment;

    GSList       *nat_subrules;
    GSList       *list_ptr1, *list_ptr2;
    GSList       *unique_nat_subrules;

    xmlNodePtr c1,c2,c3,c4,c5,c6;
    xmlNodePtr o1,o2,o3,o4,o5,o6;

    xmlNodePtr osrc,odst,osrv;
    xmlNodePtr tsrc,tdst,tsrv;
    
    elementaryNatRule     *nrptr, *nrptr1, *nrptr2;

    unique_nat_subrules=NULL;
    
    num=-1;
    for(rule=nat->xmlChildrenNode; rule; rule=rule->next) {
	if ( xmlIsBlankNode(rule) ) continue;
	num++;

	if (getBool(rule,"disabled")) continue;

	if (verbose)
	    printf("Processing rule #%d\n",num);

	
	osrc=getFirstChildXmlNode(rule,"OSrc");
	odst=getFirstChildXmlNode(rule,"ODst");
	osrv=getFirstChildXmlNode(rule,"OSrv");

	tsrc=getFirstChildXmlNode(rule,"TSrc");
	tdst=getFirstChildXmlNode(rule,"TDst");
	tsrv=getFirstChildXmlNode(rule,"TSrv");

	comment=getComment(rule);

/*
 *   We need to pick every combination of source, destination and service.
 *   There may be duplicates (because we could have used groups which have
 *   the same elements or something like that)
 */

	nat_subrules=NULL;
	
	k=0;
	for (c1=osrc->xmlChildrenNode; c1; c1=c1->next) {
	    if ( xmlIsBlankNode(c1) ) continue;

	    o1=resolveReference(c1);
	    if (o1==NULL) continue;

	    for (c2=odst->xmlChildrenNode; c2; c2=c2->next) {
		if ( xmlIsBlankNode(c2) ) continue;

		o2=resolveReference(c2);
		if (o2==NULL) continue;

		for (c3=osrv->xmlChildrenNode; c3; c3=c3->next) {
		    if ( xmlIsBlankNode(c3) ) continue;

		    o3=resolveReference(c3);
		    if (o3==NULL) continue;

		    for (c4=tsrc->xmlChildrenNode; c4; c4=c4->next) {
			if ( xmlIsBlankNode(c4) ) continue;

			o4=resolveReference(c4);
			if (o4==NULL) continue;

			for (c5=tdst->xmlChildrenNode; c5; c5=c5->next) {
			    if ( xmlIsBlankNode(c5) ) continue;

			    o5=resolveReference(c5);
			    if (o5==NULL) continue;

			    for (c6=tsrv->xmlChildrenNode; c6; c6=c6->next) {
				if ( xmlIsBlankNode(c6) ) continue;

				o6=resolveReference(c6);
				if (o6==NULL) continue;

				nrptr=createElementaryNatRule();

				nrptr->osrc=o1;
				nrptr->odst=o2;
				nrptr->osrv=o3;

				nrptr->osrc_neg=getBool(osrc,"neg");
				nrptr->odst_neg=getBool(odst,"neg");
				nrptr->osrv_neg=getBool(osrv,"neg");

				nrptr->tsrc=o4;
				nrptr->tdst=o5;
				nrptr->tsrv=o6;

				nrptr->tsrc_neg=getBool(tsrc,"neg");
				nrptr->tdst_neg=getBool(tdst,"neg");
				nrptr->tsrv_neg=getBool(tsrv,"neg");

				nrptr->num       = num;
/*
  nrptr->options=
  getFirstChildXmlNode(rule,"NATRuleOptions");
*/
				nat_subrules= g_slist_append(nat_subrules,
							      nrptr);

				k++;

			    }
			}
		    }
		}
	    }
	}

/*
 *  Now eliminate duplicates. Only unique subrules will be copied to
 *  the list   unique_nat_subrules
 */

	for (list_ptr1=nat_subrules;
	     list_ptr1!=NULL;
	     list_ptr1=g_slist_next(list_ptr1)) {

	    nrptr1=(elementaryNatRule*)(list_ptr1->data);
	    if (nrptr1->num==-1) continue;
	    
	    for (list_ptr2=g_slist_next(list_ptr1);
		 list_ptr2!=NULL;
		 list_ptr2=g_slist_next(list_ptr2)) {

		nrptr2=(elementaryNatRule*)(list_ptr2->data);

		if (cmpNatRule( nrptr1,nrptr2 )==SAME)
 		    nrptr2->num=-1;   /* mark it so we know to skip it later */
		
	    }
	    unique_nat_subrules=g_slist_append(unique_nat_subrules,nrptr1);
	}


	fwb_mem_set_mark();
	
	processNATRule(fw,
		       rule,
		       getComment(rule),
		       unique_nat_subrules );
		
	fwb_free_mem();

	list_ptr1=unique_nat_subrules;
	while( list_ptr1!=NULL ) {
 	    nrptr=(elementaryNatRule*)(list_ptr1->data);
	    unique_nat_subrules=g_slist_remove( unique_nat_subrules , nrptr );
	    list_ptr1=unique_nat_subrules;
	}

	list_ptr1=nat_subrules;
	while( list_ptr1!=NULL ) {
 	    nrptr=(elementaryNatRule*)(list_ptr1->data);
	    nat_subrules=g_slist_remove( nat_subrules , nrptr );
	    list_ptr1=nat_subrules;
	}

	nat_subrules=NULL;
	unique_nat_subrules=NULL;

    }


}



void usage(const char *name)
{
    fprintf(stderr,"Firewall Builder:  policy compiler %s\n",VERSION);
    fprintf(stderr,"Usage: %s [-v] [-V] [-f filename.xml] [-d destdir] firewall_object_name\n", name);
}

int main(int argc,char* argv[])
{
    xmlNodePtr fw , policy , int_policy , nat;
    xmlNodePtr c1;
    
    char  *fwobject    ;
    char  *destdir     = NULL;
    
    int   opt;

    
    debug=0;
    verbose=0;
    
    if (argc<=1)
    {
        usage(argv[0]);
	exit(1);
    }

    while( (opt=getopt(argc,argv,"vVxnf:d:")) != EOF )
    {
        switch(opt)
        {
	case 'd':
            destdir = strdup(optarg);
            break;
	case 'f':
            fwbuilder_file = strdup(optarg);
            break;
	case 'x':
            debug++;
            break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    usage(argv[0]);
	    exit(1);
	case 'n':
	    turn_off_optimisation=1;
	    break;
	}
    }
    
    if((argc-1) != optind)
    {
       	usage(argv[0]);
	exit(1);
    }

    fwobject = argv[optind++];

    LIBXML_TEST_VERSION;

    doc = xmlParseFile(fwbuilder_file); 
    
    if(!doc)
    {
        fprintf(stderr,"Error parsing file: %s\n", fwbuilder_file);
        exit(1);
    }

    if(destdir)
    {
        if(chdir(destdir))
        {
            fprintf(stderr,"Can't change to: %s\n", destdir);
            xmlFreeDoc(doc);
            exit(1);
        }
    }
    
    root=xmlDocGetRootElement(doc);
    
    if(!root || !root->name)
    {
        fprintf(stderr,"file: %s\n", fwbuilder_file);
        xmlFreeDoc(doc);
        exit(1);
    }

    
/*
 *  Initialize compiler
 *  Second argument is the firewall name
 */
    fw=initFWCompiler(root,fwobject);
    
/*
    if (debug)  {
	xmlElemDump(stdout,doc,root);
    }
*/    

    policy=getFirstChildXmlNode(fw , "Policy");
    crawlAndExpandGroups(root,policy);


    nat   =getFirstChildXmlNode(fw , "NAT");
    crawlAndExpandGroups(root,nat);

    for(c1=fw->xmlChildrenNode; c1; c1=c1->next) {
	if ( xmlIsBlankNode(c1) ) continue;
	if (isElement(c1,"Interface")) {
	    int_policy=getFirstChildXmlNode(c1,"InterfacePolicy");
	    if (int_policy!=NULL) crawlAndExpandGroups(root,int_policy);
	}
    }

    c1=getFirstChildXmlNode(fw,"FirewallOptions");

    prologue(fw,c1);

/***********************************************************************
 *                       NAT
 */

    if (verbose)
	printf("Processing NAT rules ...\n");
    
    scan_NAT(fw,nat);
    
/***********************************************************************
 * 
 *                       Interface Policy
 *  there may be policy attached to every interface, so we are going to
 *  have to scan them all
 */
    if (verbose)
	printf("Processing interface policy rules ...\n");
    
    for(c1=fw->xmlChildrenNode; c1; c1=c1->next) {
	if ( xmlIsBlankNode(c1) ) continue;
	if (isElement(c1,"Interface")) {
	    int_policy=getFirstChildXmlNode(c1,"InterfacePolicy");
	    if (int_policy) {
		scan_InterfacePolicy(fw, c1, int_policy );
	    }
	}
    }

    
/***********************************************************************
 *                       Policy
 */
    if (verbose)
	printf("Processing policy rules ...\n");
    
    scan_Policy(fw, policy );
    
    c1=getFirstChildXmlNode(fw,"FirewallOptions");
    epilogue(fw,c1);
    
    xmlFreeDoc(doc);  

    printf("%s: Policy compiled sucessfully\n",fwobject);
    
    exit(0);
}





