/* 

                          Firewall Builder

                 Copyright (C) 2000 Vadim Kurland

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: FirewallDialog.cc,v 1.105 2001/12/27 06:48:47 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "config.h"

#include "fwbuilder/Firewall.hh"
#include "fwbuilder/FWOptions.hh"
#include "FirewallDialog.hh"
#include "InterfaceDialog.hh"

#include "GenericBackgroundOpDialog.hh"
#include "MessageDialog.hh"

#include "fwbuilder/Policy.hh"
#include "fwbuilder/InterfacePolicy.hh"
#include "fwbuilder/NAT.hh"

#include "PolicyDialog.hh"

#include "fwbuilder/FWObjectDatabase.hh"

#include "IPAddressWidget.hh"

#include "helpers.hh"
#include "fwbuilder/snmp.hh"
#include "fwbuilder/dns.hh"

#include "FileSel.hh"
#include "Preferences.hh"

#include "main_window.hh"

#include <gtk--.h>

#define GENERAL_NOTEBOOK_PAGE          0
#define INTERFACES_NOTEBOOK_PAGE       1
#define SNMP_NOTEBOOK_PAGE             2
#define COMPILER_NOTEBOOK_PAGE         3
#define NO_FIREWALL_NOTEBOOK_PAGE      4
#define IPFILTER_NOTEBOOK_PAGE         5
#define IPTABLES_NOTEBOOK_PAGE         6
#define CISCO_NOTEBOOK_PAGE            7

#define NO_OS_NOTEBOOK_PAGE            8
#define LINUX24_NOTEBOOK_PAGE          9

using namespace libfwbuilder;




#ifdef HAVE_LIBSNMP

class SNMPSysdescQueryPool: public Pool<SNMP_sysdesc_query>
{
    private:
    
    static SNMPSysdescQueryPool *instance;
    static Mutex                 sysdesc_mutex;
    
    public:
    
    SNMPSysdescQueryPool(size_t initial, size_t max_):Pool<SNMP_sysdesc_query>(initial, max_) {}
    
    static SNMPSysdescQueryPool *getInstance()
    {
        sysdesc_mutex.lock();
        if(instance==NULL)
            instance=new SNMPSysdescQueryPool(0, 32);
        sysdesc_mutex.unlock();
        return instance;
    }
};
 
SNMPSysdescQueryPool *SNMPSysdescQueryPool::instance=NULL;
Mutex SNMPSysdescQueryPool::sysdesc_mutex;

class SNMPInterfaceQueryPool: public Pool<SNMP_interface_query>
{
    private:
    
    static SNMPInterfaceQueryPool *instance;
    static Mutex                   interface_mutex;
    
    public:
    
    SNMPInterfaceQueryPool(size_t initial, size_t max_):Pool<SNMP_interface_query>(initial, max_) {}
    
    static SNMPInterfaceQueryPool *getInstance()
    {
        interface_mutex.lock();
        if(instance==NULL)
            instance=new SNMPInterfaceQueryPool(0, 32);
        interface_mutex.unlock();
        return instance;
    }
};
 
SNMPInterfaceQueryPool *SNMPInterfaceQueryPool::instance=NULL;
Mutex SNMPInterfaceQueryPool::interface_mutex;

#endif


FirewallDialog::FirewallDialog(FWObject *obj) 
{ 
    set_name("firewall_dialog");

    interface_policy_list=NULL;

    object=obj;

    options=NULL;

    fw_addr=manage(new IPAddressWidget());
    table16->attach(*fw_addr, 1, 2, 1, 2, GTK_FILL, 0, 0, 0);
    fw_addr->changed.connect(SigC::slot(this,&FirewallDialog::on_changed));

    firewall_dialog_notebook->switch_page.connect(SigC::slot(this, 
					   &FirewallDialog::on_switch_page));


    if_list->select_row.connect(SigC::slot(this, 
				  &FirewallDialog::on_if_list_select_row));



    ((Gtk::Label*)(iptables_firewall_is_in_any->get_child()))->
	set_alignment( 0.0 , 0.0 );
//    ((Gtk::Label*)(iptables_firewall_is_in_any->get_child()))->
//	set_justify(GTK_JUSTIFY_LEFT);


    ((Gtk::Label*)(iptables_log_tcp_seq->get_child()))->set_alignment(0.0,0.5);
    ((Gtk::Label*)(iptables_log_tcp_opt->get_child()))->set_alignment(0.0,0.5);
    ((Gtk::Label*)(iptables_log_ip_opt->get_child()))->set_alignment(0.0,0.5);
    ((Gtk::Label*)(iptables_numeric_log_levels->get_child()))->set_alignment(0.0,0.5);

    ((Gtk::Label*)(iptables_debug->get_child()))->set_alignment(0.0,0.5);
    ((Gtk::Label*)(iptables_load_modules->get_child()))->set_alignment(0.0,0.5);
    ((Gtk::Label*)(iptables_no_optimisation->get_child()))->set_alignment(0.0,0.5);

    ((Gtk::Label*)(iptables_firewall_is_in_any->get_child()))->set_alignment(0.0,0.5);
    ((Gtk::Label*)(iptables_manage_virtual_addr->get_child()))->set_alignment(0.0,0.5);
    ((Gtk::Label*)(iptables_log_all_dropped->get_child()))->set_alignment(0.0,0.5);
    ((Gtk::Label*)(iptables_clamp_MSS_to_MTU->get_child()))->set_alignment(0.0,0.5);

    ((Gtk::Label*)(iptables_accept_new_tcp_with_no_syn->get_child()))->set_alignment(0.0,1.0);
    ((Gtk::Label*)(iptables_accept_new_tcp_with_no_syn->get_child()))->set_justify(GTK_JUSTIFY_LEFT);

    ((Gtk::Label*)(iptables_accept_established->get_child()))->set_alignment(0.0,1.0);
    ((Gtk::Label*)(iptables_accept_established->get_child()))->set_justify(GTK_JUSTIFY_LEFT);

//    action_on_reject_lbl->set_alignment(0.0,0.5);
//    action_on_reject_lbl->set_line_wrap(true);

//    network_tab_txt1->set_line_wrap(true);
    network_tab_txt2->set_line_wrap(true);

    host_os_menu["Other"]    =DialogPlugin::magic_empty_option;
    host_os_menu["Linux 2.4"]="linux24";

    host_os->set_menu(host_os_menu);
    host_os->changed.connect(SigC::slot(this,&FirewallDialog::on_host_os_changed));
    host_os->show_all();


    platforms=Resources::global_res->getPlatforms();
    platforms.insert( platforms.begin() , "Other" );

    fw_platform->set_menu(platforms);
    fw_platform->changed.connect(SigC::slot(this,&FirewallDialog::on_fw_platform_changed));
    fw_platform->show_all();

    log_levels["1: alert"  ]="alert"  ;
    log_levels["2: crit"   ]="crit"   ;
    log_levels["3: error"  ]="error"  ;
    log_levels["4: warning"]="warning";
    log_levels["5: notice" ]="notice" ;
    log_levels["6: info"   ]="info"   ;
    log_levels["7: debug"  ]="debug"  ;

    ipfilter_log_level->set_menu(log_levels);
    ipfilter_log_level->changed.connect(SigC::slot(this,&FirewallDialog::on_ipfilter_log_level_changed));
    ipfilter_log_level->show_all();

    iptables_log_level->set_menu(log_levels);
    iptables_log_level->changed.connect(SigC::slot(this,&FirewallDialog::on_iptables_log_level_changed));
    iptables_log_level->show_all();

    limit_suffixes.push_back("/second");
    limit_suffixes.push_back("/minute");
    limit_suffixes.push_back("/hour");
    limit_suffixes.push_back("/day");

    iptables_limit_suffix->set_menu(limit_suffixes);
    iptables_limit_suffix->changed.connect( SigC::slot(this,&FirewallDialog::on_iptables_limit_suffix_changed));
    iptables_limit_suffix->show_all();

    actions_on_reject.push_back("ICMP net unreachable");
    actions_on_reject.push_back("ICMP host unreachable");
    actions_on_reject.push_back("ICMP port unreachable");
    actions_on_reject.push_back("ICMP proto unreachable");
    actions_on_reject.push_back("ICMP net prohibited");
    actions_on_reject.push_back("ICMP host prohibited");
    actions_on_reject.push_back("TCP RST");

    iptables_action_on_reject->set_menu(actions_on_reject);
    iptables_action_on_reject->changed.connect(SigC::slot(this,&FirewallDialog::on_iptables_action_on_reject_changed));
    iptables_action_on_reject->show_all();


    on_off_menu["On"]="1";
    on_off_menu["Off"]="0";
    on_off_menu["No change"]=DialogPlugin::magic_empty_option;

    linux24_ip_forward->set_menu(on_off_menu);
    linux24_ip_forward->changed.connect(SigC::slot(this,&FirewallDialog::on_changed));
    linux24_ip_forward->show_all();

    linux24_rp_filter->set_menu(on_off_menu);
    linux24_rp_filter->changed.connect(SigC::slot(this,&FirewallDialog::on_changed));
    linux24_rp_filter->show_all();

    linux24_icmp_echo_ignore_broadcasts->set_menu(on_off_menu);
    linux24_icmp_echo_ignore_broadcasts->changed.connect(SigC::slot(this,&FirewallDialog::on_changed));
    linux24_icmp_echo_ignore_broadcasts->show_all();

    linux24_accept_source_route->set_menu(on_off_menu);
    linux24_accept_source_route->changed.connect(SigC::slot(this,&FirewallDialog::on_changed));
    linux24_accept_source_route->show_all();

    linux24_accept_redirects->set_menu(on_off_menu);
    linux24_accept_redirects->changed.connect(SigC::slot(this,&FirewallDialog::on_changed));
    linux24_accept_redirects->show_all();

    linux24_icmp_ignore_bogus_error_responses->set_menu(on_off_menu);
    linux24_icmp_ignore_bogus_error_responses->changed.connect(SigC::slot(this,&FirewallDialog::on_changed));
    linux24_icmp_ignore_bogus_error_responses->show_all();

    linux24_ip_dynaddr->set_menu(on_off_menu);
    linux24_ip_dynaddr->changed.connect(SigC::slot(this,&FirewallDialog::on_changed));
    linux24_ip_dynaddr->show_all();

    linux24_log_martians->set_menu(on_off_menu);
    linux24_log_martians->changed.connect(SigC::slot(this,&FirewallDialog::on_changed));
    linux24_log_martians->show_all();

    linux24_icmp_echo_ignore_all->set_menu(on_off_menu);
    linux24_icmp_echo_ignore_all->changed.connect(SigC::slot(this,&FirewallDialog::on_changed));
    linux24_icmp_echo_ignore_all->show_all();
                                              
                                              
    linux24_tcp_window_scaling->set_menu(on_off_menu);
    linux24_tcp_window_scaling->changed.connect(SigC::slot(this,&FirewallDialog::on_changed));
    linux24_tcp_window_scaling->show_all();

    linux24_tcp_sack->set_menu(on_off_menu);
    linux24_tcp_sack->changed.connect(SigC::slot(this,&FirewallDialog::on_changed));
    linux24_tcp_sack->show_all();

    linux24_tcp_fack->set_menu(on_off_menu);
    linux24_tcp_fack->changed.connect(SigC::slot(this,&FirewallDialog::on_changed));
    linux24_tcp_fack->show_all();

    linux24_tcp_syncookies->set_menu(on_off_menu);
    linux24_tcp_syncookies->changed.connect(SigC::slot(this,&FirewallDialog::on_changed));
    linux24_tcp_syncookies->show_all();

    linux24_tcp_ecn->set_menu(on_off_menu);
    linux24_tcp_ecn->changed.connect(SigC::slot(this,&FirewallDialog::on_changed));
    linux24_tcp_ecn->show_all();

    linux24_tcp_timestamps->set_menu(on_off_menu);
    linux24_tcp_timestamps->changed.connect(SigC::slot(this,&FirewallDialog::on_changed));
    linux24_tcp_timestamps->show_all();


    object_parameters.push_back( DialogOption( fw_name          , "name"                 ) );
    object_parameters.push_back( DialogOption( comment          , "comment"              ) );
    object_parameters.push_back( DialogOption( fw_addr          , "address"              ) );
    object_parameters.push_back( DialogOption( snmp_r_community , "snmp_read_community"  ) );
    object_parameters.push_back( DialogOption( snmp_w_community , "snmp_write_community" ) );
    object_parameters.push_back( DialogOption( fw_platform      , "platform"             ) );
    object_parameters.push_back( DialogOption( host_os          , "host_OS"              ) );
    object_parameters.push_back( DialogOption( fw_version       , "version"              ) );


    common_options.push_back(    DialogOption( dyn_addr         , "dyn_addr"             ) );

    common_options.push_back(    DialogOption( snmp_description , "snmp_description"     ) );
    common_options.push_back(    DialogOption( snmp_location    , "snmp_location"        ) );
    common_options.push_back(    DialogOption( snmp_contact     , "snmp_contact"         ) );

    common_options.push_back(    DialogOption( cmdline          , "cmdline"              ) );
    common_options.push_back(    DialogOption( compiler         , "compiler"             ) );
    common_options.push_back(    DialogOption( inst_script      , "inst_script"          ) );
    common_options.push_back(    DialogOption( inst_cmdline     , "inst_cmdline"         ) );


    ipfilter_options.push_back( DialogOption( ipfilter_log_level                        , "log_level"      ));
    ipfilter_options.push_back( DialogOption( ipfilter_log_or_block                     , "log_or_block"   ));
    ipfilter_options.push_back( DialogOption( ipfilter_log_body                         , "log_packet_body"));
					      		                      
    iptables_options.push_back( DialogOption( iptables_log_tcp_seq                      , "log_tcp_seq"    ));
    iptables_options.push_back( DialogOption( iptables_log_tcp_opt                      , "log_tcp_opt"    ));
    iptables_options.push_back( DialogOption( iptables_log_ip_opt                       , "log_ip_opt"     ));
					      		                      
    iptables_options.push_back( DialogOption( iptables_log_level                        , "log_level"       ));

    iptables_options.push_back( DialogOption( iptables_numeric_log_levels               , "use_numeric_log_levels" ));


    iptables_options.push_back( DialogOption( iptables_log_prefix                       , "log_prefix"      ));
    iptables_options.push_back( DialogOption( iptables_limit_suffix                     , "limit_suffix"    ));
    iptables_options.push_back( DialogOption( iptables_limit_value                      , "limit_value"     ));
    iptables_options.push_back( DialogOption( iptables_load_modules                     , "load_modules"    ));
					      		                      
    iptables_options.push_back( DialogOption( iptables_debug                            , "debug"           ));
    iptables_options.push_back( DialogOption( iptables_path                             , "script_env_path" ));
					     
    iptables_options.push_back( DialogOption( iptables_action_on_reject                 , "action_on_reject"));
					     
    iptables_options.push_back( DialogOption( iptables_firewall_is_in_any               , "firewall_is_part_of_any_and_networks"));
    iptables_options.push_back( DialogOption( iptables_no_optimisation                  , "no_optimisation"));
    iptables_options.push_back( DialogOption( iptables_manage_virtual_addr              , "manage_virtual_addr"     ));

    iptables_options.push_back( DialogOption( iptables_log_all_dropped                  , "log_all_dropped"));

    iptables_options.push_back( DialogOption( iptables_clamp_MSS_to_MTU                  , "clamp_mss_to_mtu"));

    iptables_options.push_back( DialogOption( iptables_accept_new_tcp_with_no_syn       , "accept_new_tcp_with_no_syn"));
					     
    iptables_options.push_back( DialogOption( iptables_accept_established               , "accept_established"));
					     
    linux24_options.push_back( DialogOption(  linux24_ip_forward                        , "linux24_ip_forward"));
    linux24_options.push_back( DialogOption(  linux24_rp_filter                         , "linux24_rp_filter"));
    linux24_options.push_back( DialogOption(  linux24_icmp_echo_ignore_broadcasts       , "linux24_icmp_echo_ignore_broadcasts"));
    linux24_options.push_back( DialogOption(  linux24_accept_source_route               , "linux24_accept_source_route"));
    linux24_options.push_back( DialogOption(  linux24_accept_redirects                  , "linux24_accept_redirects"));
    linux24_options.push_back( DialogOption(  linux24_icmp_ignore_bogus_error_responses , "linux24_icmp_ignore_bogus_error_responses"));
    linux24_options.push_back( DialogOption(  linux24_ip_dynaddr                        , "linux24_ip_dynaddr"));
    linux24_options.push_back( DialogOption(  linux24_log_martians                      , "linux24_log_martians"));
    linux24_options.push_back( DialogOption(  linux24_icmp_echo_ignore_all              , "linux24_icmp_echo_ignore_all"));
					    
					    
    linux24_options.push_back( DialogOption(  linux24_tcp_window_scaling                , "linux24_tcp_window_scaling"));
    linux24_options.push_back( DialogOption(  linux24_tcp_sack                          , "linux24_tcp_sack"));
    linux24_options.push_back( DialogOption(  linux24_tcp_fack                          , "linux24_tcp_fack"));
    linux24_options.push_back( DialogOption(  linux24_tcp_syncookies                    , "linux24_tcp_syncookies"));
    linux24_options.push_back( DialogOption(  linux24_tcp_ecn                           , "linux24_tcp_ecn"));
    linux24_options.push_back( DialogOption(  linux24_tcp_timestamps                    , "linux24_tcp_timestamps"));
    linux24_options.push_back( DialogOption(  linux24_tcp_fack                          , "linux24_tcp_fack"));
					    
    linux24_options.push_back( DialogOption(  linux24_tcp_fin_timeout                    , "linux24_tcp_fin_timeout" ));
    linux24_options.push_back( DialogOption(  linux24_tcp_keepalive_interval             , "linux24_tcp_keepalive_interval"));

}

FirewallDialog::~FirewallDialog() 
{ 
}

void FirewallDialog::hideAllFirewallTabs()
{
    firewall_dialog_notebook->get_nth_page(NO_FIREWALL_NOTEBOOK_PAGE)->hide();
    firewall_dialog_notebook->get_nth_page(IPFILTER_NOTEBOOK_PAGE)->hide();
    firewall_dialog_notebook->get_nth_page(IPTABLES_NOTEBOOK_PAGE)->hide();
    firewall_dialog_notebook->get_nth_page(CISCO_NOTEBOOK_PAGE)->hide();
}

void FirewallDialog::hideAllOSTabs()
{
    firewall_dialog_notebook->get_nth_page(NO_OS_NOTEBOOK_PAGE)->hide();
    firewall_dialog_notebook->get_nth_page(LINUX24_NOTEBOOK_PAGE)->hide();
}

void FirewallDialog::showFirewallTab(guint tab_no)
{
    assert (tab_no>=NO_FIREWALL_NOTEBOOK_PAGE && tab_no<=CISCO_NOTEBOOK_PAGE);

    hideAllFirewallTabs();
    firewall_dialog_notebook->get_nth_page(tab_no)->show();
}

void FirewallDialog::showOSTab(guint tab_no)
{
    assert (tab_no>=NO_OS_NOTEBOOK_PAGE && tab_no<=LINUX24_NOTEBOOK_PAGE);

    hideAllOSTabs();
    firewall_dialog_notebook->get_nth_page(tab_no)->show();
}

void FirewallDialog::showChosenFirewallTab()
{
    string new_platform=fw_platform->get_value();

    if ( new_platform=="ipfilter") {
	showFirewallTab(IPFILTER_NOTEBOOK_PAGE);
    } else 
	if ( new_platform=="iptables") {
	    showFirewallTab(IPTABLES_NOTEBOOK_PAGE);
	} else
	    if ( new_platform=="cisco") {
		showFirewallTab(CISCO_NOTEBOOK_PAGE);
	    } else
		showFirewallTab(NO_FIREWALL_NOTEBOOK_PAGE);
}

void FirewallDialog::showChosenOSTab()
{
    string new_os=host_os->get_value();

    if ( new_os=="linux24") {
	showOSTab(LINUX24_NOTEBOOK_PAGE);
    } else
	showOSTab(NO_OS_NOTEBOOK_PAGE);
}


/**
 *   this callback is called when user changes host OS
 */
void FirewallDialog::on_host_os_changed()
{
/* hide all host OS related pages */
    hideAllOSTabs();
    showChosenOSTab();

    string new_os=host_os->get_value();

    saveOptionMenu(object, host_os , "host_OS");
    Firewall::cast(object)->setDefaults(); 

    if (new_os=="linux24")	loadPageOptions(options,linux24_options );

    on_changed();
}

/**
 *   this callback is called when user changes firewall platform
 */
void FirewallDialog::on_fw_platform_changed()
{
    hideAllFirewallTabs();
    showChosenFirewallTab();

    string new_platform=fw_platform->get_value();

    saveOptionMenu(object, fw_platform , "platform");
    Firewall::cast(object)->setDefaults();  // set default parameters for chosen platform

    if (new_platform=="ipfilter")  loadPageOptions(options,ipfilter_options );
    if (new_platform=="iptables")  loadPageOptions(options,iptables_options );

    on_changed();
}

void FirewallDialog::on_ipfilter_log_level_changed()
{
    string nl=ipfilter_log_level->get_value();
    string  ol=options->getStr("log_level" );
    if (nl!=ol) on_changed();
}

void FirewallDialog::on_iptables_log_level_changed()
{
    string nl=iptables_log_level->get_value();
    string ol=options->getStr("log_level" );
    if (nl!=ol) on_changed();
}

void FirewallDialog::on_iptables_limit_suffix_changed()
{
    string nl=iptables_limit_suffix->get_value();
    string  ol=options->getStr("limit_suffix" );
    if (nl!=ol) on_changed();
}

void FirewallDialog::on_iptables_action_on_reject_changed()
{
    string naor=iptables_action_on_reject->get_value();
    string oaor=options->getStr("action_on_reject" );
    if (naor!=oaor) on_changed();
}

void FirewallDialog::on_switch_page(Gtk::Notebook_Helpers::Page* p0,guint p1)
{
    // do nothing if dialog is not visible
    // it turns out, "switch_page" signal gets invoked 
    // when notebook pages are destroyed
    if(!is_visible()) 
        return;

    updateMainMenu();
    switchToPage(p1);
}


void FirewallDialog::showInterfaces()
{
    if_list->freeze();
    if_list->clear();

    vector<FWObject*>::iterator m;
    gint row;
    Interface       *intf;

    if ( ((Gtk::Bin*)interface_policy_vprt)->get_child()!=NULL )
	((Gtk::Bin*)interface_policy_vprt)->remove();

    for (row=0,m=((Host*)object)->begin(); 
	 m!=((Host*)object)->end(); 
	 ++m) {

	if(Interface::isA((*m))) 
        {
	    if_list->addInterface( Interface::cast(*m) );
	    ++row;
	}
    }
    if_list->thaw();
    if_list->show_all();

    intf=Interface::cast( object->getFirstByType( Interface::TYPENAME ) );

    if (intf)
	showInterfacePolicy( intf );
}

void FirewallDialog::showInterfacePolicy( Interface* intf )
{
    InterfacePolicy *ipol;

    if (intf) {
	ipol=InterfacePolicy::cast(
	    intf->getFirstByType( InterfacePolicy::TYPENAME ) );
	if (ipol==NULL) {
	    ipol=new InterfacePolicy();
	    intf->add(ipol);
	}

//	DialogPlugin *dlg= manage( ipol->CreateDialog() );

	if ( ((Gtk::Bin*)interface_policy_vprt)->get_child()!=NULL )
	    ((Gtk::Bin*)interface_policy_vprt)->remove();

	interface_policy_list=manage(new PolicyList(ipol));
	interface_policy_vprt->add( *interface_policy_list );
	interface_policy_list->Build();
	interface_policy_list->show_all();

	interface_policy_frame->set_label( 
	    string("Policy attached to interface ")+intf->getName() );

	/*  When user presses Enter while one of the policy items is selected,
	 *  or chooses "Edit" pop-up menu item ,
	 *    PolicyList sends signal open_item. 
	 */

	interface_policy_list->open_item.connect(SigC::slot(this,&FirewallDialog::on_open_item));

    }
}

void FirewallDialog::on_open_item(void *arg)
{
    FWObject *obj=(FWObject*)arg;
    if (obj!=NULL) main_w->schedule_open_object(obj->getId());
}


/*
 *  Regarding FirewallOptions *options
 *
 *  Unfortunately we have to find this objects every time we load data
 *  into dialog because it may change. In particular, it changes when
 *  user clicks "Undo". This happens because dialog actually works
 *  with a copy of original object in ScratchPad. When user clicks
 *  Undo this copy is overriden with data from original; when this
 *  happens, all child objects are recreated.  (vk)
 */

/**
 *  This method is called to load data from the object into dialog
 *  This happens at the very beginning and when user clicks "Undo"
 *  (vk)
 */
void FirewallDialog::wrk2dlg()
{
    options=(Firewall::cast(object))->getOptionsObject();
    assert(options!=NULL);

    fw_name->grab_focus();


#ifndef HAVE_LIBSNMP
    button4->set_sensitive(false);
    button16->set_sensitive(false);
#endif


    loadPageOptions(object, object_parameters );

    loadPageOptions(options, common_options );

    showInterfaces();

    showChosenFirewallTab();
    showChosenOSTab();

    loadPageOptions(options,  ipfilter_options );
    loadPageOptions(options,  iptables_options );
    loadPageOptions(options,  linux24_options );
}

/**
 *    This method is called to copy dialog data into the object (when
 *    user clicks "Apply" ) (vk)
 */
bool FirewallDialog::dlg2wrk()
{
/*
 *  check if name is valid
 */
    if ( ! checkObjectName(fw_name->get_text()) ) return(false);
    if ( ! dyn_addr->get_active() &&
	 ! checkIPaddress(fw_addr->getAsString()) ) return(false);

    savePageOptions(object, object_parameters );

    savePageOptions(options, common_options );

    if (string(object->getStr("platform"))=="ipfilter") 
	savePageOptions(options,  ipfilter_options );
    if (string(object->getStr("platform"))=="iptables") 
	savePageOptions(options,  iptables_options );

    if (string(object->getStr("host_OS"))=="linux24") 
	savePageOptions(options,  linux24_options );


    return(true);
}

void FirewallDialog::on_changed()
{
    data_changed_flag(true);
}

void FirewallDialog::on_new_if_clicked()
{ 
    bool res;

    Interface *intf=new Interface();

    InterfaceDialog *id = new InterfaceDialog(intf);
    res=id->run();
    delete id;
    if (res) {
	if ( (Firewall::cast(object))->Appropriate(intf) ) 
	    object->add(intf);

	intf->add( new InterfacePolicy() );

	showInterfaces();
	data_changed_flag(true);
    } else
	delete intf;
}

void FirewallDialog::on_del_if_clicked()
{   
    Interface *intf=getSelectedInterface();
    if (intf) {
	object->remove(intf);
	showInterfaces();
	data_changed_flag(true);
    }
}


void FirewallDialog::on_edit_clicked()
{   
    bool res;

    Interface *intf=getSelectedInterface();
    if(intf) 
    {
	InterfaceDialog *id = new InterfaceDialog( intf );
	res=id->run();
	delete id;
	if(res) 
        {
	    showInterfaces();
	    data_changed_flag(true);
	}
    }
}

void FirewallDialog::on_snmp_get_descr_released()
{
#ifdef HAVE_LIBSNMP

    string addr;
    string rcomm=snmp_r_community->get_text();

    if ( dyn_addr->get_active() ) {
	addr=fw_name->get_text();
    } else {
	if ( ! checkIPaddress(fw_addr->getAsString()) ) return;
	addr=fw_addr->getAsString();
    }


    if(rcomm.empty()) 
    {
	MessageDialog::Error("No SNMP community string defined");
	return;
    }

    int t=Preferences::global_prefs->getOptInt("/FWBuilderPreferences/Network/SNMPTimeout", -1);
    SNMP_sysdesc_query *q=SNMPSysdescQueryPool::getInstance()->lease();
    q->init(addr , rcomm,
	    Preferences::global_prefs->getOptInt("/FWBuilderPreferences/Network/SNMPRetries",  SNMP_DEFAULT_RETRIES),
	    t==-1?SNMP_DEFAULT_TIMEOUT:(1000000L*t)
    );


    GenericBackgroundOpDialog bdisp(q);
    
    try
    {
        bdisp.execute();
        snmp_description->set_point(0);
        snmp_description->forward_delete( snmp_description->get_length() );
        snmp_description->insert( q->getDescr());
        
        snmp_location->set_text(q->getLocation());
        snmp_contact->set_text(q->getContact());

        data_changed_flag(true);
    } catch(const FWException &ex)
    {
        //do nothing
    }
#endif
}

void FirewallDialog::on_snmp_get_clicked()
{
}

void FirewallDialog::on_snmp_get_released()
{   
#ifdef HAVE_LIBSNMP
    string addr;
    string rcomm=snmp_r_community->get_text();

    if ( dyn_addr->get_active() ) {
	addr=fw_name->get_text();
    } else {
	if ( ! checkIPaddress(fw_addr->getAsString()) ) return;
	addr=fw_addr->getAsString();
    }

    if ( rcomm.empty() ) {
	MessageDialog::Error("No SNMP community string defined");
	return;
    }

    FWObject *o;
    while ( (o=object->getFirstByType(Interface::TYPENAME))!=NULL )
	object->remove(o);

    int t=Preferences::global_prefs->getOptInt("/FWBuilderPreferences/Network/SNMPTimeout", -1);
    SNMP_interface_query *q=SNMPInterfaceQueryPool::getInstance()->lease();
    q->init(addr,  rcomm,
	    Preferences::global_prefs->getOptInt("/FWBuilderPreferences/Network/SNMPRetries",  SNMP_DEFAULT_RETRIES),
	    t==-1?SNMP_DEFAULT_TIMEOUT:(1000000L*t)
    );

    GenericBackgroundOpDialog bdisp(q);
    
    try
    {
        bdisp.execute();
        
        const map<int, Interface> &intf = q->getInterfaces();
        for(map<int, Interface>::const_iterator i=intf.begin();i!=intf.end(); ++i)
        {
            Interface *ifs=new Interface((*i).second);
            if((Firewall::cast(object))->Appropriate(ifs))
            {
                ifs->add(new InterfacePolicy());
                object->add( ifs );
            }
        }
        showInterfaces();
        data_changed_flag(true);
    } catch(const FWException &ex)
    {
        //do nothing
    }
#endif
}

void FirewallDialog::on_find_compiler_clicked()
{   
    FileSel *fs=new FileSel("Find compiler",
                            Preferences::global_prefs->getWdir(),
                            "");
    string cmp=fs->run();
    delete fs;
    if(cmp!="")
	compiler->set_text( cmp );
}

void FirewallDialog::on_find_install_script_clicked()
{   
    FileSel *fs=new FileSel("Find install script",
                            Preferences::global_prefs->getWdir(),
                            "");
    string ins=fs->run();
    delete fs;
    if(ins!="")
	inst_script->set_text( ins );
}

void FirewallDialog::on_if_list_select_row(gint row, 
					   gint column, 
					   GdkEvent *event)
{   
    Interface *intf;

    string id=if_list->getIdByRow(row);
    if(!id.empty()) 
    {
	intf = Interface::cast( object->getById( id ) );
	if(intf) 
            showInterfacePolicy( intf );
    }
}

void FirewallDialog::on_add_rule_clicked()
{   
    interface_policy_list->appendRuleAtBottom();
    return;

    Interface *intf=getSelectedInterface();
    if (intf) {
	InterfacePolicy *ip=InterfacePolicy::cast(
	    intf->getFirstByType( InterfacePolicy::TYPENAME ) );
	if (ip) {
	    ip->appendRuleAtBottom();
	    showInterfacePolicy( intf );
	}
    }
}

void FirewallDialog::on_del_rule_clicked()
{   
}

Interface* FirewallDialog::getInterfaceByRow(int row)
{
    Interface *intf=NULL;
    string id=if_list->getIdByRow(row);
    if(!id.empty()) 
	intf = Interface::cast( object->getById( id ) );

    return intf;
}

Interface* FirewallDialog::getSelectedInterface()
{
    Interface *intf=NULL;

    string id=if_list->getSelectedInterfaceId();
    if(!id.empty()) 
	intf = Interface::cast( object->getById( id ) );

    return intf;
}

void FirewallDialog::on_dyn_addr_toggled()
{   
    fw_addr->set_sensitive( ! dyn_addr->get_active() );
}

void FirewallDialog::on_dns_lookup()
{
    vector<IPAddress> addr;

    try {
	addr=DNS::getHostByName( fw_name->get_text() );
    } catch (FWException &e)
    {
	MessageDialog::Error( e.toString() );
	return;
    }

    fw_addr->setAddress( addr.front() );
    data_changed_flag(true);
}

void FirewallDialog::switchToPage(guint page_n)
{
    Gtk::MenuBar *main_menubar=main_w->getMainMenuBar();
    Gtk::Widget *itm;

    if (page_n==INTERFACES_NOTEBOOK_PAGE) {

	itm=find_widget("insert_top",main_menubar);
	if (itm) itm->set_sensitive(true);

	itm=find_widget("append_bottom",main_menubar);
	if (itm) itm->set_sensitive(true);

	itm=find_widget("add_after",main_menubar);
	if (itm) itm->set_sensitive(true);

	itm=find_widget("remove_current",main_menubar);
	if (itm) itm->set_sensitive(true);

	itm=find_widget("move_up",main_menubar);
	if (itm) itm->set_sensitive(true);

	itm=find_widget("move_down",main_menubar);
	if (itm) itm->set_sensitive(true);
    }

}


/**
 *  enable and disable menu items which make sence only for this object
 */
void FirewallDialog::updateMainMenu()
{
    Gtk::MenuBar *main_menubar=main_w->getMainMenuBar();
    Gtk::Widget *itm;

    DialogPlugin::updateMainMenu();

    itm=find_widget("policy",main_menubar);
    if (itm) itm->set_sensitive(true);

    itm=find_widget("rules_druid",main_menubar);
    if (itm) itm->set_sensitive(true);

    itm=find_widget("compile_policy",main_menubar);
    if (itm) itm->set_sensitive(true);

    Firewall *fwo = Firewall::cast(object);

    FWOptions *fwopt = fwo->getOptionsObject();

    string inst_script=fwopt->getStr("inst_script");
    itm=find_widget("install_policy",main_menubar);
    if (itm) itm->set_sensitive( inst_script!="" );

}



