/*
 *  linux/ibcs/socket.c
 *
 *  Copyright (C) 1994, 1996  Mike Jagdis (jaggy@purplet.demon.co.uk)
 *
 * $Id: socket.c,v 1.11 1998/10/11 10:56:51 jaggy Exp $
 * $Source: /u/CVS/ibcs/iBCSemul/socket.c,v $
 */

#include <linux/config.h>

#include <linux/module.h>
#include <linux/version.h>

#include <asm/uaccess.h>

#include <linux/types.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/ptrace.h>
#include <linux/net.h>
#include <linux/socket.h>
#include <linux/sys.h>

#include <ibcs/ibcs.h>
#include <ibcs/socket.h>
#include <ibcs/map.h>
#include <ibcs/trace.h>


int
ibcs_setsockopt(unsigned long *sp)
{
	int error;
	int level, optname;

	error = verify_area(VERIFY_READ,
			((unsigned long *)sp),
			5*sizeof(long));
	if (error)
		return error;

	get_user(level, ((unsigned long *)sp)+1);
	get_user(optname, ((unsigned long *)sp)+2);

#ifdef IBCS_TRACE
	{
	unsigned long optval, optlen;
	get_user(optval, ((unsigned long *)sp)+3);
	get_user(optlen, ((unsigned long *)sp)+4);
	if ((ibcs_trace & (TRACE_STREAMS|TRACE_SOCKSYS))) {
		printk(KERN_DEBUG "%d iBCS: setsockopt level=%d, "
			"optname=%d, optval=0x%08lx, optlen=0x%08lx\n",
			current->pid,
			level, optname, optval, optlen);
	}
	}
#endif

	switch (level) {
		case 0: /* IPPROTO_IP aka SOL_IP */
			/* This is correct for the SCO family. Hopefully
			 * it is correct for other SYSV...
			 */
			optname--;
			if (optname == 0)
				optname = 4;
			if (optname > 4) {
				optname += 24;
				if (optname <= 33)
					optname--;
				if (optname < 32 || optname > 36)
					return -EINVAL;
			}
			put_user(optname, ((unsigned long *)sp)+2);
			break;

		case 0xffff:
			put_user(SOL_SOCKET, ((unsigned long *)sp)+1);
			optname = map_value(sopt_map, optname, 0);
			put_user(optname, ((unsigned long *)sp)+2);

			switch (optname) {
				case SO_LINGER: {
					unsigned long optlen;

					/* SO_LINGER takes a struct linger
					 * as the argument but some code
					 * uses an int and expects to get
					 * away without an error. Sigh...
					 */
					get_user(optlen, ((unsigned long *)sp)+4);
					if (optlen == sizeof(int))
						return 0;
					break;
				}

				/* The following are not currently implemented
				 * under Linux so we must fake them in
				 * reasonable ways. (Only SO_PROTOTYPE is
				 * documented in SCO's man page).
				 */
				case SO_PROTOTYPE:
				case SO_ORDREL:
				case SO_SNDTIMEO:
				case SO_RCVTIMEO:
					return -ENOPROTOOPT;

				case SO_USELOOPBACK:
				case SO_SNDLOWAT:
				case SO_RCVLOWAT:
					return 0;

				/* The following are not currenty implemented
				 * under Linux and probably aren't settable
				 * anyway.
				 */
				case SO_IMASOCKET:
					return -ENOPROTOOPT;
			}

		default:
			/* FIXME: We assume everything else uses the
			 * same level and option numbers. This is true
			 * for IPPROTO_TCP(/SOL_TCP) and TCP_NDELAY
			 * but is known to be incorrect for other
			 * potential options :-(.
			 */
			break;
	}

	return SYS(socketcall)(SYS_SETSOCKOPT, sp);
}


int
ibcs_getsockopt(unsigned long *sp)
{
	int error;
	int level, optname;
	char *optval;
	long *optlen;

	error = verify_area(VERIFY_READ,
			((unsigned long *)sp),
			5*sizeof(long));
	if (error)
		return error;

	get_user(level, ((unsigned long *)sp)+1);
	get_user(optname, ((unsigned long *)sp)+2);
	get_user(optval, ((unsigned long *)sp)+3);
	get_user(optlen, ((unsigned long *)sp)+4);

#ifdef IBCS_TRACE
	if ((ibcs_trace & (TRACE_STREAMS|TRACE_SOCKSYS))) {
		long l;
		get_user(l, optlen);
		printk(KERN_DEBUG "%d iBCS: getsockopt level=%d, "
			"optname=%d, optval=0x%08lx, optlen=0x%08lx[%ld]\n",
			current->pid,
			level, optname, (unsigned long)optval,
			(unsigned long)optlen, l);
	}
#endif

	switch (level) {
		case 0: /* IPPROTO_IP aka SOL_IP */
			/* This is correct for the SCO family. Hopefully
			 * it is correct for other SYSV...
			 */
			optname--;
			if (optname == 0)
				optname = 4;
			if (optname > 4) {
				optname += 24;
				if (optname <= 33)
					optname--;
				if (optname < 32 || optname > 36)
					return -EINVAL;
			}
			put_user(optname, ((unsigned long *)sp)+2);
			break;

		case 0xffff:
			put_user(SOL_SOCKET, ((unsigned long *)sp)+1);
			optname = map_value(sopt_map, optname, 0);
			put_user(optname, ((unsigned long *)sp)+2);

			switch (optname) {
				case SO_LINGER: {
					long l;

					/* SO_LINGER takes a struct linger
					 * as the argument but some code
					 * uses an int and expects to get
					 * away without an error. Sigh...
					 */
					get_user(l, optlen);
					if (l == sizeof(int)) {
						put_user(0, (long *)optval);
						return 0;
					}
					break;
				}

				/* The following are not currently implemented
				 * under Linux so we must fake them in
				 * reasonable ways. (Only SO_PROTOTYPE is
				 * documented in SCO's man page).
				 */
				case SO_PROTOTYPE: {
					unsigned long len;
					error = get_user(len, optlen);
					if (error)
						return error;
					if (len < sizeof(long))
						return -EINVAL;

					error = verify_area(VERIFY_WRITE,
							(char *)optval,
							sizeof(long));
					if (!error) {
						put_user(0, (long *)optval);
						put_user(sizeof(long),
							optlen);
					}
					return error;
				}

				case SO_ORDREL:
				case SO_SNDTIMEO:
				case SO_RCVTIMEO:
					return -ENOPROTOOPT;

				case SO_USELOOPBACK:
				case SO_SNDLOWAT:
				case SO_RCVLOWAT:
				case SO_IMASOCKET: {
					unsigned long len;
					error = get_user(len, optlen);
					if (error)
						return error;
					if (len < sizeof(long))
						return -EINVAL;

					error = verify_area(VERIFY_WRITE,
							(char *)optval,
							sizeof(long));
					if (!error) {
						put_user(1, (long *)optval);
						put_user(sizeof(long),
							optlen);
					}
					return error;
				}
			}

		default:
			/* FIXME: We assume everything else uses the
			 * same level and option numbers. This is true
			 * for IPPROTO_TCP(/SOL_TCP) and TCP_NDELAY
			 * but is known to be incorrect for other
			 * potential options :-(.
			 */
			break;
	}

	return SYS(socketcall)(SYS_GETSOCKOPT, sp);
}
