/*
 * Program to control ICOM radios
 *
 * Tables and chairs
 */
#include "icom.h"
#include <stdio.h>

/*
 * Keyboard mode commands
 */
struct cmdtable cmd[] = {
	{"am",		C_MODEG,	"set mode AM"},
	{"ant",		C_ANT,		"set antenna"},
	{"atten",	C_ATTEN,	"set attenuator"},
	{"band",	C_BAND,		"set band limits"},
	{"bank",	C_BANK,		"set bank (R8500)"},
	{"cap",		C_PROBE,	"query capability"},
	{"chan",	C_CHAN,		"read channel"},
	{"clear",	C_CLEAR,	"clear channel"},
	{"comp",	C_COMP,		"set VFO compensation"},
	{"misc",	C_MISC,		"miscellaneous control"},
	{"cw",		C_MODEG,	"set mode CW"},
	{"down",	C_DOWN,		"step down"},
	{"duplex",	C_DUPLEX,	"set transmit duplex"},
	{"fm",		C_MODEG,	"set mode FM"},
	{"freq",	C_FREQ,		"set frequency"},
	{"key",		C_KEY,		"send CW message"},
	{"lsb",		C_MODEG,	"set mode LSB"},
	{"meter",	C_SIGNAL,	"set meter"},
	{"mode",	C_MODE,		"set mode and BFO compensation"},
	{"offset",	C_OFFSET,	"set VFO offset"},
	{"pad",		C_KEYPAD,	"switch to keypad mode"},
	{"quit",	C_QUIT,		"exit program"},
	{"radio",	C_RADIO,	"select radio"},
	{"rate",	C_RATE,		"set tuning rate"},
	{"restore",	C_RESTORE,	"restore channels"},
	{"reverse",	C_REVER,	"reverse split"},
	{"rsplit",	C_RSPLT,	"set receive split"},
	{"rtty",	C_MODEG,	"set mode RTTY"},
	{"save",	C_SAVE,		"save channels"},
	{"say",		C_ANNC,		"announce control"},
	{"scan",	C_SCAN,		"scan control"},
	{"simplex",	C_SMPLX,	"receive on transmit frequency"},
	{"step",	C_STEP,		"set tuning step"},
	{"trace",	C_DEBUG,	"trace CI-V messages"},	
	{"tune",	C_TUNE,		"set dial tuning step"},
	{"up",		C_UP,		"step up"},
	{"usb",		C_MODEG,	"set mode USB"},
	{"verbose",	C_VERB,		"set verbose"},
	{"vfo",		C_VFO,		"VFO control"},
	{"write",	C_WRITE,	"write channel"},
	{"xsplit",	C_XSPLT,	"set transmit split"},
#ifdef AUDIO
	{"gain",	C_GAIN,		"audio output gain"},
	{"mute",	C_MUTE,		"audio mute"},
	{"port",	C_PORT,		"audio port"},
#endif /* AUDIO */
	{"\0",		C_FREQX,	"set VFO frequency"}
};

/*
 * Keypad mode commands
 */
struct cmdtable key[] = {
	{"\\",		C_ERASE,	"erase input"},
	{"/", 		C_SMPLX,	"receive on transmit frequency"},
	{"=",		C_REVER,	"reverse split"},
	{"[",		C_RSPLT,	"set receive split"},
	{"[a",		C_UP,		"UP step up"},
	{"[b",		C_DOWN,		"DOWN step down"},
	{"[c",		C_RUP,		"RIGHT tuning rate up"},
	{"[d",		C_RDOWN,	"LEFT tuning rate down"},
	{"]",		C_XSPLT,	"set transmit split"},
	{"a",		C_ANT,		"antenna control"},
	{"c",		C_CHAN,		"read channel"},
	{"f",		C_MODEG,	"set mode FM"},
	{"l",		C_MODEG,	"set mode LSB"},
	{"m",		C_MODEG,	"set mode AM"},
	{"q",		C_KEYBD,	"switch to keyboard mode"},
	{"s",		C_STEP,		"set tuning step"},
	{"u",		C_MODEG,	"set mode USB"},
	{"w",		C_WRITE,	"write channel"},
	{"x",		C_CLEAR,	"clear channel"},
#ifdef AUDIO
	{"g",		C_GAIN,		"adjust output gain"},
	{"o",		C_MUTE,		"mute output (toggle)"},
	{"p",		C_PORT,		"select input port"},
#endif /* AUDIO */
	{"\0",		C_FREQX,	"set VFO frequency"}
};

/*
 * Trace control subcommands
 */
struct cmdtable dbx[] = {
	{"all",		P_TRACE|P_ERMSG, "trace bus, packets"},
	{"bus",		P_TRACE,	"trace bus"},
	{"none",	0x0,		"trace none"},
	{"packet",	P_ERMSG,	"trace packet errors"},
	{"\0",		C_ERROR,	"invalid argument"}
};

/*
 * Probe control subcommands
 */
struct cmdtable probe[] = {
	{"ant",		V_SANT | S_ANT1<<8,	"1"},
	{"tune",	V_TUNE | 0x00<<8,	"0"},
	{"scan",	V_SCAN | 0x00<<8,	"stop"},
	{"vfo",		V_SVFO | S_DWOFF<<8,	"nowatch"},
	{"split",	V_SPLIT | S_OFF<<8,	"nosplit"},
	{"atten",	V_ATTEN | S_ATOFF<<8,	"off"},
	{"say",		V_ANNC | S_ALL<<8,	"all"},
	{"meter",	V_SQSG | S_RDSQ<<8,	"squelch"},
	{"misc",	V_CTRL | S_LCL<<8,	"local"},
	{"\0",		C_ERROR,		""}
};

/*
 * Verbose control subcommands
 */
struct cmdtable verbx[] = {
	{"on",		P_VERB,		"verbose"},
	{"off",		0,		"terse"},
	{"\0",		C_ERROR,	"invalid argument"}
};

/*
 * Set vfo (V_SVFO) subcommands
 */
struct cmdtable vfo[] = {
	{"a",		S_VFOA,		"select vfo a"},
	{"b",		S_VFOB,		"select vfo b"},
	{"btoa",	S_BTOA,		"vfo a <- vfo b"},
	{"equal",	S_EQUAL,	"main -> sub"},
	{"mainband",	S_MBAND,	"access main band"},
	{"nowatch",	S_DWOFF,	"dual watch off"},
	{"subband",	S_SBAND,	"access sub band"},
	{"swap",	S_XCHNG,	"main <-> sub"},
	{"watch",	S_DWON,		"dual watch on"},
	{"\0",		C_ERROR,	"invalid argument"}
};

/*
 * Set bank (V_MCHAN) subcommands
 */
struct cmdtable bank[] = {
	{"0",		0,		"bank 0"},
	{"1",		1,		"bank 1"},
	{"2",		2,		"bank 2"},
	{"3",		3,		"bank 3"},
	{"4",		4,		"bank 4"},
	{"5",		5,		"bank 5"},
	{"6",		6,		"bank 6"},
	{"7",		7,		"bank 7"},
	{"8",		8,		"bank 8"},
	{"9",		9,		"bank 9"},
	{"10",		10,		"bank 10"},
	{"11",		11,		"bank 11"},
	{"12",		12,		"bank 12"},
	{"13",		13,		"bank 13"},
	{"14",		14,		"bank 14"},
	{"15",		15,		"bank 15"},
	{"16",		16,		"bank 16"},
	{"17",		17,		"bank 17"},
	{"18",		18,		"bank 18"},
	{"19",		19,		"bank 19"},
	{"auto",	21,		"bank auto"},
	{"skip",	22,		"bank skip"},
	{"prog",	23,		"bank prog"},
	{"prio",	24,		"bank prior"},
	{"\0",		C_ERROR,	"invalid argument"}
};

/*
 * Scan control (V_SCAN) subcommands
 */
struct cmdtable scan[] = {
	{"10",		S_DF10,		"delta-f 10 kHz"},
	{"2.5",		S_DF2,		"delta-f 2.5 kHz"},
	{"20",		S_DF20,		"delta-f 20 kHz"},
	{"5",		S_DF5,		"delta-f 5 kHz"},
	{"50",		S_DF50,		"delta-f 50 kHz"},
	{"deltaf",	S_DFST,		"delta-f scan"},
	{"fdeltaf",	S_FDST,		"fine delta-f scan"},
	{"fix",		S_FIX,		"fix center frequency"},
	{"fprog",	S_FPST,		"fine program scan"},
	{"mem",		S_MSST,		"memory scan"},
	{"memauto",	S_AMST,		"auto write scan"},
	{"memchan",	S_MEMN,		"memory channel scan number"},
	{"memnum",	S_SNST,		"memory scan number"},
	{"memsel",	S_SMST,		"selected mode memory scan"},
	{"noresume",	S_SROFF,	"scan resume off"},
	{"notresume",	S_SRNOT,	"scan resume never"},
	{"novsc",	S_VSOFF,	"VSC off"},
	{"prog",	S_PSST,		"program scan"},
	{"resumea",	S_SRA,		"scan resume a (delay)"},
	{"resumeb",	S_SRB,		"scan resume b"},
	{"skip",	S_DSBM,		"disable memory channel"},
	{"start",	S_START,	"scan"},
	{"stop",	S_OFF,		"stop scan"},
	{"pstop",	S_PXST,		"priority scan"},
	{"unfix",	S_UNFIX,	"unfix center frequency"},
	{"unskip",	S_ENBM,		"enable memory channel"},
	{"vsc",		S_VSON,		"VSC on"},
	{"\0",		C_ERROR,	"invalid argument"}
};
 
/*
 * Split control (S_SPLIT) subcommands
 */
struct cmdtable split[] = {
	{"nosplit",	S_OFF,		"split off"},
	{"split",	S_ON,		"split on"},
	{"cancel",	S_DUPOF,	"cancel duplex"},
	{"-",		S_DUPM,		"select -duplex"},
	{"+",		S_DUPP,		"select +duplex"},
	{"\0",		C_ERROR,	"invalid argument"}
};

/*
 * Attenuator control (S_ATTN) subcommands
 */
struct cmdtable atten[] = {
	{"0",		S_ATOFF,	"off"},
	{"10",		S_AT10,		"10 dB"},
	{"20",		S_AT20,		"20 dB"},
	{"30",		S_AT30,		"30 dB"},
	{"off",		S_ATOFF,	"off"},
	{"\0",		C_ERROR,	"invalid argument"}
};

/*
 * Select antenna (V_SANT) subcommands
 */
struct cmdtable ant[] = {
	{"1",		S_ANT1,		"antenna 1"},
	{"2",		S_ANT2,		"antenna 2"},
	{"\0",		C_ERROR,	"invalid argument"}
};

/*
 * Announce control (V_ANNC) subcommands
 */
struct cmdtable annc[] = {
	{"all",		S_ALL,		"announce all"},
	{"freq",	S_FREQ,		"announce freq"},
	{"\0",		C_ERROR,	"invalid argument"}
};

/*
 * Read signal strength (S_SQSG) subcommands
 */
struct cmdtable meter[] = {
	{"signal",	S_RDSS,		"read signal"},
	{"squelch",	S_RDSQ,		"read squelch"},
	{"\0",		C_ERROR,	"invalid argument"}
};

/*
 * Miscellaneous control (S_CTRL) subcommands
 */
struct cmdtable misc[] = {
	{"ctss",	S_CTSS,		"read CTSS"},
	{"dcs",		S_DCS,		"read DCS"},
	{"dtmf",	S_DTMF,		"read DTMF"},
	{"freq",	S_NXFM,		"next frequency"},
	{"id",		S_RDID,		"read ID"},
	{"local",	S_LCL,		"local control"},
	{"nosearch",	S_SMOFF,	"disable search"},
	{"nospeaker",	S_SPOFF,	"mute speaker"},
	{"notape",	S_TPOFF,	"disable tape recorder"},
	{"nowindow",	S_5OFF,		"disable search window"},
	{"opto",	S_OPTO,		"read OPTO"},
	{"remote",	S_RMT,		"remote control"},
	{"search",	S_SMON,		"enable search"},
	{"speaker",	S_SPON,		"unmute speaker"},
	{"tape",	S_TPON,		"enable tape recorder"},
	{"window",	S_5ON,		"enable search window"},
	{"\0",		C_ERROR,	"invalid argument"}
};

/*
 * Radio identifier decode
 */
struct cmdtable ident[] = {
	{"1271",	0x24,		"1271 UHF Transceiver"},
	{"1275",	0x18,		"1275 UHF Transceiver"},
	{"271",		0x20,		"271 VHF Transceiver"},
	{"275",		0x10,		"275 VHF Transceiver"},
	{"375",		0x12,		"375 VHF Transceiver"},
	{"471",		0x22,		"471 UHF Transceiver"},
	{"475",		0x14,		"475 UHF Transceiver"},
	{"575",		0x16,		"575 VHF Transceiver"},
	{"575",		0x28,		"725 HF Transceiver"},
	{"726",		0x30,		"726 HF Transceiver"},
	{"735",		0x04,		"735 HF Transceiver"},
	{"751",		0x1c,		"751 HF Transceiver"},
	{"761",		0x1e,		"761 HF Transceiver"},
	{"765",		0x2c,		"765 HF Transceiver"},
	{"775",		0x46,		"775 HF Transceiver"},
	{"781",		0x26,		"781 HF Transceiver"},
	{"970",		0x2e,		"970 HF Transceiver"},
	{"R7000",	0x08,		"R7000 VHF/UHF Receiver"},
	{"R71",		0x1A,		"R71 HF Receiver"},
	{"R7100",	0x34,		"R7100 VHF/UHF Receiver"},
	{"R72",		0x32,		"R72 HF Receiver"},
	{"R8500",	0x4a,		"R8500 HF/VHF/UHF Receiver"},
	{"R9000",	0x2a,		"R9000 VHF/UHF Receiver"},
	{"\0",		C_ERROR,	"unknown radio"}
};

/*
 * Broadband HF/VHF/UHF receiver mode decode (R7000)
 *
 * Note that panel FM mode is really wideband FM and panel FMn mode is
 * really FM mode on other radios. Clearing a channel changes the mode
 * to 0xff, but does not change the frequency. Reactivate the channel
 * by loading valid mode.
 */
static struct cmdtable mode2[] = {
	{"AM",		0x0002,		"AM"},
	{"m",		0x0002,		"AM (keypad)"},
	{"FMn",		0x8205,		"FM"},
	{"WFM",		0x0005,		"WFM"},
	{"f",		0x8205,		"FM (keypad)"},
	{"USB",		0x8005,		"USB"},
	{"u",		0x8005,		"USB (keypad)"},
	{"\0",		C_ERROR,	"invalid mode"}
};

/*
 * Broadband MF/HF/VHF/UHF receiver mode decode (R8500)
 */
static struct cmdtable mode3[] = {
	{"LSB",		0x0001,		"LSB"},
	{"l",		0x0001,		"LSB (keypad)"},
	{"USB",		0x8101,		"USB"},
	{"u",		0x8101,		"USB (keypad)"},
	{"AM",		0x8202,		"AM"},
	{"m",		0x8202,		"AM (keypad)"},
	{"AMn",		0x8302,		"AM narrow"},
	{"AMw",		0x8102,		"AM wide"},
	{"CW",		0x8103,		"CW"},
	{"CWn",		0x8203,		"CW narrow"},
	{"FM",		0x8105,		"FM"},
	{"f",		0x8105,		"FM (keypad)"},
	{"FMn",		0x8205,		"FM narrow"},
	{"WFM",		0x8106,		"WFM"},
	{"\0",		C_ERROR,	"invalid mode"}
};

/*
 * Narrowband MF/HF/VHF/UHF transceiver and receiver mode decode
 */
static struct cmdtable mode1[] = {
	{"AM",		0x0002,		"AM"},
	{"AMn",		0x8202,		"AM narrow"},
	{"AMw",		0x8102,		"AM wide"},
	{"m",		0x0002,		"AM (keypad)"},
	{"CW",		0x0003,		"CW"},
	{"CWn",		0x8203,		"CW narrow"},
	{"CWw",		0x8103,		"CW wide"},
	{"FM",		0x0005,		"FM"},
	{"FMn",		0x8205,		"FM narrow"},
	{"FMw",		0x8105,		"FM wide"},
	{"f",		0x0005,		"FM (keypad)"},
	{"LSB",		0x0000,		"LSB"},
	{"LSBn",	0x8200,		"LSB narrow"},
	{"LSBw",	0x8100,		"LSB wide"},
	{"l",		0x0000,		"LSB (keypad)"},
	{"RTTY",	0x0004,		"RTTY"},
	{"RTTYn",	0x8204,		"RTTY narrow"},
	{"RTTYw",	0x8104,		"RTTY wide"},
	{"USB",		0x0001,		"USB"},
	{"USBn",	0x8201,		"USB narrow"},
	{"USBw",	0x8101,		"USB wide"},
	{"u",		0x0001,		"USB (keypad)"},
	{"\0",		C_ERROR,	"invalid mode"}
};

/*
 * Radio control initialization
 */
struct namestruct name[] = {
	{"1271",	0x24,	32,	mode1,	0,	NULL},
	{"1275",	0x18,	32,	mode1,	0,	NULL},
	{"271",		0x20,	32,	mode1,	0,	NULL},
	{"275",		0x10,	101,	mode1,	0,	NULL},
	{"375",		0x12,	101,	mode1,	0,	NULL},
	{"471",		0x22,	32,	mode1,	0,	NULL},
	{"475",		0x14,	101,	mode1,	0,	NULL},
	{"575",		0x16,	101,	mode1,	0,	NULL},
	{"725",		0x28,	26,	mode1,	0,	NULL},
	{"726",		0x30,	26,	mode1,	0,	NULL},
	{"735",		0x04,	12,	mode1,	F_735,	NULL},
	{"751",		0x1c,	32,	mode1,	0,	NULL},
	{"761",		0x1e,	32,	mode1,	0,	NULL},
	{"765",		0x2c,	32,	mode1,	0,	NULL},
	{"775",		0x46,	101,	mode1,	0,	NULL},
	{"781",		0x26,	101,	mode1,	0,	NULL},
	{"970",		0x2e,	101,	mode1,	0,	NULL},
	{"R7000",	0x08,	99,	mode2,	0,	NULL},
	{"R71",		0x1A,	32,	mode1,	0,	NULL},
	{"R7100",	0x34,	99,	mode2,	0,	NULL},
	{"R72",		0x32,	101,	mode1,	0,	NULL},
	{"R8500",	0x4a,	1000,	mode3,	0,	NULL},
	{"R9000",	0x2a,	99,	mode3,	0,	NULL},
	{"\0",		0x0,	0,	0,	0,	NULL}  /* end */
};
