/* xpm.c
 * 
 * Copyright (C) 1998-2001  Oskar Liljeblad
 *
 * This file is part of icoutils.
 *
 * This software is copyrighted work licensed under the terms of the
 * GNU General Public License. Please consult the file "LICENSE" for
 * details.
 */

#include "icotool.h"

#include <stdio.h>
#include <string.h>
#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif

/* We don't free this variable, since it is used throughout the program. */
static int *colortable = NULL;
static int table_size = 0;

/*
 *
 * Function declarations
 *
 */
 
static char *make_xpm_color_id (int, int);
static void print_xpm_colors (IconFile *, IconImage *, FILE *, int, int);

/*
 *
 * Macros
 *
 */

#define XPM_COLOR_IDS " .+@#$%&*=-;>,')!~{]^/(:<[}|1234567890abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ`"
#define XPM_ID_COUNT (sizeof(XPM_COLOR_IDS)/sizeof(char)-1)

/*
 *
 * Functions
 *
 */

static char *
make_xpm_color_id (int color, int chars)
{
    int maxv, c;
    static char name[5];		/* enough for about 68M colors :) */

    maxv = 1;
    for (c = 0 ; c < chars-1 ; c++)
		maxv *= XPM_ID_COUNT;
    for (c = 0 ; maxv >= 1 && c < 4 ; c++) {
		name[c] = XPM_COLOR_IDS[color/maxv];
		color = color % maxv;
		maxv /= XPM_ID_COUNT;
    }
    name[c] = '\0';

    return name;
}

static void
print_xpm_colors(IconFile *fi, IconImage *icon, FILE *out,
		int used_colors, int color_chars)
{
	int c, base;

	base = (image_mode == IMAGEMODE_BOTH ? 1 : 0);
	if (icon->color_count == 1<<24) {
		for (c = 0; c < used_colors - base; c++) {
			fprintf(out, "\"%s\tc #%02x%02x%02x\",\n",
					make_xpm_color_id(c + base, color_chars),
					(colortable[c]) & 0xFF,
					(colortable[c] >> 8) & 0xFF,
					(colortable[c] >> 16) & 0xFF);
		}
	} else {
		for (c = 0 ; c < used_colors - base ; c++) {
			fprintf(out, "\"%s\tc #%02x%02x%02x\",\n",
					make_xpm_color_id(c + base, color_chars),
					icon->rgb_data[c].red,
					icon->rgb_data[c].green,
					icon->rgb_data[c].blue);
		}
	}
}

void
make_xpm_image (IconFile *fi, IconImage *icon)
{
    FILE *out;
    char *outname;
    int used_colors, color_chars, c, d, e;

    /* Get name of file we are going to write to. */
    outname = get_destination_name(fi, icon->entry->width,
    	    icon->entry->height, icon->color_count);
    if (outname == NULL) {
		out = stdout;
    } else {
		out = fopen(outname, "wb");
		if (out == NULL) {
	    	put_error("%s: %s\n", outname, strerror(errno));
	    	return;
		}
    }

   	/* Calculate number of colors used in image. */
   	if (image_mode == IMAGEMODE_MASK) {
		used_colors = 2;
    } else if (icon->color_count == 1<<24) {
		/* Count number of colors used. This is horribly slow, */
		/* but don't want to drag in code for hash tables or */
		/* trees. */
		used_colors = 0;

		for (d = 0; d < icon->entry->height; d++) {
	    	for (c = 0; c < icon->entry->width; c++) {
				int color;

				color = get_bitmap_image_pixel(icon, c, d);
				for (e = 0; e < used_colors; e++) {
					if (colortable[e] == color)
						break;
				}
				if (e == used_colors) {
					if (used_colors == table_size) {
						table_size = (table_size == 0 ? 1024 : table_size * 2);
						colortable = xrealloc(colortable, sizeof(int)*table_size);
					}
					colortable[used_colors++] = color;
				}
			}
		}
		used_colors++;
	} else {
		/* One extra for background color */
    	used_colors = icon->color_count + 1;
	}

    /* find out how many characters per color */
    c = sizeof(XPM_COLOR_IDS)/sizeof(char) - 1;
    for (color_chars = 1; used_colors > c && c < c*90; color_chars++)
		c *= sizeof(XPM_COLOR_IDS)/sizeof(char) - 1;

    /* Write XPM header */
    fprintf(out, "/* XPM */\n");
    fprintf(out, "static char * icotool_xpm[] = {\n");
    fprintf(out, "/* width height ncolors chars_per_pixel */\n");
    fprintf(out, "\"%d %d %d %d\",\n", icon->entry->width,
    	    icon->entry->height, used_colors, color_chars);
    fprintf(out, "/* colors */\n");

    /* write colors */
    switch (image_mode) {
    case IMAGEMODE_MASK:
		fprintf(out, "\"%s\ts None\tc None\",\n", make_xpm_color_id(0, color_chars));
		fprintf(out, "\"%s\tc Black\",\n", make_xpm_color_id(1, color_chars));
		break;
    case IMAGEMODE_BOTH:
		fprintf(out, "\"%s\ts None\tc None\",\n", make_xpm_color_id(0, color_chars));
		print_xpm_colors(fi, icon, out, used_colors, color_chars);
		break;
    case IMAGEMODE_IMAGE:
		print_xpm_colors(fi, icon, out, used_colors, color_chars);
		break;
    }

    /* check hotspot */
    if (fi->resource_type == RESOURCE_CURSOR)
		put_message(1, "hotspot x=%d y=%d\n", icon->entry->hotspot_x, icon->entry->hotspot_y);

    /* write pixel data */
    fprintf(out, "/* pixels */\n");
    for (d = 0; d < icon->entry->height; d++) {
		if (d != 0)
			fprintf(out, "\",\n");
		fprintf(out, "\"");
		for (c = 0; c < icon->entry->width; c++) {
			int color = 0;

    		switch (image_mode) {
			case IMAGEMODE_MASK:
		    	color = get_bitmap_mask_pixel(icon, c, d);
		    	break;
			case IMAGEMODE_IMAGE:
		    	color = get_bitmap_image_pixel(icon, c, d);
				if (icon->color_count == 1<<24) {
					for (e = 0; e < used_colors; e++) {
						if (colortable[e] == color) {
							color = e;
							break;
						}
					}
				}
		    	break;
			case IMAGEMODE_BOTH:
				if (get_bitmap_mask_pixel(icon, c, d) != 0) {
					color = 0;
				} else if (icon->color_count == 1<<24) {
					color = get_bitmap_image_pixel(icon, c, d);
					for (e = 0; e < used_colors; e++) {
						if (colortable[e] == color) {
							color = e + 1;
							break;
						}
					}
				} else {
		    		color = get_bitmap_image_pixel(icon, c, d)+1;
				}
			}

			fprintf(out, "%s", make_xpm_color_id(color, color_chars));
	    }
    }
    fprintf(out, "\"};\n");
}
