/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iuniformgridfileloader.h"


#include "icalculator.h"
#include "icommoneventobservers.h"
#include "idatalimits.h"
#include "idatasubject.h"
#include "ierrorstatus.h"
#include "ifile.h"
#include "iparallel.h"
#include "iparallelmanager.h"
#include "iparallelworker.h"
#include "iuniformgriddata.h"
#include "iviewmodule.h"

#include <vtkFloatArray.h>
#include <vtkInformation.h>
#include <vtkPointData.h>

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"
#include "ibuffertemplate.h"
#include "icalculatortemplate.h"


using namespace iParameter;


//
//  Helper classes for parallel executions
//
class iUniformGridHelper : protected iParallelWorker
{

public:

	iUniformGridHelper(iUniformGridFileLoader *loader);

	void ShiftData(int numCom, int dim, int dn, int dataDims[3], float *data, float *buf);
	void ExpandUp(float *array, int nc, int oldDims[3], int newDims[3]);
	void CollapseDown(float *array, int nc, int oldDims[3], int newDims[3]);
	void PadPeriodically(float *array, int nc, int oldDims[3], int newDims[3]);

protected:

	virtual int ExecuteStep(int step, iParallel::ProcessorInfo &p);

	void ExecuteShiftData(iParallel::ProcessorInfo &p);
	void ExecutePadPeriodically(iParallel::ProcessorInfo &p);

	iUniformGridFileLoader *mLoader;

	int mNumCom;
	int *mDims;
	float *mArr1, *mArr2;

	float mProgStart, mProgStep;

	int mItmp1, mItmp2;
};


class iUniformScalarsHelper : protected iParallelWorker
{

public:

	iUniformScalarsHelper(iUniformScalarsFileLoader *loader);

	void OperateOnData1(int nvar, int dims[3], const iString &calculatorExpression, bool doVector, float ps, float dp);
	void OperateOnData2(int nvar, int dims[3], float *fMin, float *fMax, bool &overflow, float ps, float dp);

protected:

	virtual int ExecuteStep(int step, iParallel::ProcessorInfo &p);

	void ExecuteOperateOnData1(iParallel::ProcessorInfo &p);
	void ExecuteOperateOnData2(iParallel::ProcessorInfo &p);

	iUniformScalarsFileLoader *mLoader;
	iProgressEventObserver *mObserver;

	iString mCalculatorExpression;

	int mNumVars, mNumProcs, *mDims;

	iBuffer<float> mMin, mMax;
	float mProgStart, mProgStep;

	bool mDoVector, mOverflow;
};


//
//  Main classes
//
//  ********************************************************************
//
//  Generic GridFileLoader class
//
//  ********************************************************************
//
iUniformGridFileLoader::iUniformGridFileLoader(iDataReader *r, int priority, bool usePeriodicOffsets) : iFieldFileLoader(r,priority), mMaxDimension(1670000), mBufferSizeOffset(1), mUsePeriodicOffsets(usePeriodicOffsets)
{
	mNumVars = mNumComponents = 0;

	mOriginOffset[0] = mOriginOffset[1] = mOriginOffset[2] = 0.0;

	mVoxelLocation = VoxelLocation::Vertex;

	mFileDims[0] = mFileDims[1] = mFileDims[2] = 0;
	mDataDims[0] = mDataDims[1] = mDataDims[2] = 0;

	mBufferSize = 0;
	mBuffer = 0;

	mScaledDim = -1;

	mHelper = new iUniformGridHelper(this); IERROR_ASSERT(mHelper);
}


iUniformGridFileLoader::~iUniformGridFileLoader()
{
	delete mHelper;
}


void iUniformGridFileLoader::ReadFileBody(const iString &suffix, const iString &fname)
{
	//
	//  is the suffix valid?
	//
	if(suffix.Lower()!="txt" && suffix.Lower()!="bin")
	{
		this->GetErrorStatus()->Set("Incorrect file suffix.");
		return;
	}

	//
	//  Read the data
	//
	if(suffix.Lower() == "txt") this->ReadTxtFile(fname); else this->ReadBinFile(fname);
}


void iUniformGridFileLoader::ReadBinFile(const iString &fname)
{
	int i;
	vtkIdType l, ntot;
	float vmin, vmax, *d;
	iFile F(fname);
	bool err;

	if(!F.Open(iFile::_Read,iFile::_Binary))
	{
		this->GetErrorStatus()->Set("File does not exist.");
		return;
	}

	mObserver->Started(iProgressEventObserver::_Reading);
	mObserver->SetProgress(0.0);

	//
	// auto-detect data endiness
	//
	if(!this->DetectFortranFileStructure(F,3*sizeof(int)))
	{ 
		F.Close(); 
		mObserver->Finished();
		this->GetErrorStatus()->Set("Corrupted header.");
		return;
	}

	//
	//  Read the header
	//
	err = false;
	int ndim[3];
	if(!this->ReadFortranRecord(F,Buffer(ndim,3),0.0,0.0)) err = true;

    if(ndim[0]<1 || ndim[0]>mMaxDimension || ndim[1]<1 || ndim[1]>mMaxDimension || ndim[2]<1 || ndim[2]>mMaxDimension) err = true;

	if(err) 
	{ 
		F.Close(); 
		mObserver->Finished();
		this->GetErrorStatus()->Set("Corrupted header.");
		return;
	}

	ntot = (vtkIdType)ndim[0]*ndim[1]*ndim[2];
	//
	//  Measure the file size by skipping blocks with size of one variable
	//
	int nvar, nrec;
	int mar = F.SetMarker();
	for(nrec=0; nrec<999 && this->SkipFortranRecord(F,sizeof(float)*ntot); nrec++);
	F.ReturnToMarker(mar,true);

	nvar = nrec;
	this->SetIndicies(nvar,nrec);
	//
	//  If no records, exit
	//
	if(nvar < 1)
	{ 
		F.Close(); 
		mObserver->Finished(); 
		this->GetErrorStatus()->Set("Corrupted data.");
		return; 
	}
	mObserver->SetProgress(0.01);
	//
	//  parameters for the Progress Bar
	//
	float prog1 = 0.99/nrec;
	//
	//  Allocate memory; erase the old mBuffer if needed.
	//
	vtkIdType newBufferSize = (vtkIdType)(ndim[0]+mBufferSizeOffset)*(vtkIdType)(ndim[1]+mBufferSizeOffset)*(vtkIdType)(ndim[2]+mBufferSizeOffset)*nvar;
	if(newBufferSize>mBufferSize || newBufferSize<mBufferSize/2) // do not use too much extra memory
	{
		if(mBuffer != 0) delete [] mBuffer;
		mBuffer = new float[newBufferSize];
		mBufferSize = newBufferSize;
		if(mBuffer == 0)
		{
			F.Close();
			mObserver->Finished();
			this->GetErrorStatus()->Set("Not enough memory to load the data.");
			return;
		}
	}
	//
	//  Read the data from the file.
	//
	d = new float[ntot]; 
	if(d == 0)
	{
		this->EraseBuffer();
		F.Close();
		mObserver->Finished();
		this->GetErrorStatus()->Set("Not enough memory to load the data.");
		return;
	}

	for(i=0; !err && i<nrec; i++) 
	{
		err = !this->ReadFortranRecord(F,Buffer(d,ntot),0.01+prog1*i,prog1);
		if(err || mObserver->IsAborted()) break;

		this->AssignBinData(nvar,ntot,i,d);
		vmin = vmax = d[0];
		for(l=1; l<ntot; l++)
		{
			if(vmin > d[l]) vmin = d[l];
			if(vmax < d[l]) vmax = d[l];
			mVarMin[i] = vmin;
			mVarMax[i] = vmax;
		}
	}

	delete [] d;

	mObserver->SetProgress(1.0);
	mObserver->Finished();
	F.Close();

	if(err || mObserver->IsAborted())
	{
		this->EraseBuffer();
		if(err) this->GetErrorStatus()->Set("Corrupted data."); else this->GetErrorStatus()->SetAbort();
		return;
	}

	mNumVars = nvar;
	for(i=0; i<3; i++) mFileDims[i] = ndim[i];
}


void iUniformGridFileLoader::ReadTxtFile(const iString &fname)
{
	int i, n1, n2, n3, ret;
	iString s;
	vtkIdType ntot, l;
	float f[999];
	iFile F(fname);
	bool err;

	if(!F.Open(iFile::_Read,iFile::_Text))
	{
		this->GetErrorStatus()->Set("File does not exist.");
		return;
	}

	mObserver->Started(iProgressEventObserver::_Reading);
	mObserver->SetProgress(0.0);
	//
	//  Read the header
	//
	err = false;
	if(!F.ReadLine(s)) err = true;

	ret = sscanf(s.ToCharPointer(),"%d %d %d",&n1,&n2,&n3);
	if(ret != 3) err = true;

	if(n1<1 || n1>mMaxDimension || n2<1 || n2>mMaxDimension || n3<1 || n3>mMaxDimension) err = true;

	if(err)
	{ 
		F.Close();
		mObserver->Finished();
		this->GetErrorStatus()->Set("Corrupted header.");
		return;
	}

	ntot = (vtkIdType)n1*n2*n3;
	//
	//  Find out the number of variables
	//
	if(!F.ReadLine(s)) err = true;
	ret = sscanf(s.ToCharPointer(),"%g %g %g %g %g %g %g %g %g %g",&f[0],&f[1],&f[2],&f[3],&f[4],&f[5],&f[6],&f[7],&f[8],&f[9]);
	if(ret<1 || ret>10) err = true;
	int nvar = 0, nrec = ret;

	nvar = nrec;
	this->SetIndicies(nvar,nrec);
	//
	//  If no records or an error, exit
	//
	if(nvar<1 || err)
	{ 
		F.Close();
		mObserver->Finished();
		this->GetErrorStatus()->Set("Corrupted data.");
		return;
	}

	mObserver->SetProgress(0.01);
	//
	//  Allocate memory; erase the old mBuffer if needed.
	//
	vtkIdType newBufferSize = (vtkIdType)(n1+mBufferSizeOffset)*(vtkIdType)(n2+mBufferSizeOffset)*(vtkIdType)(n3+mBufferSizeOffset)*nvar;
	if(newBufferSize>mBufferSize || newBufferSize<mBufferSize/2) // do not use too much extra memory
	{
		if(mBuffer != 0) delete [] mBuffer;
		mBuffer = new float[newBufferSize];
		mBufferSize = newBufferSize;
		if(mBuffer == 0)
		{
			F.Close();
			mObserver->Finished();
			this->GetErrorStatus()->Set("Not enough memory to create the data.");
			return;
		}
	}
	//
	//  Save the first line
	//
	this->AssignTxtData(nvar,ntot,0,f);
	for(i=0; i<nvar; i++)
	{
		mVarMin[i] = f[i];
		mVarMax[i] = f[i];
	}

	//
	//  Read the data from the file.
	//
	for(l=1; !err && l<ntot; l++) 
	{
		if(!F.ReadLine(s)) err = true;
		ret = sscanf(s.ToCharPointer(),"%g %g %g %g %g %g %g %g %g %g",&f[0],&f[1],&f[2],&f[3],&f[4],&f[5],&f[6],&f[7],&f[8],&f[9]);
		if(ret != nrec) err = true;
		if((100*l)/ntot < (100*(l+1))/ntot) 
		{
			mObserver->SetProgress(0.01+(float)l/ntot);
			if(mObserver->IsAborted())
			{
				F.Close();
				mObserver->Finished();
				this->GetErrorStatus()->SetAbort();
				return;
			}
		}

		this->AssignTxtData(nvar,ntot,l,f);
		for(i=0; i<nvar; i++)
		{
			if(mVarMin[i] > f[i]) mVarMin[i] = f[i];
			if(mVarMax[i] < f[i]) mVarMax[i] = f[i];
		}
	}	

	mObserver->SetProgress(1.0);
	mObserver->Finished();
	F.Close();

	if(err || mObserver->IsAborted())
	{
		this->EraseBuffer();
		if(err) this->GetErrorStatus()->Set("Corrupted data."); else this->GetErrorStatus()->SetAbort();
		return;
	}

	mNumVars = nvar;
	mFileDims[0] = n1;
	mFileDims[1] = n2;
	mFileDims[2] = n3;
}


void iUniformGridFileLoader::EraseBuffer()
{
	delete [] mBuffer; 
	mBuffer = 0; 
	mBufferSize = 0;
}


void iUniformGridFileLoader::FinalizeBody()
{
	int i;

	//
	//  Do not do anything if there is no data
	//
	if(mBuffer!=0 && mFileDims[0]>0 && mFileDims[1]>0 && mFileDims[2]>0)
	{
		//
		//  Configure Limits and components
		//
		mNumComponents = 0;
		this->GetStream(0)->Subject->GetLimits()->ResetVars();
		this->SetArraysAndComponents();
#ifdef I_CHECK1
		if(mNumVars != mNumComponents)
		{
			IERROR_LOW("Bug in iUniformGridFileLoader.");
		}
#endif
		
		this->OperateOnData();

		//
		//  Specify data dimensions
		//
		for(i=0; i<3; i++) mDataDims[i] = mFileDims[i];

		//
		//  Implement periodic BC
		//
		if(mUsePeriodicOffsets && mVoxelLocation==VoxelLocation::Center && this->IsBoxPeriodic())
		{
			//
			//  Expand the box and add extra values.
			//
			for(i=0; i<3; i++) if(this->IsDirectionPeriodic(i)) mDataDims[i]++;
			mHelper->ExpandUp(mBuffer,mNumComponents,mFileDims,mDataDims);
			mHelper->PadPeriodically(mBuffer,mNumComponents,mFileDims,mDataDims);
		}
		
		this->ReleaseData();
	}
}


void iUniformGridFileLoader::SetVoxelLocation(int m)
{ 
	if(VoxelLocation::IsValid(m) && mVoxelLocation!=m)
	{
		int oldLoc = mVoxelLocation;
		mVoxelLocation = m;

		if(this->IsThereData())
		{
			if(mUsePeriodicOffsets && this->IsBoxPeriodic())
			{
				int i;
				if(oldLoc==VoxelLocation::Vertex && mVoxelLocation==VoxelLocation::Center)
				{
					for(i=0; i<3; i++) if(this->IsDirectionPeriodic(i)) mDataDims[i] = mFileDims[i] + 1;
					mHelper->ExpandUp(mBuffer,mNumComponents,mFileDims,mDataDims);
					mHelper->PadPeriodically(mBuffer,mNumComponents,mFileDims,mDataDims);
				}
				if(oldLoc==VoxelLocation::Center && mVoxelLocation==VoxelLocation::Vertex)
				{
					mHelper->CollapseDown(mBuffer,mNumComponents,mDataDims,mFileDims);
					for(i=0; i<3; i++) mDataDims[i] = mFileDims[i];
				}
			}
			this->ReleaseData();
			this->NotifyDependencies();
		}
	}
}


void iUniformGridFileLoader::ReleaseData()
{
	double org[3], spa[3];
	this->ComputeSpacing(org,spa);

	iUniformGridData *data = iUniformGridData::New(); IERROR_ASSERT(data);
	data->SetDimensions(mDataDims);
	data->SetOrigin(org);
	data->SetSpacing(spa);

	data->SetInternalProperties(mVoxelLocation,mFileDims,mDataDims);

	this->AttachBuffer(data);
	this->AttachDataToStream(0,data);
		
	data->Delete();
}


void iUniformGridFileLoader::Polish(vtkDataSet *ds)
{
	//
	//  Extra class needed under VTK 5. What a mess!
	//
	vtkImageData *d = vtkImageData::SafeDownCast(ds);
	if(d != 0)
	{
		vtkInformation *info = d->GetPipelineInformation();
		if(info == 0)
		{
			info = vtkInformation::New(); IERROR_ASSERT(info);
			d->SetPipelineInformation(info);
		}

		double spa[3], org[3];
		d->GetOrigin(org);
		d->GetSpacing(spa);

		info->Set(vtkDataObject::ORIGIN(),org,3);
		info->Set(vtkDataObject::SPACING(),spa,3);
	}
}


void iUniformGridFileLoader::SetScaledDimension(int v)
{
	if(v>=-2 && v<=2)
	{
		mScaledDim = v;

		if(!this->IsBoxPeriodic() && this->IsThereData())
		{
			//
			//  Reset spacing
			//
			vtkStructuredPoints *data = iRequiredCast<vtkStructuredPoints>(INFO,this->GetStream(0)->ReleasedData);

			double org[3], spa[3];
			this->ComputeSpacing(org,spa);
			data->SetOrigin(org);
			data->SetSpacing(spa);
			this->ShiftDataBody(data,mShift);
		}
	}
}


void iUniformGridFileLoader::ComputeSpacing(double org[3], double spa[3])
{
	int i;

	if(mFileDims[0]<=0 || mFileDims[1]<=0 || mFileDims[2]<=0) return;

	//
	//  Now we need to decide whether the data are CellData or PointData. We use voxelLocation for that: 
	//  if it is 0, we have point data and we place the first point into (-1,-1,-1) corner;
	//  if it is 1, we have cell data and we place the first point into the center of the first cell (mOriginOffset).
	//
	float shift;
	if(mVoxelLocation == VoxelLocation::Vertex)
	{
		shift = 0.0;
	}
	else
	{
		shift = 0.5;
	}
	for(i=0; i<3; i++) mOriginOffset[i] = shift*2.0/mFileDims[i];

	//
	//  Find longest and short dimensions
	//
	int nmax, nmin;
	int imax = 0, imin = 0;
	nmax = nmin = mFileDims[0];
	for(i=1; i<3; i++)
	{
		if(nmax < mFileDims[i]) 
		{
			nmax = mFileDims[i];
			imax = i;
		}
		if(mFileDims[i]>1 && nmin>mFileDims[i]) 
		{
			nmin = mFileDims[i];
			imin = i;
		}
	}

	for(i=0; i<3; i++) this->SetDirectionPeriodic(i,mFileDims[i]==nmax);

	if(this->IsBoxPeriodic())
	{
		i = imax;
	}
	else
	{
		switch (mScaledDim)
		{
		case -1:
			{
				i = imin;
				break;
			}
		case  0:
		case  1:
		case  2:
			{
				if(mFileDims[mScaledDim] > 1) i = mScaledDim; else i = imin;
				break;
			}
		case -2: 
		default:
			{ 
				i = imax;
				break;
			}
		}
	}

	double s = (2.0-2*mOriginOffset[i])/(mFileDims[i]-1);

	for(i=0; i<3; i++) 
	{	
		org[i] = -1.0 + shift*s;
		spa[i] = s;
	}
}


void iUniformGridFileLoader::ShiftDataBody(vtkDataSet *dataIn, double dx[3])
{
	vtkStructuredPoints *data = vtkStructuredPoints::SafeDownCast(dataIn);
	if(data==0 || !this->IsThereData(0)) return;

	int d, dn;
	for(d=0; d<3; d++)
	{
		dn = round(dx[d]*mDataDims[d]) % mDataDims[d];
		if(this->IsDirectionPeriodic(d) && dn!=0)
		{
			vtkIdType size = (vtkIdType)mDataDims[0]*mDataDims[1]*mDataDims[2]*mNumComponents;
			float *buf = new float[mNumComponents*mDataDims[d]]; if(buf == 0) return;
			if(data->GetPointData()!=0 && data->GetPointData()->GetScalars()!=0)
			{
				mHelper->ShiftData(mNumComponents,d,dn,mDataDims,(float *)data->GetPointData()->GetScalars()->GetVoidPointer(0),buf);
			}
			if(data->GetPointData()!=0 && data->GetPointData()->GetVectors()!=0)
			{
				mHelper->ShiftData(mNumComponents,d,dn,mDataDims,(float *)data->GetPointData()->GetVectors()->GetVoidPointer(0),buf);
			}
			if(data->GetPointData()!=0 && data->GetPointData()->GetTensors()!=0)
			{
				mHelper->ShiftData(mNumComponents,d,dn,mDataDims,(float *)data->GetPointData()->GetTensors()->GetVoidPointer(0),buf);
			}
			delete[] buf;

			double org[3];
			data->GetOrigin(org);
			org[d] += 2.0*(dx[d]-float(dn)/mDataDims[d]);
			data->SetOrigin(org);
		}
		else
		{
			double org[3];
			data->GetOrigin(org);
			org[d] += 2.0*dx[d];
			data->SetOrigin(org);
		}
	}

	data->Modified();
}


bool iUniformGridFileLoader::IsCompatible(iUniformGridFileLoader *other) const
{
	return (this->IsThereData() && other!=0 && other->IsThereData() && mVoxelLocation==other->mVoxelLocation && mFileDims[0]==other->mFileDims[0] && mFileDims[1]==other->mFileDims[1] && mFileDims[2]==other->mFileDims[2]);
}


//
//  ********************************************************************
//
//  iUniformVectorsFileLoader class
//
//  ********************************************************************
//
iUniformVectorsFileLoader::iUniformVectorsFileLoader(iDataReader *r) : iUniformGridFileLoader(r,100,false)
{
}


void iUniformVectorsFileLoader::SetIndicies(int &nvar, int &ncomp)
{
	nvar = 3;
	ncomp = 3;
}


void iUniformVectorsFileLoader::AssignBinData(int, vtkIdType ntot, int com, float *d)
{
	vtkIdType l;
	for(l=0; l<ntot; l++) mBuffer[com+3*l] = d[l];
}


void iUniformVectorsFileLoader::AssignTxtData(int, vtkIdType, vtkIdType ind, float *f)
{
	int n;
	for(n=0; n<3; n++) mBuffer[n+3*ind] = f[n];
}


void iUniformVectorsFileLoader::SetArraysAndComponents()
{
	mNumComponents = 3;
	this->GetStream(0)->Subject->GetLimits()->AddVar(0);
	if(!this->GetStream(0)->Subject->GetFixedLimits())
	{
		this->GetStream(0)->Subject->GetLimits()->SetMin(0,0.0f);
		this->GetStream(0)->Subject->GetLimits()->SetMax(0,sqrt(mVarMax[0]*mVarMax[0]+mVarMax[1]*mVarMax[1]+mVarMax[2]*mVarMax[2]));
	}
}


void iUniformVectorsFileLoader::AttachBuffer(vtkStructuredPoints *data)
{
	//
	//  Attach the actual data
	//
	vtkIdType size = (vtkIdType)mDataDims[0]*mDataDims[1]*mDataDims[2]*3;
	vtkFloatArray *array = vtkFloatArray::New(); IERROR_ASSERT(array);
	array->SetNumberOfComponents(3);
	array->SetArray(mBuffer,size,1);
	array->SetName("Vectors");
	data->GetPointData()->SetVectors(array);
	array->Delete();
}


//
//  ********************************************************************
//
//  iUniformTensorsFileLoader class
//
//  ********************************************************************
//
iUniformTensorsFileLoader::iUniformTensorsFileLoader(iDataReader *r) : iUniformGridFileLoader(r,200,false)
{
}


void iUniformTensorsFileLoader::SetIndicies(int &nvar, int &ncomp)
{
	nvar = 9;
	ncomp = 6;
}


void iUniformTensorsFileLoader::AssignBinData(int, vtkIdType ntot, int com, float *d)
{
	int noff1, noff2;
	vtkIdType l;

	switch(com)
	{
	case 0: { noff1 = 0; noff2 = -1; break; }
	case 1: { noff1 = 1; noff2 = 3; break; }
	case 2: { noff1 = 2; noff2 = 6; break; }
	case 3: { noff1 = 4; noff2 = -1; break; }
	case 4: { noff1 = 5; noff2 = 7; break; }
	case 5: { noff1 = 8; noff2 = -1; break; }
	default: 
		{
			noff1 = 0; noff2 = -1;
			IERROR_LOW("Bug detected.");
		}
	}

	for(l=0; l<ntot; l++) mBuffer[noff1+9*l] = d[l];
	if(noff2 > 0) for(l=0; l<ntot; l++) mBuffer[noff2+9*l] = d[l];
}


void iUniformTensorsFileLoader::AssignTxtData(int, vtkIdType, vtkIdType ind, float *f)
{
	//
	//  Fill in 3x3 tensor with 6 components
	//
	mBuffer[0+9*ind] = f[0];
	mBuffer[1+9*ind] = f[1];
	mBuffer[2+9*ind] = f[2];
	mBuffer[3+9*ind] = f[1];
	mBuffer[4+9*ind] = f[3];
	mBuffer[5+9*ind] = f[4];
	mBuffer[6+9*ind] = f[2];
	mBuffer[7+9*ind] = f[4];
	mBuffer[8+9*ind] = f[5];
}


void iUniformTensorsFileLoader::SetArraysAndComponents()
{
	mNumComponents = 9;
	this->GetStream(0)->Subject->GetLimits()->AddVar(0);
	if(this->GetStream(0)->Subject->GetFixedLimits())
	{
		this->GetStream(0)->Subject->GetLimits()->SetMin(0,0.0f);
		this->GetStream(0)->Subject->GetLimits()->SetMax(0,sqrt(mVarMax[0]*mVarMax[0]+mVarMax[1]*mVarMax[1]+mVarMax[2]*mVarMax[2]+mVarMax[3]*mVarMax[3]+mVarMax[4]*mVarMax[4]+mVarMax[5]*mVarMax[5]+mVarMax[6]*mVarMax[6]+mVarMax[7]*mVarMax[7]+mVarMax[8]*mVarMax[8]));
	}
}


void iUniformTensorsFileLoader::AttachBuffer(vtkStructuredPoints *data)
{
	//
	//  Attach the actual data
	//
	vtkIdType size = (vtkIdType)mDataDims[0]*mDataDims[1]*mDataDims[2]*9;
	vtkFloatArray *array = vtkFloatArray::New(); IERROR_ASSERT(array);
	array->SetNumberOfComponents(9);
	array->SetArray(mBuffer,size,1);
	array->SetName("Tensors");
	data->GetPointData()->SetTensors(array);
	array->Delete();
}


//
//  ********************************************************************
//
//  iUniformScalarsFileLoader class
//
//  ********************************************************************
//
iUniformScalarsFileLoader::iUniformScalarsFileLoader(iDataReader *r, iUniformVectorsFileLoader *vs) : iUniformGridFileLoader(r,0,true)
{
	mVectorFieldLoader = vs;

	mCalculatorOutput = 0;
//	mCalculatorExpression = "";

	mHelper2 = new iUniformScalarsHelper(this); IERROR_ASSERT(mHelper2);
}


iUniformScalarsFileLoader::~iUniformScalarsFileLoader()
{
	delete mHelper2;
}


void iUniformScalarsFileLoader::SetIndicies(int &nvar, int &ncomp)
{
	//
	//  Try to set all the records from the file. DataLimits will either expand to accommodate 
	//  all of them or limit the allowed number to the number of listed records.
	//
	this->GetStream(0)->Subject->GetLimits()->BlockNotifications(true);
	this->GetStream(0)->Subject->GetLimits()->AssignVars(nvar);
	this->GetStream(0)->Subject->GetLimits()->BlockNotifications(false);
	ncomp = nvar = this->GetStream(0)->Subject->GetLimits()->GetNumVars();
}


void iUniformScalarsFileLoader::AssignBinData(int nvar, vtkIdType ntot, int com, float *d)
{
	vtkIdType l;
	for(l=0; l<ntot; l++) mBuffer[com+nvar*l] = d[l];
}


void iUniformScalarsFileLoader::AssignTxtData(int nvar, vtkIdType ntot, vtkIdType ind, float *f)
{
	int n;
	for(n=0; n<nvar; n++) mBuffer[n+nvar*ind] = f[n];
}


void iUniformScalarsFileLoader::AttachBuffer(vtkStructuredPoints *data)
{
	//
	//  Attach the actual data
	//
	vtkIdType size = (vtkIdType)mDataDims[0]*mDataDims[1]*mDataDims[2]*mNumVars;
	vtkFloatArray *array = vtkFloatArray::New(); IERROR_ASSERT(array);
	array->SetNumberOfComponents(mNumVars);
	array->SetArray(mBuffer,size,1);
	array->SetName("Scalars");
	data->GetPointData()->SetScalars(array);
	array->Delete();

	data->SetScalarTypeToFloat();
	data->SetNumberOfScalarComponents(mNumVars);
}


void iUniformScalarsFileLoader::SetArraysAndComponents()
{
	mNumComponents = mNumVars;
	this->GetStream(0)->Subject->GetLimits()->AssignVars(mNumVars);
	//
	//  Limits are set after the data are operated upon
	//
}


void iUniformScalarsFileLoader::SetCalculatorOutput(int n)
{
	if(n>=0 && n<this->GetStream(0)->Subject->GetLimits()->GetNumVars()) 
	{
		mCalculatorOutput = n;
	}
}


void iUniformScalarsFileLoader::SetCalculatorExpression(const iString &s)
{
	mCalculatorExpression = s;
}


void iUniformScalarsFileLoader::OperateOnData()
{
	int n;
	float fMin = iMath::_LargeFloat, fMax = -iMath::_LargeFloat, f = 0.0;

	iDataLimits *slim = this->GetStream(0)->Subject->GetLimits();
	iDataLimits *vlim = mVectorFieldLoader->GetSubject(0)->GetLimits();
	slim->BlockNotifications(true);
	int nvar = slim->GetNumVars();

	//
	//  Do the array function first
	//
	if(mCalculatorOutput>=0 && mCalculatorOutput<nvar && !mCalculatorExpression.IsEmpty())
	{
		this->GetObserver()->Started(iProgressEventObserver::_Operating);
		this->GetObserver()->SetProgress(0.0);

		bool doVector = false;
		if(this->IsCompatible(mVectorFieldLoader)) doVector = true;

		mHelper2->OperateOnData1(nvar,mFileDims,mCalculatorExpression,doVector,0.0,1.0);
	}

	this->GetObserver()->Started(iProgressEventObserver::_Formatting);
	this->GetObserver()->SetProgress(0.0);

	//
	//  Format everything in parallel
	//
	mVarMin.Extend(nvar);
	mVarMax.Extend(nvar);
	mHelper2->OperateOnData2(nvar,mFileDims,mVarMin,mVarMax,mOverflow,0.0,1.0);

#ifdef I_CHECK2
	//
	//  Format everything serially
	//
	vtkIdType lMin, lMax, size = (vtkIdType)mFileDims[0]*mFileDims[1]*mFileDims[2];
	for(n=0; n<nvar; n++)
	{
		fMin = iMath::_LargeFloat;
		fMax = -iMath::_LargeFloat;
		vtkIdType l;
		for(l=0; l<size; l++)
		{
			f = mBuffer[n+nvar*l];
			if(f > fMax)
			{
				lMax = l;
				fMax = f;
			}
			if(f < fMin)
			{
				lMin = l;
				fMin = f;
			}
		}
		if(fabs(mVarMin[n]-fMin)>iMath::_FloatRes || fabs(mVarMax[n]-fMax)>iMath::_FloatRes)
		{
			IERROR_REPORT_BUG;
		}
	}
#endif

	this->GetObserver()->SetProgress(1.0);
	this->GetObserver()->Finished();

	//
	//  Correct limits in log stretching
	//
	if(!this->GetStream(0)->Subject->GetFixedLimits())
	{
		for(n=0; n<nvar; n++)
		{
			fMin = mVarMin[n];
			fMax = mVarMax[n];
			if(slim->GetStretch(n) == 1)
			{
				fMin = 0.1*floor(10.0*fMin);
				fMax = 0.1*floor(1+10.0*fMax);
			}
			slim->SetMin(n,fMin);
			slim->SetMax(n,fMax);
		}
	}

	slim->BlockNotifications(false);
}


//
//  Helper class
//
iUniformGridHelper::iUniformGridHelper(iUniformGridFileLoader *loader) : iParallelWorker(loader->GetViewModule()->GetParallelManager())
{
	mLoader = loader;
}


void iUniformGridHelper::ShiftData(int numCom, int dim, int dn, int dataDims[3], float *data, float *buf)
{
	mNumCom = numCom;
	mDims = dataDims;
	mItmp2 = dn;
	mItmp1 = dim;
	mArr1 = data;
	mArr2 = buf;

	this->ParallelExecute(1);
}


void iUniformGridHelper::ExpandUp(float *array, int nc, int oldDims[3], int newDims[3])
{
	//
	//  Can we do this in parallel?
	//
	int i, j, k;
	float *ptr;

	if(array==0 || nc<1 || oldDims[0]<1 || oldDims[1]<1 || oldDims[2]<1)
	{
		IERROR_FATAL("Invalid use of iUniformGridHelper::ExpandUp.");
		return;
	}
	for(i=0; i<3; i++) if(oldDims[i]!=newDims[i] && oldDims[i]+1!=newDims[i])
	{
		IERROR_FATAL("Incorrect dimensions for expanding data.");
		return;
	}

	if(mLoader->GetObserver() != 0)
	{
		mLoader->GetObserver()->Started(iProgressEventObserver::_Formatting);
	}

	vtkIdType oldN12 = nc*oldDims[0]*oldDims[1]; 
	vtkIdType newN12 = nc*newDims[0]*newDims[1];
	vtkIdType size12 = oldN12*sizeof(float);

	if(newN12 > oldN12)
	{
		//
		//  Move plains
		//
		for(k=oldDims[2]-1; k>=1; k--)
		{
			if(mLoader->GetObserver() != 0)
			{
				mLoader->GetObserver()->SetProgress(0.5*float(oldDims[2]-1-k)/oldDims[2]);
				if(mLoader->GetObserver()->IsAborted()) return;
			}
			memmove(array+k*newN12,array+k*oldN12,size12); // will memcpy work?
		}
	}

	vtkIdType oldN1 = nc*oldDims[0]; 
	vtkIdType newN1 = nc*newDims[0];
	vtkIdType size1 = oldN1*sizeof(float);

	if(newN1 > oldN1)
	{
		//
		//  Move rows
		//
		for(k=0; k<oldDims[2]; k++)
		{
			if(mLoader->GetObserver() != 0)
			{
				mLoader->GetObserver()->SetProgress(0.5+0.5*float(k)/oldDims[2]);
				if(mLoader->GetObserver()->IsAborted()) return;
			}
			ptr = array + k*newN12;
			for(j=oldDims[1]-1; j>=1; j--)
			{
				memmove(ptr+j*newN1,ptr+j*oldN1,size1);
			}
		}
	}

	if(mLoader->GetObserver() != 0)
	{
		mLoader->GetObserver()->Finished();
	}
}


void iUniformGridHelper::CollapseDown(float *array, int nc, int oldDims[3], int newDims[3])
{
	//
	//  Can we do this in parallel?
	//
	int i, j, k;
	float *ptr;

	if(array==0 || nc<1 || newDims[0]<1 || newDims[1]<1 || newDims[2]<1)
	{
		IERROR_FATAL("Invalid use of iUniformGridHelper::CompressDown.");
		return;
	}
	for(i=0; i<3; i++) if(oldDims[i]!=newDims[i] && oldDims[i]-1!=newDims[i])
	{
		IERROR_FATAL("Incorrect dimensions for collapsing data.");
		return;
	}

	vtkIdType oldN12 = nc*oldDims[0]*oldDims[1]; 
	vtkIdType oldN1 = nc*oldDims[0]; 
	vtkIdType newN1 = nc*newDims[0];
	vtkIdType size1 = newN1*sizeof(float);

	if(mLoader->GetObserver() != 0)
	{
		mLoader->GetObserver()->Started(iProgressEventObserver::_Formatting);
	}

	if(newN1 < oldN1)
	{
		//
		//  Move rows
		//
		for(k=0; k<newDims[2]; k++)
		{
			if(mLoader->GetObserver() != 0)
			{
				mLoader->GetObserver()->SetProgress(0.5*float(k)/newDims[2]);
				if(mLoader->GetObserver()->IsAborted()) return;
			}

			ptr = array + k*oldN12;
			for(j=1; j<newDims[1]; j++)
			{
				memmove(ptr+j*newN1,ptr+j*oldN1,size1);
			}
		}
	}

	vtkIdType newN12 = nc*newDims[0]*newDims[1];
	vtkIdType size12 = newN12*sizeof(float);

	if(newN12 < oldN12)
	{
		//
		//  Move plains
		//
		for(k=1; k<newDims[2]; k++)
		{
			if(mLoader->GetObserver() != 0)
			{
				mLoader->GetObserver()->SetProgress(0.5+0.5*float(k)/newDims[2]);
				if(mLoader->GetObserver()->IsAborted()) return;
			}
			memmove(array+k*newN12,array+k*oldN12,size12); // will memcpy work?
		}
	}

	if(mLoader->GetObserver() != 0)
	{
		mLoader->GetObserver()->Finished();
	}
}


void iUniformGridHelper::PadPeriodically(float *array, int nc, int oldDims[3], int newDims[3])
{
	int i, j, k;
	float *ptr;
	vtkIdType size, offset;

	if(array==0 || nc<1 || oldDims[0]<1 || oldDims[1]<1 || oldDims[2]<1)
	{
		IERROR_FATAL("Invalid use of iUniformGridHelper::PadPeriodically.");
		return;
	}

	for(i=0; i<3; i++) if(oldDims[i]!=newDims[i] && oldDims[i]+1!=newDims[i])
	{
		IERROR_FATAL("Incorrect dimensions for padding data.");
		return;
	}

	//
	//  Do x-direction
	//
	if(newDims[0] > oldDims[0])
	{
		size = nc*sizeof(float);
		offset = oldDims[0]*nc;
		for(k=0; k<oldDims[2]; k++)
		{
			for(j=0; j<oldDims[1]; j++)
			{
				ptr = array + nc*newDims[0]*(j+newDims[1]*k);
				memcpy(ptr+offset,ptr,size);
			}
		}
	}

	//
	//  Do y-direction
	//
	if(newDims[1] > oldDims[1])
	{
		size = newDims[0]*nc*sizeof(float);
		offset = oldDims[1]*newDims[0]*nc;
		for(k=0; k<oldDims[2]; k++)
		{
			ptr = array + nc*newDims[0]*newDims[1]*k;
			memcpy(ptr+offset,ptr,size);
		}
	}

	//
	//  Do z-direction
	//
	if(newDims[2] > oldDims[2])
	{
		size = newDims[1]*newDims[0]*nc*sizeof(float);
		offset = oldDims[2]*newDims[1]*newDims[0]*nc;

		memcpy(array+offset,array,size);
	}

#ifdef I_CHECK2
	bool ok = true;
	int l, ijk1[3], ijk2[3];
	for(k=0; ok && k<3; k++) if(newDims[k] > oldDims[k])
	{
		i = (k+1) % 3;
		j = (k+2) % 3;
		ijk1[k] = 0;
		ijk2[k] = oldDims[k];

		for(ijk1[j]=ijk2[j]=0; ok && ijk1[j]<newDims[j]; ijk2[j]=++ijk1[j])
		{
			for(ijk1[i]=ijk2[i]=0; ok && ijk1[i]<newDims[i]; ijk2[i]=++ijk1[i])
			{
				for(l=0; ok && l<nc; l++)
				{
					ok = (fabs(array[l+nc*(ijk1[0]+newDims[0]*(ijk1[1]+newDims[1]*ijk1[2]))]-array[l+nc*(ijk2[0]+newDims[0]*(ijk2[1]+newDims[1]*ijk2[2]))]) < iMath::_FloatTolerance);
				}
			}
		}
	}
	if(!ok)
	{
		IERROR_FATAL("Failed check in iUniformGridHelper::PadPeriodically.");
	}
#endif
}


int iUniformGridHelper::ExecuteStep(int step, iParallel::ProcessorInfo &p)
{
	switch(step)
	{
	case 1:
		{
			this->ExecuteShiftData(p);
			return 0;
		}
	default: 
		{
#ifdef I_CHECK1
			IERROR_REPORT_BUG;
#endif
			return 2;
		}
	}
}


//
//  Shift an nc-component vertical array (treat the mesh array as nvar 1-component vertical arrays)
//
void iUniformGridHelper::ExecuteShiftData(iParallel::ProcessorInfo &p)
{
	int n, l1, l2, l, lold, add1;
	int d1 = 0, d2 = 0;
	
	int d = mItmp1;
	int dn = mItmp2;

	switch(d) 
	{
	case 0: { d1 = 1; d2 = 2; break; }
	case 1: { d1 = 0; d2 = 2; break; }
	case 2: { d1 = 0; d2 = 1; break; }
	}
	
	int ddim = mDims[d];
	vtkIdType off0 = 0, off1 = 0, add2;
	
	int kbeg, kend, kstp;
	iParallel::SplitRange(p,mDims[d1],kbeg,kend,kstp);

	for(l1=kbeg; l1<kend; l1++)
	{
		if(mLoader->GetObserver() != 0)
		{
			if(this->IsMaster(p)) mLoader->GetObserver()->SetProgress((d+(float)(l1-kbeg)/(kend-kbeg))/3.0);
			if(mLoader->GetObserver()->IsAborted()) return;
		}
		for(l2=0; l2<mDims[d2]; l2++)
		{
			switch(d) 
			{
			case 0: { off0 = mNumCom*(vtkIdType)mDims[0]*(l1+mDims[1]*l2); off1 = mNumCom;	break; }
			case 1: { off0 = mNumCom*(l1+(vtkIdType)mDims[0]*mDims[1]*l2); off1 = mNumCom*mDims[0]; break; }
			case 2: { off0 = mNumCom*(l1+(vtkIdType)mDims[0]*l2); off1 = mNumCom*(vtkIdType)mDims[0]*mDims[1]; break; }
			}
			
			for(l=0; l<ddim-1; l++) 
			{
				lold = l - dn;
				while(lold < 0) lold += ddim;
				while(lold >= ddim) lold -= ddim;
				
				add1 = mNumCom*l;
				add2 = off0 + off1*lold;

				for(n=0; n<mNumCom; n++) mArr2[n+add1] = mArr1[n+add2];
			}
			add1 = mNumCom*(ddim-1);
			for(n=0; n<mNumCom; n++) mArr2[n+add1] = mArr2[n];
			
			for(l=0; l<ddim; l++) 
			{
				add1 = mNumCom*l;
				add2 = off0 + off1*l;
				
				for(n=0; n<mNumCom; n++) mArr1[n+add2] = mArr2[n+add1];
			}
		}
	}
}


//
//  Helper class
//
iUniformScalarsHelper::iUniformScalarsHelper(iUniformScalarsFileLoader *subject) : iParallelWorker(subject->GetViewModule()->GetParallelManager())
{
	mLoader = subject;
	mObserver = subject->GetObserver();
}


void iUniformScalarsHelper::OperateOnData1(int nvar, int dims[3], const iString &calculatorExpression, bool doVector, float ps, float dp)
{
	mNumVars = nvar;
	mDims = dims;
	mCalculatorExpression = calculatorExpression;
	mDoVector = doVector;

	mProgStart = ps;
	mProgStep = dp;

	iCalculator<float> calc;
	iCalculator<float>::real_var var("Var",mNumVars,&calc);
	iCalculator<float>::real_var vec("Vec",3);
	if(mDoVector) calc.AddVariable(&vec);

	if(!calc.Compile(mCalculatorExpression) || !calc.Link())
	{
		IERROR_LOW(calc.GetErrorMessage());
	}
	else
	{
		this->ParallelExecute(1);
	}
}


void iUniformScalarsHelper::OperateOnData2(int nvar, int dims[3], float *fMin, float *fMax, bool &overflow, float ps, float dp)
{
	int i, n;

	mNumVars = nvar;
	mDims = dims;
	mNumProcs = this->GetManager()->GetNumberOfProcessors();
	mMin.Extend(mNumVars*mNumProcs);
	mMax.Extend(mNumVars*mNumProcs);
	
	mProgStart = ps;
	mProgStep = dp;
	mOverflow = false;

	this->ParallelExecute(2);

	for(n=0; n<nvar; n++)
	{
		fMin[n] = mMin[n];
		fMax[n] = mMax[n];
		for(i=1; i<mNumProcs; i++)
		{
			if(fMin[n] > mMin[n+nvar*i]) fMin[n] = mMin[n+nvar*i];
			if(fMax[n] < mMax[n+nvar*i]) fMax[n] = mMax[n+nvar*i];
		}
	}

	overflow = mOverflow;
}


int iUniformScalarsHelper::ExecuteStep(int step, iParallel::ProcessorInfo &p)
{
	switch(step)
	{
	case 1:
		{
			this->ExecuteOperateOnData1(p);
			return 0;
		}
	case 2:
		{
			if(mNumProcs != p.NumProcs) return 3;
			this->ExecuteOperateOnData2(p);
			return 0;
		}
	default: 
		{
#ifdef I_CHECK1
			IERROR_REPORT_BUG;
#endif
			return 2;
		}
	}
}


//
//  Operate on mesh data
//
void iUniformScalarsHelper::ExecuteOperateOnData1(iParallel::ProcessorInfo &p)
{
	float *arr1 = mLoader->GetDataPointer();
	float *arr2 = mLoader->mVectorFieldLoader->GetDataPointer();

	if(arr1==0 || (mDoVector && arr2==0)) return;

	vtkIdType size = (vtkIdType)mDims[0]*mDims[1]*mDims[2];
	int nout = mLoader->GetCalculatorOutput();
	vtkIdType l, kbeg, kend, kstp;
	iParallel::SplitRange(p,size,kbeg,kend,kstp);

	float progScale = mProgStep/kstp;

	//
	// Create our calculator
	//
	iCalculator<float> calc;

	iCalculator<float>::real_ptr var(arr1+kbeg*mNumVars,"Var",mNumVars,&calc);
	iCalculator<float>::real_ptr vec(arr2==0?arr1:arr2+kbeg*3,"Vec",3);
	if(mDoVector)
	{
		calc.AddVariable(&vec);
	} 

	if(!calc.Compile(mCalculatorExpression) || !calc.Link()) return;

	const float *output = calc.GetResultData();

	for(l=kbeg; l<kend; l++) 
	{
		if(l%1000==0 && mObserver!=0)
		{
			if(this->IsMaster(p)) mObserver->SetProgress(mProgStart+progScale*(l-kbeg));
			if(mObserver->IsAborted()) return;
		}
		if(!calc.Execute()) break;		
		arr1[nout+mNumVars*l] = *output;
		++var;
		++vec;
	}
}


void iUniformScalarsHelper::ExecuteOperateOnData2(iParallel::ProcessorInfo &p)
{
	float fMin, fMax;
	vtkIdType size = (vtkIdType)mDims[0]*mDims[1]*mDims[2];
	vtkIdType kbeg, kend, kstp;
	iParallel::SplitRange(p,size,kbeg,kend,kstp);

	//
	//  Format everything
	//
	iDataLimits *lim = mLoader->GetSubject(0)->GetLimits();

	vtkIdType l, larr;
	int n;

	float *arr1 = mLoader->GetDataPointer();
	if(arr1 == 0) return;

	float progScale = mProgStep/(mNumVars*kstp);

	for(n=0; n<mNumVars; n++)
	{
		fMin = iMath::_LargeFloat;
		fMax = -iMath::_LargeFloat;

		for(larr=kbeg; larr<kend; larr++)
		{
			if(larr%10000==0 && mObserver!=0)
			{
				if(this->IsMaster(p)) mObserver->SetProgress(mProgStart+progScale*(larr-kbeg+n*kstp));
				if(mObserver->IsAborted()) return;
			}
			l = n + mNumVars*larr;
			if(arr1[l] < -iMath::_LargeFloat)
			{
				mOverflow = true;
				arr1[l] = -iMath::_LargeFloat;
			}
			if(arr1[l] > iMath::_LargeFloat)
			{
				mOverflow = true;
				arr1[l] = iMath::_LargeFloat;
			}
			if(arr1[l] > fMax) fMax = arr1[l];
			if(arr1[l] < fMin) fMin = arr1[l];
		}
		
		if(mNumProcs == p.NumProcs)
		{
			mMin[n+mNumVars*p.ThisProc] = fMin;
			mMax[n+mNumVars*p.ThisProc] = fMax;
		}
	}
}

