/* Auport, a versatile, portable sound IO library.
 *
 * Copyright (C) 1999, 2000 IRCAM - Centre Georges Pompidou
 * IRCAM, 1 pl. Stravinsky, F-75004 Paris 
 *
 * Author: Peter Hanappe, peter@hanappe.com
 *
 * This file is part of the Varese program. 
 * Varese is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 *
 */
/* auport.c
 */
#include "iiwu_auport.h"

/*
 * iiwu_pcm_data_t
 */
struct _iiwu_pcm_data_t {
  int channels;
  double sample_rate;
  int sample_format;
  int bits_per_sample;
};

iiwu_pcm_data_t* new_iiwu_pcm_data(int ch, double sr, int sf);
void delete_iiwu_pcm_data(iiwu_pcm_data_t* data);

/*
 * iiwu_audio_app
 */
typedef struct {
  iiwu_auport_callback_t write;
  void* buffer;
  void* data;
} iiwu_audio_app_t;

iiwu_audio_app_t* new_iiwu_audio_app(iiwu_auport_t* port, iiwu_auport_callback_t write, void* data);
void delete_iiwu_audio_app(iiwu_audio_app_t* app);

/*
 * iiwu_codec_t
 *
 * Codec interface. The codec converts the data in the input buffer
 * to the output buffer. The meaning of inlen depends on the audio
 * format. For the PCM format it is the number of frames of the
 * input format. The codec should return the size of the output
 * buffer (#frames of output format) in the outlen parameter. 
 */
typedef struct iiwu_codec_t {
  void (*convert)(struct iiwu_codec_t* data, void* in, int inlen, 
		  void* out, int outlen);
  int samples_per_frame;                   /* Short cut for PCM formats */
  int bytes_per_frame;                     /* Short cut for copy codec */
/*    void* buffer; */
} iiwu_codec_t;

iiwu_codec_t* new_iiwu_codec(iiwu_pcm_data_t* from, iiwu_pcm_data_t* to, int frames);
void delete_iiwu_codec(iiwu_codec_t* codec);


/*
 * iiwu_auport_t
 */
struct _iiwu_auport_t { 
  char* driver_name;
  char* device_name;
  iiwu_pcm_data_t* app_format;
  iiwu_pcm_data_t* dev_format;
  int queue_size;
  int buffer_size;
  int state;  
  iiwu_audio_app_t* app;
  iiwu_codec_t* codec;
  iiwu_audio_driver_t* driver;
  iiwu_audio_driver_t* (*new_driver)(iiwu_auport_t* port);
  int (*delete_driver)(iiwu_audio_driver_t* p);
  char error[256];
};


/*
 * iiwu_adriver_definition
 */
struct iiwu_adriver_definition_t {
  char* name;
  iiwu_audio_driver_t* (*new_driver)(iiwu_auport_t* port);
  int (*delete_driver)(iiwu_audio_driver_t* p);
};

#if ALSA_SUPPORT
extern iiwu_audio_driver_t* new_iiwu_alsa_audio_driver(iiwu_auport_t* port);
extern int delete_iiwu_alsa_audio_driver(iiwu_audio_driver_t* p);
#endif

#if OSS_SUPPORT
extern iiwu_audio_driver_t* new_iiwu_oss_audio_driver(iiwu_auport_t* port);
extern int delete_iiwu_oss_audio_driver(iiwu_audio_driver_t* p);
#endif

#if DSOUND_SUPPORT
extern iiwu_audio_driver_t* new_iiwu_dsound_audio_driver(iiwu_auport_t* port);
extern int delete_iiwu_dsound_audio_driver(iiwu_audio_driver_t* p);
#endif

struct iiwu_adriver_definition_t iiwu_audio_drivers[] = {
#if OSS_SUPPORT
  { "oss", new_iiwu_oss_audio_driver, delete_iiwu_oss_audio_driver },
#endif
#if ALSA_SUPPORT
  { "alsa", new_iiwu_alsa_audio_driver, delete_iiwu_alsa_audio_driver },
#endif
#if DSOUND_SUPPORT
  { "dsound", new_iiwu_dsound_audio_driver, delete_iiwu_dsound_audio_driver },
#endif
  { NULL, NULL, NULL }
};

#if OSS_SUPPORT
char* iiwu_default_audio_driver = "oss";
#elif ALSA_SUPPORT
char* iiwu_default_audio_driver = "alsa";
#elif DSOUND_SUPPORT
char* iiwu_default_audio_driver = "dsound";
#else
#error No audio driver specified
#endif


/* 
 * Conversion routines 
 */
void iiwu_copy(iiwu_codec_t* data, 
		 void* in, int inlen, 
		 void* out, int outlen);
void iiwu_double_to_u8(iiwu_codec_t* codec, 
		  void* in, int inlen, 
		  void* out, int outlen);
void iiwu_double_to_s8(iiwu_codec_t* codec, 
		  void* in, int inlen, 
		  void* out, int outlen);
void iiwu_double_to_s16le(iiwu_codec_t* codec, 
		     void* in, int inlen, 
		     void* out, int outlen);
void iiwu_double_to_s16be(iiwu_codec_t* codec, 
		     void* in, int inlen, 
		     void* out, int outlen);
void iiwu_double_to_s16he(iiwu_codec_t* codec, 
		     void* in, int inlen, 
		     void* out, int outlen);
void iiwu_double_to_u16le(iiwu_codec_t* codec, 
		     void* in, int inlen, 
		     void* out, int outlen);
void iiwu_double_to_u16be(iiwu_codec_t* codec, 
		     void* in, int inlen, 
		     void* out, int outlen);
void iiwu_double_to_u16he(iiwu_codec_t* codec, 
		     void* in, int inlen, 
		     void* out, int outlen);
void iiwu_float_to_u8(iiwu_codec_t* codec, 
		 void* in, int inlen, 
		 void* out, int outlen);
void iiwu_float_to_s8(iiwu_codec_t* codec, 
		 void* in, int inlen, 
		 void* out, int outlen);
void iiwu_float_to_s16le(iiwu_codec_t* codec, 
		    void* in, int inlen, 
		    void* out, int outlen);
void iiwu_float_to_s16be(iiwu_codec_t* codec, 
		    void* in, int inlen, 
		    void* out, int outlen);
void iiwu_float_to_s16he(iiwu_codec_t* codec, 
		    void* in, int inlen, 
		    void* out, int outlen);
void iiwu_float_to_u16le(iiwu_codec_t* codec, 
		    void* in, int inlen, 
		    void* out, int outlen);
void iiwu_float_to_u16be(iiwu_codec_t* codec, 
		    void* in, int inlen, 
		    void* out, int outlen);
void iiwu_float_to_u16he(iiwu_codec_t* codec, 
		    void* in, int inlen, 
		    void* out, int outlen);

/***************************************************************
 *
 *                         iiwu_auport
 */
iiwu_auport_t* new_iiwu_auport(char* driver, char* device, 
			       int app_format, int dev_format,
			       int sr, int ch, int bs, int qs, 
			       iiwu_auport_callback_t callback, 
			       void* data)
{
  int i, len;
  iiwu_auport_t* port = NULL;
  iiwu_codec_t* codec = NULL;
  /* Before allocating any system resources, check the queue and
   * buffer size and whether we have an appropriate codec
   * available. */

  /* Simple way to test acceptable buffer and queue sizes. Does it
   * have to be more complicated? */
  if ((bs != 64) && (bs != 128) && (bs != 256) 
      && (bs != 512) && (bs != 1024) 
      && (bs != 2048) && (bs != 4096)) {
    IIWU_LOG(ERR, "Buffer size not supported");
    return NULL;
  }
  len = qs / bs;
  if ((len != 2) && (len != 4) && (len != 8) && (len != 16) && (len != 32)) {
    IIWU_LOG(ERR, "Queue size should be 2, 4, 8, 16, or 32 times the buffer size");
    return NULL;
  }
  qs = len * bs;

  /* create the audio port */
  port = (iiwu_auport_t*) IIWU_MALLOC(sizeof(iiwu_auport_t));
  if (port == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    return NULL;
  }
  IIWU_MEMSET(port, 0, sizeof(iiwu_auport_t));

  /* create the sample formats */
  port->app_format = new_iiwu_pcm_data(ch, sr, app_format);
  if (port->app_format == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    goto error_recovery;
  }
  port->dev_format = new_iiwu_pcm_data(ch, sr, dev_format);
  if (port->dev_format == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    goto error_recovery;
  }

  /* Try opening the codec */
  codec = new_iiwu_codec(port->app_format, port->dev_format, bs);
  if (codec == NULL) {
    return NULL;
  }
  port->codec = codec;
  port->queue_size = qs;
  port->buffer_size = bs;
  port->state = IIWU_AUPORT_PLAYING;
  port->error[0] = 0;

  /* create the application interface */
  port->app = new_iiwu_audio_app(port, callback, data);
  if (port->app == NULL) {
    goto error_recovery;
  }

  /* copy the device name */
  if (device != NULL) {
    port->device_name = IIWU_MALLOC(IIWU_STRLEN(device) + 1);
    if (port->device_name == NULL) {
      IIWU_LOG(ERR, "Out of memory");
      IIWU_FREE(port);
      return NULL;
    }
    IIWU_STRCPY(port->device_name, device);
  }

  /* choose the default driver if none is specified */
  if (driver == NULL) {
    driver = iiwu_default_audio_driver;
  }

  /* copy the driver name */
  port->driver_name = IIWU_MALLOC(IIWU_STRLEN(driver) + 1);
  if (port->driver_name == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    IIWU_FREE(port);
    return NULL;
  }
  IIWU_STRCPY(port->driver_name, driver);

  /* find the driver definition */
  i = 0;
  while (iiwu_audio_drivers[i].name != NULL) {
    if (IIWU_STRCMP(driver, iiwu_audio_drivers[i].name) == 0) {
      IIWU_LOG(DBG, "Using %s audio driver\n", iiwu_audio_drivers[i].name);
      port->new_driver = iiwu_audio_drivers[i].new_driver;
      port->delete_driver = iiwu_audio_drivers[i].delete_driver;
      break;
    }
    i++;
  }

  if ((port->new_driver == NULL) || (port->delete_driver == NULL)) {
    IIWU_LOG(ERR, "Couldn't find the requested driver");
    goto error_recovery;
  }

  /* as soon as we open the audio device, the sound will start flowing */
  port->driver = port->new_driver(port);
  if (port->driver == NULL) {
    goto error_recovery;
  }
  return port;

 error_recovery:
  delete_iiwu_auport(port);
  return NULL;  
}

/*
 * delete_iiwu_auport
 */
int delete_iiwu_auport(iiwu_auport_t* p)
{
  if (p != NULL) {
    p->state = IIWU_AUPORT_STOPPED;
    if (p->driver != NULL) {
      p->delete_driver(p->driver);
    }
    if (p->device_name != NULL) {
      IIWU_FREE(p->device_name);
    }
    if (p->driver_name != NULL) {
      IIWU_FREE(p->driver_name);
    }
    if (p->app_format != NULL) {
      delete_iiwu_pcm_data(p->app_format);
    }
    if (p->dev_format != NULL) {
      delete_iiwu_pcm_data(p->dev_format);
    }
    if (p->app != NULL) {
      delete_iiwu_audio_app(p->app);
    }
    if (p->codec != NULL) {
      delete_iiwu_codec(p->codec);
    }
    IIWU_FREE(p);
  }
  return IIWU_OK;
}

/*
 * iiwu_auport_write
 */
void iiwu_auport_write(iiwu_auport_t* p, void* buf, int frames)
{
  int stop;
  stop = p->app->write(p->app->data, p->app->buffer, p->buffer_size);
  p->codec->convert(p->codec, p->app->buffer, p->buffer_size, buf, p->buffer_size);
  if (stop != 0) {
    p->state = IIWU_AUPORT_STOPPED;
  }
  return;
}

/*
 * iiwu_auport_get_state
 */
int iiwu_auport_get_state(iiwu_auport_t* p)
{
  return p->state;
}

/*
 * iiwu_auport_get_dev_format
 */
iiwu_pcm_data_t* iiwu_auport_get_dev_format(iiwu_auport_t* p) 
{
  return p->dev_format;
}

/*
 * iiwu_auport_get_buffer_size
 */
int iiwu_auport_get_buffer_size(iiwu_auport_t* p) 
{
  return p->buffer_size;
}

/*
 * iiwu_auport_get_queue_size
 */
int iiwu_auport_get_queue_size(iiwu_auport_t* p) 
{
  return p->queue_size;
}

/*
 * iiwu_auport_get_device_name
 */
char* iiwu_auport_get_device_name(iiwu_auport_t* p) 
{
  return p->device_name;
}

/*
 * iiwu_auport_set_device_name
 */
void iiwu_auport_set_device_name(iiwu_auport_t* p, char* name) 
{
  if (p->device_name != NULL) {
    IIWU_FREE(p->device_name);
    p->device_name = NULL;
  }
  if (name == NULL) {
    return;
  }
  p->device_name = IIWU_MALLOC(IIWU_STRLEN(name) + 1);
  if (p->device_name == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    return;
  }
  IIWU_STRCPY(p->device_name, name);
}

/***************************************************************
 *
 *                         iiwu_pcm_data_t
 */

/*
 * new_iiwu_pcm_data 
 */
iiwu_pcm_data_t* new_iiwu_pcm_data(int ch, double sr, int sf) 
{
  iiwu_pcm_data_t* format;
  format = IIWU_NEW(iiwu_pcm_data_t);
  if (format == NULL) {
    return NULL;
  }
  format->channels = ch;
  format->sample_rate = sr;
  format->sample_format = sf;
  switch (sf) {
  case IIWU_SAMPLE_DOUBLE: 
    format->bits_per_sample = 8 * sizeof(double); 
    break;
  case IIWU_SAMPLE_FLOAT: 
    format->bits_per_sample = 8 * sizeof(float); 
    break;
  case IIWU_SAMPLE_S16HE: 
  case IIWU_SAMPLE_S16BE: 
  case IIWU_SAMPLE_S16LE: 
  case IIWU_SAMPLE_U16HE: 
  case IIWU_SAMPLE_U16BE: 
  case IIWU_SAMPLE_U16LE: 
    format->bits_per_sample = 16; 
    break;
  case IIWU_SAMPLE_S8: 
  case IIWU_SAMPLE_U8: 
    format->bits_per_sample = 8; 
    break;
  }    
  return format;
}

/*
 * iiwu_pcm_data_framesize
 */
int iiwu_pcm_data_framesize(iiwu_pcm_data_t* format) 
{
  return format->channels * format->bits_per_sample / 8; 
}

/*
 * iiwu_pcm_data_get_bps
 */
int iiwu_pcm_data_get_bps(iiwu_pcm_data_t* format) 
{
  return format->bits_per_sample;
}

/*
 * iiwu_pcm_data_get_channels
 */
int iiwu_pcm_data_get_channels(iiwu_pcm_data_t* format) 
{
  return format->channels;
}

/*
 * iiwu_pcm_data_get_format
 */
int iiwu_pcm_data_get_format(iiwu_pcm_data_t* format) 
{
  return format->sample_format;
}

/*
 * iiwu_pcm_data_get_sample_rate
 */
int iiwu_pcm_data_get_sample_rate(iiwu_pcm_data_t* format) 
{
  return (int) format->sample_rate;
}

/*
 * delete_iiwu_pcm_data 
 */
void delete_iiwu_pcm_data(iiwu_pcm_data_t* data) 
{
  if (data == NULL) {
    return;
  }
  IIWU_FREE(data);
}

/***************************************************************
 *
 *                         iiwu_audio_app
 */
/*
 * new_iiwu_audio_app
 */
iiwu_audio_app_t* new_iiwu_audio_app(iiwu_auport_t* port, iiwu_auport_callback_t write, void* data) 
{
  iiwu_audio_app_t* app;
  app = (iiwu_audio_app_t*) IIWU_MALLOC(sizeof(iiwu_audio_app_t));
  if (app == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    return NULL;
  }
  IIWU_MEMSET(app, 0, sizeof(iiwu_audio_app_t));
  app->data = data;
  app->write = write;
  app->buffer = NULL;
  if (app->write != NULL) {
    app->buffer = IIWU_MALLOC(port->buffer_size * iiwu_pcm_data_framesize(port->app_format));
    if (app->buffer == NULL) {
      IIWU_LOG(ERR, "Out of memory");
      IIWU_FREE(app);
      return NULL;      
    }
  }
  return app;
}

/*
 * delete_iiwu_audio_app
 */
void delete_iiwu_audio_app(iiwu_audio_app_t* app)
{
  if (app == NULL) {
    return;
  }
  if (app->buffer != NULL) {
    IIWU_FREE(app->buffer);
  }
  IIWU_FREE(app);
}

/***************************************************************
 *
 *                         iiwu_codec
 */
/*
 * new_iiwu_codec
 */
iiwu_codec_t* new_iiwu_codec(iiwu_pcm_data_t* from, iiwu_pcm_data_t* to, int frames) 
{
  iiwu_codec_t* codec = NULL;
  void (*convert_fcn)(iiwu_codec_t* data, void* in, int inlen, 
		      void* out, int outlen);
  int from_format;
  int to_format;
  convert_fcn = NULL;
  if (from->channels != from->channels) {
    IIWU_LOG(ERR, "Can't convert between different number of channels");
    return NULL;
  }
  from_format = from->sample_format;
  to_format = to->sample_format;
  if (from_format == to_format) {
    convert_fcn = iiwu_copy;
  } else if (from_format == IIWU_SAMPLE_DOUBLE) {
    switch (to_format) {
    case IIWU_SAMPLE_S8: convert_fcn = iiwu_double_to_s8; break;
    case IIWU_SAMPLE_U8: convert_fcn = iiwu_double_to_u8; break;
    case IIWU_SAMPLE_S16HE: convert_fcn = iiwu_double_to_s16he; break;
    case IIWU_SAMPLE_S16BE: convert_fcn = iiwu_double_to_s16be; break;
    case IIWU_SAMPLE_S16LE: convert_fcn = iiwu_double_to_s16le; break;
    case IIWU_SAMPLE_U16HE: convert_fcn = iiwu_double_to_u16he; break;
    case IIWU_SAMPLE_U16BE: convert_fcn = iiwu_double_to_u16be; break;
    case IIWU_SAMPLE_U16LE: convert_fcn = iiwu_double_to_u16le; break;
    }
  } else if (from_format == IIWU_SAMPLE_FLOAT) {
    switch (to_format) {
    case IIWU_SAMPLE_S8: convert_fcn = iiwu_float_to_s8; break;
    case IIWU_SAMPLE_U8: convert_fcn = iiwu_float_to_u8; break;
    case IIWU_SAMPLE_S16HE: convert_fcn = iiwu_float_to_s16he; break;
    case IIWU_SAMPLE_S16BE: convert_fcn = iiwu_float_to_s16be; break;
    case IIWU_SAMPLE_S16LE: convert_fcn = iiwu_float_to_s16le; break;
    case IIWU_SAMPLE_U16HE: convert_fcn = iiwu_float_to_u16he; break;
    case IIWU_SAMPLE_U16BE: convert_fcn = iiwu_float_to_u16be; break;
    case IIWU_SAMPLE_U16LE: convert_fcn = iiwu_float_to_u16le; break;
    }
  }
  if (convert_fcn == NULL) {
    IIWU_LOG(ERR, "Can't convert between the requested audio formats");
    return NULL;
  }
  codec = (iiwu_codec_t*) IIWU_MALLOC(sizeof(iiwu_codec_t));
  if (codec == NULL) {
    IIWU_LOG(ERR, "Out of memory");
    return NULL;
  }
  codec->convert = convert_fcn;
  codec->samples_per_frame = from->channels;
  codec->bytes_per_frame = iiwu_pcm_data_framesize(from);
/*    buffer_size = frames * iiwu_pcm_data_framesize(to); */
/*    codec->buffer = IIWU_MALLOC(buffer_size); */
/*    if (codec->buffer == NULL) { */
/*      IIWU_LOG(ERR, "Out of memory"); */
/*      IIWU_FREE(codec); */
/*      return NULL; */
/*    } */
  return codec;
}

/*
 * delete_iiwu_codec
 */
void delete_iiwu_codec(iiwu_codec_t* data) 
{
  IIWU_FREE(data);
}

/***************************************************************
 *
 *                  Conversion routines
 */

/*
 * iiwu_copy
 */
void iiwu_copy(iiwu_codec_t* data, void* in, int inlen, void* out, int outlen) {
  IIWU_MEMCPY(out, in, inlen * data->bytes_per_frame);
}

/**
 * iiwu_double_to_u8
 *
 * Convert an array of double samples to unsigned 8 bits integers.
 */
void iiwu_double_to_u8(iiwu_codec_t* data, void* in, int inlen, void* out, int outlen) 
{
  int i;
  double* f = (double*) in;
  unsigned char* s = (unsigned char*) out;
  int len = inlen * data->samples_per_frame;
  for (i = 0; i < len; i++) {
    s[i] = (unsigned char) (128.0 + 127.0 * f[i]);
  }
}

/**
 * iiwu_double_to_s8
 *
 * Convert an array of double samples to signed 8 bits integers.
 */
void iiwu_double_to_s8(iiwu_codec_t* data, void* in, int inlen, void* out, int outlen) 
{
  int i;
  double* f = (double*) in;
  signed char* s = (signed char*) out;
  int len = inlen * data->samples_per_frame;
  for (i = 0; i < len; i++) {
    s[i] = (signed char) (127.0 * f[i]);
  }
}

/**
 * iiwu_double_to_s16le
 *
 * Convert an array of double samples to signed 16 bits little endian integers.
 */
void iiwu_double_to_s16le(iiwu_codec_t* data, void* in, int inlen, void* out, int outlen) 
{
  int i, k;
  signed short v;
  double* f = (double*) in;
  unsigned char* s = (unsigned char*) out;
  int len = inlen * data->samples_per_frame;
  for (i = 0, k = 0; i < len; i++) {
    v = (signed short) (32767.0 * f[i]);
    s[k++] = (unsigned char) (v & 0xff);
    s[k++] = (unsigned char) ((v >> 8) & 0xff);
  }
}

/**
 * iiwu_double_to_s16be
 *
 * Convert an array of double samples to signed 16 bits big endian integers.
 */
void iiwu_double_to_s16be(iiwu_codec_t* data, void* in, int inlen, void* out, int outlen) 
{
  int i, k;
  signed short v;
  double* f = (double*) in;
  unsigned char* s = (unsigned char*) out;
  int len = inlen * data->samples_per_frame;
  for (i = 0, k = 0; i < len; i++) {
    v = (signed short) (32767.0 * f[i]);
    s[k++] = (unsigned char) ((v >> 8) & 0xff);
    s[k++] = (unsigned char) (v & 0xff);
  }
}

/**
 * iiwu_double_to_s16he
 *
 * Convert an array of double samples to signed 16 bits "host" endian integers.
 */
void iiwu_double_to_s16he(iiwu_codec_t* data, void* in, int inlen, void* out, int outlen) 
{
  int i;
  double* f = (double*) in;
  signed short* v = (signed short*) out;
  int len = inlen * data->samples_per_frame;
  for (i = 0; i < len; i++) {
    v[i] = (signed short) (32767.0 * f[i]);
  }
}

/**
 * iiwu_double_to_u16le
 *
 * Convert an array of double samples to unsigned 16 bits little endian integers.
 */
void iiwu_double_to_u16le(iiwu_codec_t* data, void* in, int inlen, void* out, int outlen) 
{
  int i, k;
  unsigned short v;
  double* f = (double*) in;
  unsigned char* s = (unsigned char*) out;
  int len = inlen * data->samples_per_frame;
  for (i = 0, k = 0; i < len; i++) {
    v = (unsigned short) (32767.0 + 32767.0 * f[i]);
    s[k++] = (unsigned char) (v & 0xff);
    s[k++] = (unsigned char) ((v >> 8) & 0xff);
  }
}

/**
 * iiwu_double_to_u16be
 *
 * Convert an array of double samples to unsigned 16 bits big endian integers.
 */
void iiwu_double_to_u16be(iiwu_codec_t* data, void* in, int inlen, void* out, int outlen) 
{
  int i, k;
  unsigned short v;
  double* f = (double*) in;
  unsigned char* s = (unsigned char*) out;
  int len = inlen * data->samples_per_frame;
  for (i = 0, k = 0; i < len; i++) {
    v = (unsigned short) (32767.0 + 32767.0 * f[i]);
    s[k++] = (unsigned char) ((v >> 8) & 0xff);
    s[k++] = (unsigned char) (v & 0xff);
  }
}

/**
 * iiwu_double_to_u16he
 *
 * Convert an array of double samples to unsigned 16 bits "host" endian integers.
 */
void iiwu_double_to_u16he(iiwu_codec_t* data, void* in, int inlen, void* out, int outlen) 
{
  int i;
  double* f = (double*) in;
  unsigned short* v = (unsigned short*) out;
  int len = inlen * data->samples_per_frame;
  for (i = 0; i < len; i++) {
    v[i] = (unsigned short) (32767.0 + 32767.0 * f[i]);
  }
}

/**
 * iiwu_float_to_u8
 *
 * Convert an array of floating samples to unsigned 8 bits integers.
 */
void iiwu_float_to_u8(iiwu_codec_t* data, void* in, int inlen, void* out, int outlen) 
{
  int i;
  float* f = (float*) in;
  unsigned char* s = (unsigned char*) out;
  int len = inlen * data->samples_per_frame;
  for (i = 0; i < len; i++) {
    s[i] = (unsigned char) (128.0 + 127.0 * f[i]);
  }
}

/**
 * iiwu_float_to_s8
 *
 * Convert an array of floating samples to signed 8 bits integers.
 */
void iiwu_float_to_s8(iiwu_codec_t* data, void* in, int inlen, void* out, int outlen) 
{
  int i;
  float* f = (float*) in;
  signed char* s = (signed char*) out;
  int len = inlen * data->samples_per_frame;
  for (i = 0; i < len; i++) {
    s[i] = (signed char) (127.0 * f[i]);
  }
}

/**
 * iiwu_float_to_s16le
 *
 * Convert an array of floating samples to signed 16 bits little endian integers.
 */
void iiwu_float_to_s16le(iiwu_codec_t* data, void* in, int inlen, void* out, int outlen) 
{
  int i, k;
  signed short v;
  float* f = (float*) in;
  unsigned char* s = (unsigned char*) out;
  int len = inlen * data->samples_per_frame;
  for (i = 0, k = 0; i < len; i++) {
    v = (signed short) (32767.0 * f[i]);
    s[k++] = (unsigned char) (v & 0xff);
    s[k++] = (unsigned char) ((v >> 8) & 0xff);
  }
}

/**
 * iiwu_float_to_s16be
 *
 * Convert an array of floating samples to signed 16 bits big endian integers.
 */
void iiwu_float_to_s16be(iiwu_codec_t* data, void* in, int inlen, void* out, int outlen) 
{
  int i, k;
  signed short v;
  float* f = (float*) in;
  unsigned char* s = (unsigned char*) out;
  int len = inlen * data->samples_per_frame;
  for (i = 0, k = 0; i < len; i++) {
    v = (signed short) (32767.0 * f[i]);
    s[k++] = (unsigned char) ((v >> 8) & 0xff);
    s[k++] = (unsigned char) (v & 0xff);
  }
}

/**
 * iiwu_float_to_s16he
 *
 * Convert an array of floating samples to signed 16 bits "host" endian integers.
 */
void iiwu_float_to_s16he(iiwu_codec_t* data, void* in, int inlen, void* out, int outlen) 
{
  int i;
  float* f = (float*) in;
  signed short* v = (signed short*) out;
  int len = inlen * data->samples_per_frame;
  for (i = 0; i < len; i++) {
    v[i] = (signed short) (32767.0 * f[i]);
  }
}

/**
 * iiwu_float_to_u16le
 *
 * Convert an array of floating samples to unsigned 16 bits little endian integers.
 */
void iiwu_float_to_u16le(iiwu_codec_t* data, void* in, int inlen, void* out, int outlen) 
{
  int i, k;
  unsigned short v;
  float* f = (float*) in;
  unsigned char* s = (unsigned char*) out;
  int len = inlen * data->samples_per_frame;
  for (i = 0, k = 0; i < len; i++) {
    v = (unsigned short) (32767.0 + 32767.0 * f[i]);
    s[k++] = (unsigned char) (v & 0xff);
    s[k++] = (unsigned char) ((v >> 8) & 0xff);
  }
}

/**
 * iiwu_float_to_u16be
 *
 * Convert an array of floating samples to unsigned 16 bits big endian integers.
 */
void iiwu_float_to_u16be(iiwu_codec_t* data, void* in, int inlen, void* out, int outlen) 
{
  int i, k;
  unsigned short v;
  float* f = (float*) in;
  unsigned char* s = (unsigned char*) out;
  int len = inlen * data->samples_per_frame;
  for (i = 0, k = 0; i < len; i++) {
    v = (unsigned short) (32767.0 + 32767.0 * f[i]);
    s[k++] = (unsigned char) ((v >> 8) & 0xff);
    s[k++] = (unsigned char) (v & 0xff);
  }
}

/**
 * iiwu_float_to_u16he
 *
 * Convert an array of floating samples to unsigned 16 bits "host" endian integers.
 */
void iiwu_float_to_u16he(iiwu_codec_t* data, void* in, int inlen, void* out, int outlen) 
{
  int i;
  float* f = (float*) in;
  unsigned short* v = (unsigned short*) out;
  int len = inlen * data->samples_per_frame;
  for (i = 0; i < len; i++) {
    v[i] = (unsigned short) (32767.0 + 32767.0 * f[i]);
  }
}
