/*
 * jMax
 * Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * See file LICENSE for further informations on licensing terms.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * Based on Max/ISPW by Miller Puckette.
 *
 * Authors: Francois Dechelle, Norbert Schnell, Peter Hanappe
 *
 */

#include <fts/fts.h>
#include <stdlib.h>
#include "iiwusynth.h"

static void fts_iiwu_synth_module_init(void);
void fts_iiwu_synth_config(void);

fts_module_t iiwusynth_module = { "iiwusynth", "iiwusynth object", fts_iiwu_synth_module_init, 0, 0};

static void fts_iiwu_synth_module_init(void)
{
  fts_iiwu_synth_config();
}

typedef struct 
{
  fts_midiport_t head;
  iiwu_synth_t* synth;
} fts_iiwu_synth_t;

static fts_symbol_t fts_iiwu_synth_symbol = 0;

void
fts_iiwu_synth_put(fts_object_t *o, int winlet, fts_symbol_t s, int ac, const
fts_atom_t *at)
{
  fts_iiwu_synth_t *this = (fts_iiwu_synth_t *)o;
  fts_dsp_descr_t* dsp = (fts_dsp_descr_t *)fts_get_ptr(at);
  fts_atom_t a[4];
  
  fts_set_ptr(a + 0, this);
  fts_set_symbol(a + 1, fts_dsp_get_output_name(dsp, 0));
  fts_set_symbol(a + 2, fts_dsp_get_output_name(dsp, 1));
  fts_set_int(a + 3, fts_dsp_get_output_size(dsp, 0));
  
  fts_dsp_add_function(fts_iiwu_synth_symbol, 4, a);
}

static void
fts_iiwu_synth_ftl(fts_word_t *argv)
{
  fts_iiwu_synth_t *this = (fts_iiwu_synth_t *)fts_word_get_ptr(argv + 0);
  float *out_left = (float *)fts_word_get_ptr(argv + 1);
  float *out_right = (float *)fts_word_get_ptr(argv + 2);
  int n = fts_word_get_int(argv + 3);
  int i, j;
  
  /* FIXME: should not test here but in init */
  if (this->synth == NULL) {
    return;
  }

  iiwu_synth_write_lr(this->synth, n, out_left, 0, 1, out_right, 0, 1); 

}

/************************************************************
 *
 *  MIDI port interface methods
 *
 */

static void
fts_iiwu_synth_send_note(fts_object_t *o, int channel, int number, int value, double time)
{
  fts_iiwu_synth_t *this = (fts_iiwu_synth_t *)o;

  if (this->synth != NULL) {
    iiwu_synth_noteon(this->synth, channel - 1, number, value);
  }
}

static void
fts_iiwu_synth_send_poly_pressure(fts_object_t *o, int channel, int number, int value, double time)
{
  fts_iiwu_synth_t *this = (fts_iiwu_synth_t *)o;

  /* FIXME */
}

static void
fts_iiwu_synth_send_control_change(fts_object_t *o, int channel, int number, int value, double time)
{
  fts_iiwu_synth_t *this = (fts_iiwu_synth_t *)o;

  if (this->synth != NULL) {
    iiwu_synth_cc(this->synth, channel - 1, number, value);
  }
}

static void
fts_iiwu_synth_send_program_change(fts_object_t *o, int channel, int value, double time)
{
  fts_iiwu_synth_t *this = (fts_iiwu_synth_t *)o;

  if (this->synth != NULL) {
    iiwu_synth_program_change(this->synth, channel - 1, value);
  }
}

static void
fts_iiwu_synth_send_channel_pressure(fts_object_t *o, int channel, int value, double time)
{
  fts_iiwu_synth_t *this = (fts_iiwu_synth_t *)o;

  /* FIXME */
}

static void
fts_iiwu_synth_send_pitch_bend(fts_object_t *o, int channel, int value, double time)
{
  fts_iiwu_synth_t *this = (fts_iiwu_synth_t *)o;

  if (this->synth != NULL) {
    iiwu_synth_pitch_bend(this->synth, channel - 1, value);
  }
}

static void
fts_iiwu_synth_send_system_exclusive_byte(fts_object_t *o, int value)
{
  fts_iiwu_synth_t *this = (fts_iiwu_synth_t *)o;

}

static void
fts_iiwu_synth_send_system_exclusive_flush(fts_object_t *o, double time)
{
  fts_iiwu_synth_t *this = (fts_iiwu_synth_t *)o;

}

static fts_midiport_output_functions_t fts_iiwu_synth_output_functions =
{
  fts_iiwu_synth_send_note,
  fts_iiwu_synth_send_poly_pressure,
  fts_iiwu_synth_send_control_change,
  fts_iiwu_synth_send_program_change,
  fts_iiwu_synth_send_channel_pressure,
  fts_iiwu_synth_send_pitch_bend,
  fts_iiwu_synth_send_system_exclusive_byte,
  fts_iiwu_synth_send_system_exclusive_flush,
};

/************************************************************
 *
 *  user methods
 *
 */

static void
fts_iiwu_synth_load(fts_object_t *o, int winlet, fts_symbol_t s, int ac, const fts_atom_t *at)
{ 
  char path[512]; /* big enough? */
  fts_iiwu_synth_t *this = (fts_iiwu_synth_t *)o;

  if (this->synth == NULL) {
    return;
  }

  while ((ac > 0) && fts_is_symbol(at))
    {
      fts_symbol_t name = fts_get_symbol(at);
      const char* s = fts_symbol_name(name);

      /* check for a file on the search path first */
      if (fts_file_search_in_path(s, fts_symbol_name(fts_get_search_path()), path)) {
	post("iiwusynth loading %s....\n", path);
	iiwu_synth_sfload(this->synth, path);
      } else {
	post("iiwusynth loading %s....\n", s);
	iiwu_synth_sfload(this->synth, s);
      }

      ac--;
      at++;
    }
}

static void
fts_iiwu_synth_panic(fts_object_t *o, int winlet, fts_symbol_t s, int ac, const fts_atom_t *at)
{ 
  fts_iiwu_synth_t *this = (fts_iiwu_synth_t *)o;
  int c, n;
  
  for(c=1; c<=16; c++)
    for(n=0; n<=127; n++)
      iiwu_synth_noteoff(this->synth, c, n);
}

/************************************************************
 *
 *  get variable
 *
 */
static void
fts_iiwu_synth_get_state(fts_daemon_action_t action, fts_object_t *o, fts_symbol_t property, fts_atom_t *value)
{
  fts_set_object(value, o);
}

/************************************************************
 *
 *  class
 *
 */

static void
fts_iiwu_synth_init(fts_object_t *o, int winlet, fts_symbol_t s, int ac, const fts_atom_t *at)
{ 
  fts_iiwu_synth_t *this = (fts_iiwu_synth_t *)o;
  fts_midiport_t *port = (fts_midiport_t *)o;

  this->synth = new_iiwu_synth(128, 0);

  if (this->synth == NULL) {
    /* FIXME */
    post("iiwusynth: couldn't create synth\n");
    return;
  }

  ac--;
  at++;

  while (ac > 0)
    {
      if(fts_is_symbol(at))
	fts_iiwu_synth_load(o, 0, 0, ac, at);
      else
	{
	  fts_object_set_error(o, "Wrong argument");    
	  return;
	}
      ac--;
      at++;
    }

  fts_midiport_init(port);
  fts_midiport_set_output(port, &fts_iiwu_synth_output_functions);

  fts_dsp_add_object(o);
}

static void
fts_iiwu_synth_delete(fts_object_t *o, int winlet, fts_symbol_t s, int ac, const fts_atom_t *at)
{ 
  fts_iiwu_synth_t *this = (fts_iiwu_synth_t *)o;

  if (this->synth != NULL) {
    delete_iiwu_synth(this->synth);
    this->synth = NULL;
  }

  fts_dsp_remove_object(o);
}

static fts_status_t
fts_iiwu_synth_instantiate(fts_class_t *cl, int ac, const fts_atom_t *at)
{
  fts_class_init(cl, sizeof(fts_iiwu_synth_t), 1, 2, 0);

  fts_midiport_class_init(cl);

  fts_method_define_varargs(cl, fts_SystemInlet, fts_s_init, fts_iiwu_synth_init);
  fts_method_define_varargs(cl, fts_SystemInlet, fts_s_delete, fts_iiwu_synth_delete);
  fts_method_define_varargs(cl, fts_SystemInlet, fts_s_put, fts_iiwu_synth_put);

  /* define variable */
  fts_class_add_daemon(cl, obj_property_get, fts_s_state, fts_iiwu_synth_get_state);

  /* user method */
  fts_method_define_varargs(cl, 0, fts_new_symbol("load"), fts_iiwu_synth_load);
  fts_method_define_varargs(cl, 0, fts_new_symbol("panic"), fts_iiwu_synth_panic);

  /* dsp */
  fts_dsp_declare_function(fts_iiwu_synth_symbol, fts_iiwu_synth_ftl);
  fts_dsp_declare_outlet(cl, 0);
  fts_dsp_declare_outlet(cl, 1);

  return fts_Success;
}

void
fts_iiwu_synth_config(void)
{
  fts_iiwu_synth_symbol = fts_new_symbol("iiwusynth~");
  fts_class_install(fts_iiwu_synth_symbol, fts_iiwu_synth_instantiate);
}

