/* IIWU Synth  A soundfont synthesizer
 *
 * Copyright (C)  2001 Peter Hanappe
 * Author: Peter Hanappe, peter@hanappe.com
 *
 * This file is part of the IIWU program. 
 * IIWU is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 *
 */
#ifndef _IIWU_MIDI_H
#define _IIWU_MIDI_H

#include "iiwusynth_priv.h"

typedef struct _iiwu_midi_event_t iiwu_midi_event_t;
typedef struct _iiwu_midi_parser_t iiwu_midi_parser_t;
typedef struct _iiwu_midi_driver_t iiwu_midi_driver_t;

iiwu_midi_parser_t* new_iiwu_midi_parser(void);
int delete_iiwu_midi_parser(iiwu_midi_parser_t* parser);
iiwu_midi_event_t* iiwu_midi_parser_parse(iiwu_midi_parser_t* parser, unsigned char c);

int iiwu_midi_send_event(iiwu_synth_t* synth, iiwu_player_t* player, iiwu_midi_event_t* evt);


/***************************************************************
 *
 *                   CONSTANTS & ENUM
 */


#define MAX_NUMBER_OF_TRACKS 128

enum iiwu_midi_event_type {
  /* channel messages */
  NOTE_OFF = 0x80,
  NOTE_ON = 0x90,
  KEY_PRESSURE = 0xa0,
  CONTROL_CHANGE = 0xb0,
  PROGRAM_CHANGE = 0xc0,
  CHANNEL_PRESSURE = 0xd0,
  PITCH_BEND = 0xe0,
  /* system exclusive */
  MIDI_SYSEX = 0xf0,
  /* system common - never in midi files */
  MIDI_TIME_CODE = 0xf1,
  MIDI_SONG_POSITION = 0xf2,
  MIDI_SONG_SELECT = 0xf3,
  MIDI_TUNE_REQUEST = 0xf6,
  MIDI_EOX = 0xf7,
  /* system real-time - never in midi files */
  MIDI_SYNC = 0xf8,
  MIDI_START = 0xfa,
  MIDI_CONTINUE = 0xfb,
  MIDI_STOP = 0xfc,
  MIDI_ACTIVE_SENSING = 0xfe,
  MIDI_SYSTEM_RESET = 0xff,
  /* meta event - for midi files only */
  MIDI_META_EVENT = 0xff
};

enum iiwu_midi_control_change {
  BANK_SELECT_MSB = 0x00,
  MODULATION_MSB = 0x01,
  BREATH_MSB = 0x02,
  FOOT_MSB = 0x04,
  PORTAMENTO_TIME_MSB = 0x05,
  DATA_ENTRY_MSB = 0x06,
  VOLUME_MSB = 0x07,
  BALANCE_MSB = 0x08,
  PAN_MSB = 0x0A,
  EXPRESSION_MSB = 0x0B,
  EFFECTS1_MSB = 0x0C,
  EFFECTS2_MSB = 0x0D,
  GPC1_MSB = 0x10, /* general purpose controller */
  GPC2_MSB = 0x11,
  GPC3_MSB = 0x12,
  GPC4_MSB = 0x13,
  BANK_SELECT_LSB = 0x20,
  MODULATION_WHEEL_LSB = 0x21,
  BREATH_LSB = 0x22,
  FOOT_LSB = 0x24,
  PORTAMENTO_TIME_LSB = 0x25,
  DATA_ENTRY_LSB = 0x26,
  VOLUME_LSB = 0x27,
  BALANCE_LSB = 0x28,
  PAN_LSB = 0x2A,
  EXPRESSION_LSB = 0x2B,
  EFFECTS1_LSB = 0x2C,
  EFFECTS2_LSB = 0x2D,
  GPC1_LSB = 0x30,
  GPC2_LSB = 0x31,
  GPC3_LSB = 0x32,
  GPC4_LSB = 0x33,
  SUSTAIN_SWITCH = 0x40,
  PORTAMENTO_SWITCH = 0x41,
  SOSTENUTO_SWITCH = 0x42,
  SOFT_PEDAL_SWITCH = 0x43,
  LEGATO_SWITCH = 0x45,
  HOLD2_SWITCH = 0x45,
  SOUND_CTRL1 = 0x46,
  SOUND_CTRL2 = 0x47,
  SOUND_CTRL3 = 0x48,
  SOUND_CTRL4 = 0x49,
  SOUND_CTRL5 = 0x4A,
  SOUND_CTRL6 = 0x4B,
  SOUND_CTRL7 = 0x4C,
  SOUND_CTRL8 = 0x4D,
  SOUND_CTRL9 = 0x4E,
  SOUND_CTRL10 = 0x4F,
  GPC5 = 0x50,
  GPC6 = 0x51,
  GPC7 = 0x52,
  GPC8 = 0x53,
  PORTAMENTO_CTRL = 0x54,
  EFFECTS_DEPTH1 = 0x5B,
  EFFECTS_DEPTH2 = 0x5C,
  EFFECTS_DEPTH3 = 0x5D,
  EFFECTS_DEPTH4 = 0x5E,
  EFFECTS_DEPTH5 = 0x5F,
  DATA_ENTRY_INCR = 0x60,
  DATA_ENTRY_DECR = 0x61,
  NRPN_LSB = 0x62,
  NRPN_MSB = 0x63,
  RPN_LSB = 0x64,
  RPN_MSB = 0x65,
  ALL_SOUND_OFF = 0x78,
  RESET_ALL = 0x79,
  LOCAL_CONTROL = 0x7A,
  ALL_NOTES_OFF = 0x7B,
  OMNI_OFF = 0x7C,
  OMNI_ON = 0x7D,
  POLY_OFF = 0x7E,
  POLY_ON = 0x7F
};

enum midi_meta_event {
  MIDI_COPYRIGHT = 0x02,
  MIDI_TRACK_NAME = 0x03,
  MIDI_INST_NAME = 0x04,
  MIDI_LYRIC = 0x05,
  MIDI_MARKER = 0x06,
  MIDI_CUE_POINT = 0x07,
  MIDI_EOT = 0x2f,
  MIDI_SET_TEMPO = 0x51,
  MIDI_SMPTE_OFFSET = 0x54,
  MIDI_TIME_SIGNATURE = 0x58,
  MIDI_KEY_SIGNATURE = 0x59,
  MIDI_SEQUENCER_EVENT = 0x7f
};

enum iiwu_player_status 
{
  IIWU_PLAYER_READY,
  IIWU_PLAYER_PLAYING,
  IIWU_PLAYER_DONE
};

enum iiwu_driver_status 
{
  IIWU_MIDI_READY,
  IIWU_MIDI_LISTENING,
  IIWU_MIDI_DONE
};

/***************************************************************
 *
 *         TYPE DEFINITIONS & FUNCTION DECLARATIONS
 */

/* From ctype.h */
#define iiwu_isascii(c)    (((c) & ~0x7f) == 0)  


/*
 * iiwu_timer
 */
typedef struct _iiwu_timer iiwu_timer;

/* if the callback function returns 1 the timer will continue; if it
   returns 0 it will stop */
typedef int (*iiwu_timer_callback)(void* data, long msec);

iiwu_timer* new_iiwu_timer(int msec, iiwu_timer_callback callback, void* data);
int delete_iiwu_timer(iiwu_timer* timer);
int iiwu_timer_join(iiwu_timer* timer);
int iiwu_timer_stop(iiwu_timer* timer);


/*
 * iiwu_midi_event_t
 */
struct _iiwu_midi_event_t {
  long dtime;
  int type;
  int channel;
  int param1;
  int param2;
  iiwu_midi_event_t* next;
};

iiwu_midi_event_t* new_iiwu_midi_event(void);
int delete_iiwu_midi_event(iiwu_midi_event_t* event);
int iiwu_midi_event_set_type(iiwu_midi_event_t* evt, int type);
int iiwu_midi_event_set_channel(iiwu_midi_event_t* evt, int chan);
int iiwu_midi_event_set_param1(iiwu_midi_event_t* evt, int v);
int iiwu_midi_event_set_param2(iiwu_midi_event_t* evt, int v);
int iiwu_midi_event_get_type(iiwu_midi_event_t* evt);
int iiwu_midi_event_get_channel(iiwu_midi_event_t* evt);
int iiwu_midi_event_get_param1(iiwu_midi_event_t* evt);
int iiwu_midi_event_get_param2(iiwu_midi_event_t* evt);


/*
 * iiwu_track
 */
struct _iiwu_track {
  char* name;
  iiwu_midi_event_t *first;
  iiwu_midi_event_t *cur;
  iiwu_midi_event_t *last;
  long ticks;
};

typedef struct _iiwu_track iiwu_track;

iiwu_track* new_iiwu_track(void);
int delete_iiwu_track(iiwu_track* track);
int iiwu_track_set_name(iiwu_track* track, char* name);
char* iiwu_track_get_name(iiwu_track* track);
int iiwu_track_add_event(iiwu_track* track, iiwu_midi_event_t* evt);
iiwu_midi_event_t* iiwu_track_first_event(iiwu_track* track);
iiwu_midi_event_t* iiwu_track_next_event(iiwu_track* track);
int iiwu_track_get_duration(iiwu_track* track);
int iiwu_track_reset(iiwu_track* track);
int iiwu_track_send_events(iiwu_track* track, 
			   iiwu_synth_t* synth,
			   iiwu_player_t* player,
			   long ticks);

#define iiwu_track_eot(track)  ((track)->cur == NULL)


/*
 * iiwu_player
 */
struct _iiwu_player_t {
  int status;
  int loop;
  int ntracks;
  iiwu_track *track[MAX_NUMBER_OF_TRACKS];
  iiwu_synth_t* synth;
  iiwu_timer* timer;
  char send_program_change; /* should we ignore the program changes? */
  int ticks_passed;         /* number of midi ticks that have passed */
  int msec_passed;          /* number of milliseconds that have passed */
  int miditempo;            /* as indicated by MIDI SetTempo: n 24th of a usec per midi-clock. bravo! */
  double deltatime;         /* milliseconds per midi tick. depends on set-tempo */
  unsigned int division;
};

int iiwu_player_add_track(iiwu_player_t* player, iiwu_track* track);
int iiwu_player_callback(void* data, long msec);


/*
 * iiwu_midi_file
 */
typedef struct {
  iiwu_file fp;
  int running_status;
  int c;
  int type;
  int ntracks;
  int uses_smpte;
  unsigned int smpte_fps;
  unsigned int smpte_res;
  unsigned int division;       /* If uses_SMPTE == 0 then division is 
				  ticks per beat (quarter-note) */
  double tempo;                /* Beats per second (SI rules =) */
  int tracklen;
  int trackpos;
  int eot;
  int varlen;
} iiwu_midi_file;

iiwu_midi_file* new_iiwu_midi_file(char* filename);
void delete_iiwu_midi_file(iiwu_midi_file* mf);
int iiwu_midi_file_read_mthd(iiwu_midi_file* midifile);
int iiwu_midi_file_load_tracks(iiwu_midi_file* midifile, iiwu_player_t* player);
int iiwu_midi_file_read_track(iiwu_midi_file* mf, iiwu_player_t* player);
int iiwu_midi_file_read_event(iiwu_midi_file* mf, iiwu_track* track);
int iiwu_midi_file_read_varlen(iiwu_midi_file* mf);
int iiwu_midi_file_getc(iiwu_midi_file* mf);
int iiwu_midi_file_push(iiwu_midi_file* mf, int c);
int iiwu_midi_file_read(iiwu_midi_file* mf, void* buf, int len);
int iiwu_midi_file_skip(iiwu_midi_file* mf, int len);
int iiwu_midi_file_read_tracklen(iiwu_midi_file* mf);
int iiwu_midi_file_eot(iiwu_midi_file* mf);
int iiwu_midi_file_get_division(iiwu_midi_file* midifile);


/*
 * iiwu_midi_parser_t
 */
struct _iiwu_midi_parser_t {
  iiwu_midi_event_t event;
  unsigned char status;
  unsigned char channel;
  unsigned char running_status;
  unsigned short p1;
  unsigned char p2;
};

int iiwu_isasciistring(char* s);
long iiwu_getlength(unsigned char *s);


/*
 * iiwu_midi_handler
 */
struct _iiwu_midi_handler_t {
  iiwu_synth_t* synth;
  char* driver_name;
  char* device_name;
  iiwu_midi_driver_t* driver;
  iiwu_midi_driver_t* (*new_driver)(iiwu_midi_handler_t* handler);
  int (*delete_driver)(iiwu_midi_driver_t* p);
  int (*join_driver)(iiwu_midi_driver_t* p);
  int (*status_driver)(iiwu_midi_driver_t* p);
  unsigned int noteon_count;
  unsigned int noteoff_count;
  unsigned int prog_count;
  unsigned int pbend_count;
  unsigned int error_count;
};

int iiwu_midi_handler_send_event(iiwu_midi_handler_t* handler, iiwu_midi_event_t* event);
void iiwu_midi_handler_set_device_name(iiwu_midi_handler_t* handler, char* name);


/*
 * iiwu_mdriver_definition
 */
struct iiwu_mdriver_definition_t {
  char* name;
  char* description;
  iiwu_midi_driver_t* (*new_driver)(iiwu_midi_handler_t* handler);
  int (*delete_driver)(iiwu_midi_driver_t* p);
  int (*join_driver)(iiwu_midi_driver_t* p);
  int (*status_driver)(iiwu_midi_driver_t* p);
};


#endif /* _IIWU_MIDI_H */
