/***************************************
  $Header: /cvsroot/petscgraphics/render.c,v 1.5 2004/08/17 15:05:56 hazelsct Exp $

  This file contains the rendering code for Illuminator, which renders 2-D or
  3-D data into an RGB(A) unsigned char array (using perspective in 3-D).
***************************************/

#include "illuminator.h"


#undef __FUNCT__
#define __FUNCT__ "pseudocolor"

/*++++++++++++++++++++++++++++++++++++++
  This little function converts a scalar value into an rgb color from red to
  blue.

  PetscScalar val Value to convert.

  PetscScalar* minmax Array with minimum and maximum values in which to scale
  val.

  guchar *pixel Address in rgb buffer where this pixel should be painted.
  ++++++++++++++++++++++++++++++++++++++*/

static inline void pseudocolor (PetscScalar val, PetscScalar* minmax,
				guchar *pixel)
{
  PetscScalar shade = (val - minmax[0]) / (minmax[1] - minmax[0]);
  /* Old stuff *
  if (shade < 0.2) {      /* red -> yellow *
    *pixel++ = 255;
    *pixel++ = 1275*shade;
    *pixel++ = 0; }
  else if (shade < 0.4) { /* yellow -> green *
    *pixel++ = 510-1275*shade;
    *pixel++ = 255;
    *pixel++ = 0; }
  else if (shade < 0.6) { /* green -> cyan *
    *pixel++ = 0;
    *pixel++ = 255;
    *pixel++ = 1275*shade-510; }
  else if (shade < 0.8) { /* cyan -> blue *
    *pixel++ = 0;
    *pixel++ = 1020-1275*shade;
    *pixel++ = 255; }
  else {                  /* blue -> magenta *
    *pixel++ = 1275*shade-1020;
    *pixel++ = 0;
    *pixel++ = 255; }
  /* New stuff */
  if (shade < 0.25) {      /* red -> yellow */
    *pixel++ = 255;
    *pixel++ = 1020*shade;
    *pixel++ = 0; }
  else if (shade < 0.5) { /* yellow -> green */
    *pixel++ = 510-1020*shade;
    *pixel++ = 255;
    *pixel++ = 0; }
  else if (shade < 0.75) { /* green -> cyan */
    *pixel++ = 0;
    *pixel++ = 255;
    *pixel++ = 1020*shade-510; }
  else { /* cyan -> blue */
    *pixel++ = 0;
    *pixel++ = 1020-1020*shade;
    *pixel++ = 255; }
}


#undef __FUNCT__
#define __FUNCT__ "pseudohueintcolor"

/*++++++++++++++++++++++++++++++++++++++
  This little function converts a vector into an rgb color with hue indicating
  direction (green, yellow, red, blue at 0, 90, 180, 270 degrees) and intensity
  indicating magnitude relative to reference magnitude in minmax[1].

  PetscScalar vx Vector's
  +latex+$x$-component.
  +html+ <i>x</i>-component.

  PetscScalar vy Vector's
  +latex+$y$-component.
  +html+ <i>y</i>-component.

  PetscScalar *minmax Array whose second entry has the reference magnitude.

  guchar *pixel Address in rgb buffer where this pixel should be painted.
  ++++++++++++++++++++++++++++++++++++++*/

static inline void pseudohueintcolor
(PetscScalar vx, PetscScalar vy, PetscScalar *minmax, guchar *pixel)
{
  PetscScalar mag=sqrt(vx*vx+vy*vy), theta=atan2(vy,vx), red, green, blue;
  if (minmax[1] <= 0.)
    {
      *pixel = *(pixel+1) = *(pixel+2) = 0;
      return;
    }
  mag = (mag > minmax[1]) ? 1.0 : mag/minmax[1];

  red = 2./M_PI * ((theta<-M_PI/2.) ? -M_PI/2.-theta :
		   ((theta<0.) ? 0. : ((theta<M_PI/2.) ? theta : M_PI/2.)));
  green = 2./M_PI * ((theta<-M_PI/2.) ? 0. :
		     ((theta<0.) ? theta+M_PI/2. :
		      ((theta<M_PI/2.) ? M_PI/2. : M_PI-theta)));
  blue = 2./M_PI * ((theta<-M_PI/2.) ? theta+M_PI :
		    ((theta<0.) ? -theta : 0.));

  *pixel++ = 255*mag*red;
  *pixel++ = 255*mag*green;
  *pixel++ = 255*mag*blue;
}

#undef __FUNCT__
#define __FUNCT__ "pseudoternarycolor"

/*++++++++++++++++++++++++++++++++++++++
  This little function converts two ternary fractions into an rgb color, with
  yellow, cyan and magenta indicating the corners.

  PetscScalar A First ternary fraction.

  PetscScalar B Second ternary fraction.

  PetscScalar *minmax Array first and second ternary fractions of each of the
  three corner values for scaling.

  guchar *pixel  Address in rgb buffer where this pixel should be painted.
  ++++++++++++++++++++++++++++++++++++++*/

static inline void pseudoternarycolor
(PetscScalar A, PetscScalar B, PetscScalar *minmax, guchar *pixel)
{
  PetscScalar x1, x2, inverse_det;

  /* Transform A,B into x1,x2 based on corners */
  inverse_det = 1./((minmax[2]-minmax[0])*(minmax[5]-minmax[1]) -
		    (minmax[4]-minmax[0])*(minmax[3]-minmax[1]));
  x1 = ((A-minmax[0])*(minmax[5]-minmax[1]) -
	(B-minmax[1])*(minmax[4]-minmax[0])) * inverse_det;
  x2 = ((B-minmax[1])*(minmax[2]-minmax[0]) -
	(A-minmax[0])*(minmax[3]-minmax[1])) * inverse_det;

  /* Now colorize */
  *pixel++ = 255*(1.-x1);
  *pixel++ = 255*(x1+x2);
  *pixel++ = 255*(1.-x2);
}


#undef __FUNCT__
#define __FUNCT__ "render_rgb_local_2d"

/*++++++++++++++++++++++++++++++++++++++
  Render data from global_array into local part of an RGB buffer.  When running
  in parallel, these local buffers should be collected and layered to produce
  the full image.

  int render_rgb_local_2d Returns zero or an error code.

  guchar *rgb RGB buffer in which to render.

  int rwidth Total width of the RGB buffer.

  int rheight Total height of the RGB buffer.

  int bytes_per_pixel Number of bytes per pixel in this RGB buffer (typically 3
  or 4).

  PetscScalar *global_array Local array of global vector data to render.

  int num_fields Number of field variables in the array.

  int display_field The (first) field we are rendering now.

  field_plot_type fieldtype The type of this field.

  PetscScalar *minmax Array of minimum and maximum values to pass to the
  various pseudocolor functions; if NULL, call minmax_scale to determine those
  values.

  int nx Width of the array.

  int ny Height of the array.

  int xs Starting
  +latex+$x$-coordinate
  +html+ <i>x</i>-coordinate
  of the local part of the global vector.

  int ys Starting
  +latex+$y$-coordinate
  +html+ <i>y</i>-coordinate
  of the local part of the global vector.

  int xm Width of the local part of the global vector.

  int ym Height of the local part of the global vector.
  ++++++++++++++++++++++++++++++++++++++*/

int render_rgb_local_2d
(guchar *rgb, int rwidth, int rheight, int bytes_per_pixel,
 PetscScalar *global_array, int num_fields, int display_field,
 field_plot_type fieldtype, PetscScalar *minmax, int nx,int ny, int xs,int ys,
 int xm,int ym)
{
  int ix,iy, ierr;
  PetscScalar local_minmax[6];

  /* Determine default min and max if none are provided */
  if (minmax == NULL)
    {
      minmax = local_minmax;
      ierr = minmax_scale (global_array, xm*ym, num_fields, display_field,
			   fieldtype, 2, minmax); CHKERRQ (ierr);
    }

  /* Do the rendering (note switch in inner loop, gotta get rid of that) */
  for (iy=rheight*ys/ny; iy<rheight*(ys+ym)/ny; iy++)
    for (ix=rwidth*xs/nx; ix<rwidth*(xs+xm)/nx; ix++)
      {
	int vecindex = (((rheight-iy-1)*ny/rheight)*nx +
			ix*nx/rwidth)*num_fields + display_field;
	guchar *pixel = rgb + bytes_per_pixel*(iy*rwidth + ix);

	switch (fieldtype)
	  {
	  case FIELD_SCALAR:
	  case FIELD_SCALAR+1:
	    {
	      pseudocolor (global_array [vecindex], minmax, pixel);
	      break;
	    }
	  case FIELD_VECTOR:
	  case FIELD_VECTOR+1:
	    {
	      pseudohueintcolor
		(global_array [vecindex], global_array [vecindex+1], minmax,
		 pixel);
	      break;
	    }
	  case FIELD_TERNARY:
	    {
	      pseudoternarycolor
		(global_array [vecindex], global_array [vecindex+1], minmax,
		 pixel);
	      break;
	    }
	  default:
	    SETERRQ (PETSC_ERR_ARG_OUTOFRANGE, "Field type not yet supported");
	  }
      }
  return 0;
}


/*++++++++++++++++++++++++++++++++++++++
  Render triangle data into an RGB buffer.  When called in parallel, the
  resulting images should be layered to give the complete picture.  Zooming is
  done by adjusting the ratio of the dir vector to the right vector.

  int render_rgb_local_3d Returns zero or an error code.

  guchar *rgb RGB buffer in which to render.

  int rwidth Total width of the RGB buffer.

  int rheight Total height of the RGB buffer.

  int bytes_per_pixel Number of bytes per pixel in this RGB buffer (typically 3
  or 4).

  int num_triangles Number of triangles to render.

  PetscScalar *vertices Table of coordinates (x1,y1,z1, x2,y2,z2, x3,y3,z3) and
  colors (RGBA 0-1) making thirteen values per triangle.

  PetscScalar *eye Point from where we're looking (x,y,z).

  PetscScalar *dir Direction we're looking (x,y,z).

  PetscScalar *right Rightward direction in physical space (x,y,z).
  ++++++++++++++++++++++++++++++++++++++*/

int render_rgb_local_3d
(guchar *rgb, int rwidth, int rheight, int bytes_per_pixel,
 int num_triangles, PetscScalar *vertices,
 PetscScalar *eye, PetscScalar *dir, PetscScalar *right)
{
  SETERRQ (PETSC_ERR_SUP, "3-D rendering is not yet ready");
}
