#include <stdio.h>
#include <stdlib.h>
#include <jpeglib.h>
#include <setjmp.h>
#include "common.h"
#include "ftplib.h"
#include "transmogrify.h"

char *stamperror[] = {
  "Exited successfully.",
  "Can't open infile: File Not Found or File Error!", 
  "Can't open outfile: File Creation Error!",
  "Cannot find your home directory!",
  "Invalid color component (greater than 255 or less than 0).",
  "Invalid shade rate (greater than 255 or less than 0).", 
  "Invalid JPEG quality (greater than 100 or less than 0).",
  "You must specify a font to use for the upper string.",
  "You must specify a font to use for the lower string.",
  "You must specify both an input and output JPEG image.",
  "You cannot specify colors when using a grayscale image.",
  "Cannot read exec results file.",
  "Text of upper string is too large.",
  "Text of lower string is too large.",
  "Cannot read upper font file.",
  "Cannot read lower font file.",
  "ftp: Could not open connection to server.", 
  "ftp: Could not login to server.",
  "ftp: Could not change to destination directory.",
  "ftp: Could not upload file to server.",
  "ftp: Could not rename temporary file.",
  "jpeglib: A misc. jpeglib error occured.",
} ;

METHODDEF(void)
stamp_jpeg_error_exit (j_common_ptr cinfo)
{
  /* cinfo->err really points to a my_error_mgr struct, so coerce pointer */
  stamp_jpeg_error_ptr myerr = (stamp_jpeg_error_ptr) cinfo->err;

  /* Always display the message. */
  /* We could postpone this until after returning, if we chose. */
  /* ebw - postpone it, since stamp handles this. 
     (*cinfo->err->output_message) (cinfo);
  */
  
  /* Return control to the setjmp point */
  longjmp(myerr->setjmp_buffer, 1);
}


void convert_to_color(struct ImageStruct *Image){
  int i,j;
  struct ScanLine *scanline;
  JSAMPLE *line;
  
  scanline = Image->FirstLine;
  for(j = 0; j<Image->image_height; j++){
    line = malloc(sizeof(JSAMPLE) * Image->image_width * 3) ;
    for(i = 0; i<Image->image_width; i++){
      line[3*i] = scanline->line[i];
      line[(3*i)+1] = scanline->line[i];
      line[(3*i)+2] = scanline->line[i];
    }
    free(scanline->line);
    scanline->line = line;
    scanline = scanline->next;
  } 
}

int JPG_Load(char *filename, struct ImageStruct *Image) {
  struct ScanLine *tmp ;
  JSAMPLE *buffer ;
  struct jpeg_decompress_struct cinfo ;  
  struct stamp_jpeg_error_mgr jerr ;
  FILE *infile ;
  int i, len, r;
  
  Image->FirstLine = NULL;
  Image->LastLine = NULL;
  
  Image->OldFirstLine = NULL;
  Image->OldLastLine = NULL;
  
  cinfo.err = jpeg_std_error(&jerr.pub) ;
  jerr.pub.error_exit = stamp_jpeg_error_exit;
 
  jpeg_create_decompress(&cinfo) ;

  if (setjmp(jerr.setjmp_buffer)) {
    /* If we get here, the JPEG code has signaled an error.
     * We need to clean up the JPEG object, close the input file, and return.
     */
    jpeg_destroy_decompress(&cinfo);
    fclose(infile);
    return (ERR_JPEGLIB);
  }

  if ((infile = fopen(filename, "rb"))  == NULL) { 
    return(ERR_INFILE) ;
  }

  jpeg_stdio_src(&cinfo,infile) ;
  jpeg_read_header(&cinfo, FALSE) ;
  jpeg_start_decompress(&cinfo) ;
  
  Image->image_width = cinfo.image_width ;
  Image->image_height = cinfo.image_height ;

  while (cinfo.output_scanline < cinfo.output_height) {
    if (Image->FirstLine == NULL) {
      
      tmp = malloc(sizeof(JSAMPLE) *
		   cinfo.output_width *
		   cinfo.output_components) ;  	  
      tmp->next = NULL ;
      Image->LastLine = tmp ;
      Image->FirstLine = Image->LastLine;
    } else {
      tmp = malloc(sizeof(JSAMPLE) *
		   cinfo.output_width *
		   cinfo.output_components) ;  	  
      tmp->next = NULL ;
      Image->LastLine->next  = tmp ; 	  
      Image->LastLine = tmp;
    }
    
    buffer = malloc(sizeof(JSAMPLE) * 
		    cinfo.output_width * 
		    cinfo.output_components) ;
    
    jpeg_read_scanlines(&cinfo, &buffer, 1) ;
    
    tmp->line = buffer ;
  }  
  
  jpeg_finish_decompress(&cinfo) ;
  fclose(infile) ;
  /* if it's b/w, make it color. */
  if(  cinfo.output_components == 1){
    convert_to_color(Image);
  }
  return(0) ;
}

int JPG_Save(char *filename, struct ImageStruct *Image,
	     struct ArgStruct *Args) {
  struct ScanLine *tmp ;
  struct jpeg_compress_struct cinfo ;
  struct jpeg_error_mgr jerr ;
  FILE *outfile ;
  
  cinfo.err  =  jpeg_std_error(&jerr) ;
  jpeg_create_compress(&cinfo) ;
  
  if ((outfile  =  fopen(filename,"wb")) == NULL) {
    return(ERR_OUTFILE) ;
  }
  
  jpeg_stdio_dest(&cinfo, outfile) ;
  
  cinfo.image_width = Image->image_width ;
  cinfo.image_height = Image->image_height ;
  cinfo.input_components = 3;
  cinfo.in_color_space = JCS_RGB ;
  
  jpeg_set_defaults(&cinfo) ;

  jpeg_default_colorspace(&cinfo) ;
  jpeg_set_quality(&cinfo, Args->Quality, TRUE) ;


  jpeg_start_compress(&cinfo,TRUE) ;
 
  tmp = Image->FirstLine ;

  while(cinfo.next_scanline < cinfo.image_height) {
    jpeg_write_scanlines(&cinfo, &tmp->line,1) ;
    tmp = tmp->next ;
  }  
  
  jpeg_finish_compress(&cinfo) ; 
  
  fclose(outfile) ;

  return(0) ;
}

void JPG_Free(struct ImageStruct *jpg){
  struct ScanLine *scanline, *tmps;
  int i;
  
  scanline = jpg->FirstLine;
  for (i = 0; i< jpg->image_height; i++){ 
    tmps = scanline;
    scanline = scanline->next;      
    free(tmps->line);
    free(tmps);
  }
  free(jpg);
}

int Transfer_File(struct ArgStruct *Args) {
  char *tmp ;

  tmp = malloc(strlen(Args->Upload) +
	       strlen(".tmp") + 1) ; 
  strcpy(tmp, Args->Upload) ;
  strcat(tmp,".tmp") ;
  
  if (!ftpOpen(Args->Host)) {
    return(ERR_FTPCONNECT) ;
  } else if (!ftpLogin(Args->Login, Args->Passwd)) {
    return(ERR_FTPLOGIN) ;
  } else if (!ftpChdir(Args->Path)) {
    return(ERR_FTPCHDIR) ;
  }

  if (!ftpPut(Args->Outfile, tmp, 'I')) {
    return(ERR_FTPPUT) ;
  } else if (!ftpRename(tmp, Args->Upload)) {
    return(ERR_FTPREN) ;
  } else {
    ftpQuit() ; 
    free(tmp);
    return(0) ;
  }
  return 0;
}

int stamp_main(struct ArgStruct *Args){
  struct ImageStruct *Image ;
  struct ScanLine *scanline, *tmps;
  FILE *output ;
  char *tmp ;
  char tmp2[255] ;
  FILE *fontfile ;
  int i, xpos ;
  char tmpfilename[255];
  int pid;
  unsigned char buf[8192] ;
  int r;

  /* ebw - use pid for filenames */
  pid = getpid();

  sprintf(tmpfilename, "/tmp/stamp.%d.tmp", pid);

  Image = malloc(sizeof(struct ImageStruct)) ;
  
  if (r = JPG_Load(Args->Infile, Image)) {
    return r;
  }
  
  if (Args->Rotate)
    JPG_Rotate_CCW(Image) ; 
  
  if (strcmp(Args->UpperString, "")) {
    
    if (Args->UStringExec) {
	
      tmp = malloc(strlen(Args->UpperString) +
		   strlen(" > ") + strlen(tmpfilename) + 1) ;
      strcpy(tmp, Args->UpperString) ;
      strcat(tmp, " > ");
      strcat(tmp, tmpfilename);
      system(tmp) ;
      free(Args->UpperString) ;
      if ((output = fopen(tmpfilename, "r")) == NULL) {
	return(ERR_BADEXEC) ;
      }
      fscanf(output, "%[^\n]", tmp2) ;
      fclose(output) ;
	
      Args->UpperString = malloc(strlen(tmp2) + 1) ;
      strcpy(Args->UpperString, tmp2) ;
    }
      
    xpos = ((Image->image_width / 2) -
	    (strlen(Args->UpperString) * (DWIDTH / 2))) ;

    if (xpos < 0) {
      return(ERR_UTOOBIG) ;
    }

    if ((fontfile = fopen(Args->UpperFont, "r")) == NULL) {
      return(ERR_BADUFONT) ;
    }
    
    for (i = 0; i < 256; i++)
      fread(&buf[i * 32], 16, 1, fontfile);

    fclose(fontfile);

    JPG_StringPut(Args->UpperString, xpos, 0, Args, Image, DHEIGHT, DWIDTH, buf) ;

  }
  
  if (strcmp(Args->LowerString, "")) {
    if (Args->LStringExec) {
      
      tmp = malloc(strlen(Args->LowerString) +
		   strlen(" > ") + strlen(tmpfilename) + 1) ;
      
      strcpy(tmp, Args->LowerString) ;
      
      strcat(tmp, " > ");
      strcat(tmp, tmpfilename);
      
      system(tmp) ;
      
      free(Args->LowerString) ;
      
      if ((output = fopen(tmpfilename, "r")) == NULL) {
	return(ERR_BADEXEC) ;
      }
      fscanf(output, "%[^\n]", tmp2) ;
      
      fclose(output) ;
      
      Args->LowerString = malloc(strlen(tmp2) + 1) ;
      strcpy(Args->LowerString, tmp2) ;
    }
    
    
    xpos = ((Image->image_width / 2) -
	    (strlen(Args->LowerString) * (DWIDTH / 2))) ;
    
    if (xpos < 0) {
      return(ERR_LTOOBIG) ;
    }
    
    if ((fontfile = fopen(Args->LowerFont, "r")) == NULL) {
      return(ERR_BADLFONT) ;
    }
    
    for (i = 0; i < 256; i++)
      fread(&buf[i * 32], 16, 1, fontfile);
    
    fclose(fontfile);
    
    JPG_StringPut(Args->LowerString, xpos, 1, Args, Image, DHEIGHT, DWIDTH, buf) ;
  }
  
  if (Args->Rotate)
    JPG_Rotate_CW(Image) ; 
  
  if (r=JPG_Save(Args->Outfile, Image, Args)){
    return r;
  }
  
  remove(tmpfilename);
  
  JPG_Free(Image);

  return(r) ;  
}

