/***************************************************************************
                         bone.cpp  -  description
                            -------------------                                         
   begin                : Wed Sep 29 999
   copyright            : (C) 1999 by Jon Anderson                         
   email                : janderson@onelink.com                                     
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#include "bone.h"
#include "vertex.h"
#include "mesh.h"

#include <objectdb.h>
#include <i3d.h>

bool Bone::draw_envelope = false;
bool Bone::draw_affected = false;
bool Bone::draw_limits = false;

int Bone::TYPE = Typed::getUID();
int Bone::mode = MODE_NORMAL;

GLUquadricObj *Bone::qobj = gluNewQuadric();
typedef multimap< Vertex *, Bone *, less< Vertex * > > VertBoneMap;
typedef VertBoneMap::value_type VertBonePair;

/** Single shared dictionary of vertex to bone mappings. */
VertBoneMap g_vert_bone_map;

Bone::Bone( Entity *b, int l )
   :
   Object( b ),
   m_effector( 0, l, 0, 1 ),
   m_envelope( .5, .5, .5, 1 ),
   m_affected(),
   m_affected_offsets(),
   m_children(),
   vXRot( 360, -360, 0, 0 ),
   vYRot( 360, -360, 0, 0 ),
   vZRot( 360, -360, 0, 0 ),
   m_box()
{
   addType( TYPE );

   // m_radius = 0.5;

   /* find the root of the skeleton. If there is a parent bone, then
    * the root is the parent's root.  If there isn't a parent, then
    * this bone is the root. (note: it's also possible to have non bone
    * parents.
    */

   if ( b != 0 && b->isA( Bone::TYPE ) )
   {
      m_root = ( ( Bone* ) b ) ->getRoot();
      ( ( Bone * ) b ) ->addChild( this );
      cerr << "Bone Parent.." << endl;
   }

   else
   {
      cerr << "No parent bone" << endl;
      m_root = this;
   }


   m_stiffness = 10;

   float r = ( float ) rand() / RAND_MAX;
   float g = ( float ) rand() / RAND_MAX;
   float b = ( float ) rand() / RAND_MAX;

   Vector4 c( r, g, b, 1 );
   m_box.setColor ( c );

   setEnvelope( m_envelope );



}

Bone::~Bone()
{
   if ( getParent() != 0 && getParent() ->isA( Bone::TYPE ) )
      ( ( Bone * ) getParent() ) ->removeChild( this );

}

void Bone::setEnvelope( Vector4 &env )
{
   m_envelope = env;

   Vector4 min( -m_envelope.x, -m_envelope.y, -m_envelope.z, 1 );
   Vector4 max( m_envelope.x, m_effector.y + m_envelope.y, m_envelope.z, 1 );

   m_box.setBounds( min, max );

}


void Bone::setParent( Entity *b )
{

   /* find the root of the skeleton. If there is a parent bone, then
    * the root is the parent's root.  If there isn't a parent, then
    * this bone is the root. (note: it's also possible to have non bone
    * parents.
    */

   parent = b;

   if ( b != 0 && b->isA( Bone::TYPE ) )
   {
      m_root = ( ( Bone* ) b ) ->getRoot();
      ( ( Bone * ) b ) ->addChild( this );
   }

   else
      m_root = this;

   //make sure all the children get re-rooted...
   list<Bone *>::iterator it = m_children.begin();

   while ( it != m_children.end() )
   {
      ( *it ) ->setRoot( m_root );
      ++it;
   }


}

void Bone::setRoot( Bone *broot )
{
   m_root = broot;
   //make sure all the children get re-rooted...
   list<Bone *>::iterator it = m_children.begin();

   while ( it != m_children.end() )
   {
      ( *it ) ->setRoot( m_root );
      ++it;
   }
}

Bone *Bone::getRoot()
{
   return m_root;
}

void Bone::addChild( Bone *b )
{
   m_children.push_back( b );
}

void Bone::removeChild( Bone *b )
{
   list<Bone *>::iterator it;
   it = find( m_children.begin(), m_children.end(), b );

   if ( it != m_children.end() )
   {
      m_children.erase( it );
   }
}

void Bone::resetAffected( bool propagate = true )
{
   //erase the current affected offsets.
   for ( int i = 0; i < ( int ) m_affected_offsets.size(); i++ )
   {
      delete m_affected_offsets[ i ];
   }

   m_affected_offsets.erase( m_affected_offsets.begin(), m_affected_offsets.end() );

   //wipe out the affected.
   clearVertBoneMapping();
   m_affected.erase( m_affected.begin(), m_affected.end() );

   if ( propagate )
   {
      list<Bone *>::iterator it = m_children.begin();

      while ( it != m_children.end() )
      {
         ( *it ) ->resetAffected( true );
         ++it;
      }
   }


}

void Bone::saveAffectedAndOffsets( bool propagate = true )
{
   //get a list of all affected vertices in the scene.
   saveAffected( propagate );
   //save the offsets
   saveAffectedOffsets( propagate );
}


void Bone::saveAffectedOffsets( bool propagate = true )
{
   Matrix44 b_inv;

   getEffectorMatrix( &b_inv );
   b_inv.Invert();

   //erase the current affected offsets.
   for ( int i = 0; i < ( int ) m_affected_offsets.size(); i++ )
   {
      delete m_affected_offsets[ i ];
   }

   m_affected_offsets.erase( m_affected_offsets.begin(), m_affected_offsets.end() );

   //for each affected
   for ( int i = 0; i < ( int ) m_affected.size(); i++ )
   {
      cerr << "Affected:" << i << endl;
      Vector4 *tp = new Vector4();

      Vertex *v = m_affected[ i ];
      v->getTransformedPosition( tp );  //transform from vert to world space.

      *tp = b_inv * *tp; //transform from world into bone space.

      //save the position of the vertex in bone space.
      m_affected_offsets.push_back( tp );

   }

   if ( propagate )
   {
      //save all the offsets for this bones children.
      list<Bone *>::iterator it = m_children.begin();

      while ( it != m_children.end() )
      {
         ( *it ) ->saveAffectedOffsets( true );
         ++it;
      }
   }

}

/** Note: vertices with more than one
  * bone are actually getting deformed twice!
  */
void Bone::deformAffected()
{


   Matrix44 v_inv;

   Matrix44 b_matrix;
   getEffectorMatrix( &b_matrix );
   b_matrix.Invert();

   Vector4 tp;
   Vector4 bp;

   for ( int i = 0; i < ( int ) m_affected.size(); i++ )
   {

      tp.assign( 0, 0, 0, 0 );

      Vertex *v_affected = m_affected[ i ];

      v_affected -> getParentMatrix( &v_inv );
      v_inv.Invert();

      /* For each bone that affects this vertex,
       * blend the deformed positions together
       */
      VertBoneMap::iterator begin = g_vert_bone_map.lower_bound( v_affected );
      VertBoneMap::iterator end = g_vert_bone_map.upper_bound( v_affected );
      VertBoneMap::iterator it = begin;
      float bone_count = 0;

      while ( it != end )
      {
         bone_count++;

         Bone *b = it->second;
         b -> getAffectedWorldSpace( v_affected, &bp );

         tp += bp;

         ++it;
      }

      tp /= bone_count;
      tp.w = 1;

      tp = v_inv * tp;  //transform from world to Object space.

      v_affected -> setPosition( tp.x, tp.y, tp.z );
   }

   //deform all this bones children.
   list<Bone *>::iterator it = m_children.begin();

   while ( it != m_children.end() )
   {
      ( *it ) ->deformAffected();
      ++it;
   }

}

void Bone::applyDeformations()
{

}

/** Calculates the complete world matrix
  * of this bones effector.
  */
void Bone::getEffectorMatrix( Matrix44 *rm )
{
   //calculate the complete matrix...
   rm->Unit();

   getOrientation().InsertInMatrix( rm->m );
   rm->SetPos( getPosition() );

   Matrix44 bm;
   bm.SetPos( m_effector );              //construct the relative end matrix

   *rm = *rm * bm;

   if ( getParent() != 0 )
   {
      Matrix44 p;

      if ( getParent() ->isA( Bone::TYPE ) )
      {
         ( ( Bone * ) getParent() ) ->getEffectorMatrix( &p );
         *rm = p * *rm;
      }
   }
}

/** Returns the complete
  * world matrix of the base
  * of this bone.
  */
void Bone::getCompleteMatrix( Matrix44 *cm )
{
   cm->Unit();
   getOrientation().InsertInMatrix( cm->m );
   cm->SetPos( getPosition() );

   if ( getParent() != 0 )
   {
      Matrix44 p;

      if ( getParent() ->isA( Bone::TYPE ) )
         ( ( Bone * ) getParent() ) ->getEffectorMatrix( &p );
      else
         getParent() ->getCompleteMatrix( &p );

      *cm = p * *cm;
   }
}
void Bone::drawLimits()
{
  gluQuadricDrawStyle( qobj, ( GLenum ) GLU_SILHOUETTE );
  gluQuadricNormals( qobj, ( GLenum ) GLU_NONE );

  glRotatef( 90, 0, 1, 0 );
  glColor4f( 1, 0, 0, 1 );
  gluPartialDisk( qobj, 0.0, m_effector.y, 20, 4, vXRot.x + 180, vXRot.x - vXRot.y );
  glRotatef( -90, 0, 1, 0 );

  glRotatef( 90, 1, 0, 0 );
  glColor4f( 0, 1, 0, 1 );
  gluPartialDisk( qobj, 0.0, m_effector.y, 20, 4, vYRot.y + 180, vYRot.x - vYRot.y );
  glRotatef( -90, 1, 0, 0 );

  glColor4f( 0, 0, 1, 1 );
  gluPartialDisk( qobj, 0.0, m_effector.y, 20, 4, vZRot.x + 180, vZRot.x - vZRot.y );
}
void Bone::drawAffected()
{
 			
 //draw crosses  the scame color as the
 //box at each affected verts position.
 Vector4 color;
 m_box.getColor( &color );
 glColor4f( color.x, color.y, color.z, .7 );

 Matrix44 e_matrix;
 getEffectorMatrix( &e_matrix );
 glBegin( GL_QUADS );

 Vector4 p;

 for ( int i = 0; i < ( int ) m_affected_offsets.size(); i++ )
 {

    p = * m_affected_offsets[ i ];
    p = e_matrix * p;



    glVertex3f( p.x - 0.05, p.y - 0.05, p.z );
    glVertex3f( p.x + 0.05, p.y - 0.05, p.z );

    glVertex3f( p.x + .05, p.y + 0.05, p.z );
    glVertex3f( p.x - .05, p.y + 0.05, p.z );

 }

 glEnd();


}

int Bone::draw( int d_options = 0 )
{
   Matrix44 m;

   glDisable( GL_CULL_FACE );

   glBegin( GL_TRIANGLE_FAN );
   glNormal3f( 0, 1, 0 );
   glVertex3f( m_effector.x, m_effector.y, m_effector.z );
   glNormal3f( -1, 0, 0 );
   glVertex3f( -.125, 0, 0 );
   glNormal3f( 0, 0, 1 );
   glVertex3f( 0, 0, .125 );
   glNormal3f( 1, 0, 0 );
   glVertex3f( .125, 0, 0 );
   glNormal3f( 0, 0, -1 );
   glVertex3f( 0, 0, -.125 );
   glNormal3f( -1, 0, 0 );
   glVertex3f( -.125, 0, 0 );
   glEnd();

   if ( Bone::getMode() == MODE_ENVELOPE || Bone::draw_envelope )
   {
      m_box.draw();
   }

   glPushMatrix();

   glPushAttrib( GL_COLOR_BUFFER_BIT | GL_LIGHTING_BIT );
   glDisable( GL_LIGHTING );

   if ( getParent() != 0 )
   {
      if ( getParent() -> isA( Bone::TYPE ) )
      {
         static_cast<Bone*>( getParent() ) -> getEffectorMatrix( &m );
      }

      else
      {
         getParent() -> getCompleteMatrix( &m );
      }

      glMultMatrixf( ( float * ) m.m );
   }

   else
   {
   }

   glTranslatef( pos.x, pos.y, pos.z );

   if ( Bone::getMode() == MODE_LIMITS || Bone::draw_limits )
   {
			drawLimits();
   }


   glPopMatrix();


   if ( Bone::getMode() == MODE_ENVELOPE || Bone::draw_affected )
   {
			drawAffected();
   }

   glPopAttrib();

   return 0;
}

void Bone::move( float x, float y, float z )
{

   switch ( Bone::getMode() )
   {
      case MODE_IK:
         moveIK( x, y, z );
         break;
      case MODE_ENVELOPE:
         moveEnvelope( x, y, z );
         break;
      case MODE_LIMITS:
         scaleLimits( x, y, z );
         break;
      default:
         Entity::move( x, y, z );
         deformAffected();
         break;
   }

}

void Bone::rotate( float amount, float x, float y, float z, float px, float py, float pz )
{
   Entity::rotate( amount, x, y, z, px, py, pz );
   deformAffected();
}

/** Moves the bone such that it's
  * effector is as close as possible to the
  * given point.
  */
void Bone::moveIK( float x, float y, float z )
{

   /* 03-11-01 - Save the bone offsets by baking it in....
   getRoot()->saveBoneAffectedOffsets();

   */
   Bone * currentLink;           //current link the skeleton that we are processing
   Vector4 desiredEnd, targetVector, curEnd, crossResult, rootPos, curVector, endPos;
   int tries = 0;

   float IK_POS_THRESH = .01;
   int MAX_IK_TRIES = 50;

   float turnAngle, cosAngle, turnDeg, dist;
   bool one = false;
   bool two = false;

   Matrix44 m;
   getCompleteMatrix( &m );
   endPos = m * m_effector;    //get the transformed end point.
   endPos.x += x;
   endPos.y += y;           //move the end pos
   endPos.z += z;

   currentLink = this;

   // Matrix44 bm;
   do
   {

      currentLink->getTransformedPosition( &rootPos ); //get the links position


      getCompleteMatrix( &m );
      curEnd = m * m_effector;                         //this chain's end point


      desiredEnd.x = endPos.x;                     //set up the desired end points.
      desiredEnd.y = endPos.y;
      desiredEnd.z = endPos.z;

      if ( curEnd.distanceSqr( desiredEnd ) > IK_POS_THRESH )
      {
         //create the vector to the current effector pos
         curVector.x = curEnd.x - rootPos.x;
         curVector.y = curEnd.y - rootPos.y;
         curVector.z = curEnd.z - rootPos.z;

         //create the desired effector position
         targetVector.x = endPos.x - rootPos.x;
         targetVector.y = endPos.y - rootPos.y;
         targetVector.z = endPos.z - rootPos.z;

         //move the vectors into the bone space...
         curVector = currentLink->getBoneSpace( curVector );
         targetVector = currentLink->getBoneSpace( targetVector );

         //normalize the vectors;
         curVector.normalize();
         targetVector.normalize();

         cosAngle = targetVector.dot3d( curVector );


         //if cosAngle = 1, already aligned (0 degrees)
         if ( cosAngle < 0.99999 )
         {
            //use the cross product to check how much to rotate
            Vector3 r = targetVector.ToVector3() * curVector.ToVector3();
            crossResult.x = r.x;
            crossResult.y = r.y;
            crossResult.z = r.z;
            crossResult.normalize();


            turnAngle = -acos( ( float ) cosAngle );
            turnDeg = turnAngle * 180 / M_PI;


            //damping.
            if ( turnDeg > currentLink->getStiffness() )
               turnDeg = currentLink->getStiffness();

            if ( turnDeg < -currentLink->getStiffness() )
               turnDeg = -currentLink->getStiffness();

            currentLink->Entity::rotate( turnDeg, crossResult.x, crossResult.y, crossResult.z, 0, 0, 0 );


            currentLink->checkDOF();


         }  //if cos < .9999

         if ( currentLink->getParent() == 0 ||
               ! currentLink->getParent() ->isA( Bone::TYPE ) )   //if it's the root bone, start at end
            currentLink = this;
         else
            currentLink = ( Bone * ) currentLink->getParent();
      } //if to far away

      dist = curEnd.distanceSqr( desiredEnd ) ;

      one = ( tries++ < MAX_IK_TRIES );

      two = ( dist > IK_POS_THRESH );

   }

   while ( one && two );


   getRoot() ->deformAffected();
}


void Bone::getBoundingMax( Vector4 *v )
{

/*
   Matrix44 m;
   getCompleteMatrix( &m );

   if ( Bone::getMode() == MODE_IK )         //return the end position
      *v = m * m_effector;
   else
      *v = m.GetPos();
*/
	//*v = pos;
	v -> assign( .125, 1, .125, 1 );
}

void Bone::getBoundingMin( Vector4 *v )
{
/*
   Matrix44 m;
   getCompleteMatrix( &m );

   if ( Bone::getMode() == MODE_IK )         //return the end position
      *v = m * m_effector;
   else
      *v = m.GetPos();
*/
	//*v = pos;
	
	v -> assign( -.125, 0, -.125, 1 );
}

void Bone::scale( float x, float y, float z, float ox = 0, float oy = 0, float oz = 0 )
{

   switch ( Bone::getMode() )
   {
      case MODE_ENVELOPE:
         scaleEnvelope( x, y, z );
         break;
      case MODE_LIMITS:
         scaleLimits( x, y, z );
         break;
      default:
         float av = x + y + z;
         av = av / 3;
         m_effector.y *= av;
         setEnvelope( m_envelope );
         break;
   }

}

void Bone::scaleEnvelope( float x, float y , float z )
{
   m_envelope.x *= x;
   m_envelope.y *= y;
   m_envelope.z *= z;


   setEnvelope( m_envelope );
}

void Bone::scaleLimits( float x, float y, float z )
{
   vXRot.x *= x;
   vXRot.y *= x;

   vYRot.x *= y;
   vYRot.y *= y;

   vZRot.x *= z;
   vZRot.y *= z;


}

void Bone::moveEnvelope( float x, float y, float z )
{
   m_envelope.x += x;
   m_envelope.y += y;
   m_envelope.z += z;

   setEnvelope( m_envelope );
}

void Bone::checkDOF()
{
   Vector4 euler;
   euler = quat.getEuler();

   if ( euler.x > vXRot.x )
      euler.x = vXRot.x;

   if ( euler.x < vXRot.y )
      euler.x = vXRot.y;

   if ( euler.y > vYRot.x )
      euler.y = vYRot.x;

   if ( euler.y < vYRot.y )
      euler.y = vYRot.y;

   if ( euler.z > vZRot.x )
      euler.z = vZRot.x;

   if ( euler.z < vZRot.y )
      euler.z = vZRot.y;

   quat.setEuler( euler );

}

/**Take a vector in world space and convert it into
  * the space relative to the bones base.
  */
Vector4 Bone::getBoneSpace( Vector4 & v )
{
   Matrix44 m;
   m.Unit();

   if ( parent != 0 )
      ( ( Bone * ) parent ) ->getEffectorMatrix( &m );

   //  getCompleteMatrix( &m );
   m.Invert();

   //transform vector into bone space.
   v = m * v;

   return v;

}

/**Take a vector in world space and convert it into
  * the space relative to the bones effector.
  */
Vector4 Bone::getEffectorSpace( Vector4 & v )
{
   Matrix44 m;
   m.Unit();
   getEffectorMatrix( &m );

   m.Invert();

   //transform vector into bone space.
   v = m * v;

   return v;

}

/** Forcibly adds a vertex to the bone.
 * This is usually used for file IO.
 */
void Bone::addAffected( Vertex * v )
{
   m_affected.push_back( v );
   g_vert_bone_map.insert( VertBonePair( v, this ) );

}

/** Searches through the Object database,
  * grabbing all vertices within the envelope
  * of this bone. For now, only meshes...
  */
void Bone::saveAffected( bool propagate )
{
   //clears the current list.
   clearVertBoneMapping();
   m_affected.clear();

   vector<Selectable *> * meshes = I3D::getDB() ->getSelectables( Mesh::TYPE );

   int num_verts;

   if ( meshes == 0 )
   {
      cerr << "no vertices found" << endl;
      return ;
   }

   Vector4 wp;
   Matrix44 matrix;
   Matrix44 b_matrix;

   getCompleteMatrix( &b_matrix );
   b_matrix.Invert();

   for ( int i = 0; i < ( int ) meshes->size(); i++ )
   {

      Mesh * m = ( Mesh * ) ( *meshes ) [ i ];
      num_verts = m -> numVerts();

      for ( int j = 0; j < num_verts; j++ )
      {

         Vertex *v = m->getVertex( j );

         v -> getParentObject() -> getCompleteMatrix( &matrix );
         wp = matrix * v->getPosition();

         wp = b_matrix * wp;

         if ( m_box.contains( wp ) )
         {
            m_affected.insert( m_affected.end(), v );
            g_vert_bone_map.insert( VertBonePair( v, this ) );
         }

      }

   }

   /*
   cerr<< "Bone "<< getID() <<" affecting "<<m_affected.size()<<" verts: "<<endl;
   for(int i=0; i< (int)m_affected.size(); i++ ) {
    cerr<< m_affected[i]->getParentObject()->getID() << ":" << m_affected[i]->getParentIndex()<<" ";
}

   cerr << endl;

   */

   // if necessary, do the same for all kids.
   if ( propagate )
   {
      list<Bone *>::iterator it = m_children.begin();

      while ( it != m_children.end() )
      {
         ( *it ) ->saveAffected( propagate );
         ++it;
      }
   }



}

/** Clears out this bones affected verts from the
  * global bone vert map.
  */
void Bone::clearVertBoneMapping()
{

   for ( int i = 0; i < ( int ) m_affected.size(); i++ )
   {
      Vertex *v = m_affected[ i ];

      VertBoneMap::iterator begin = g_vert_bone_map.lower_bound( v );
      VertBoneMap::iterator end = g_vert_bone_map.upper_bound( v );

      g_vert_bone_map.erase( begin, end );

   }

}

/** Returns the given affected position in worldspace worldspace.
  */
void Bone::getAffectedWorldSpace( Vertex *v, Vector4 *p )
{

   int i = 0;

   for ( i = 0; i < ( int ) m_affected.size(); i++ )
   {
      if ( m_affected[ i ] == v )
         break;
   }

   if ( i == ( int ) m_affected.size() )
   {
      cerr << "Bone:: couldn't find vertex." << endl;
      v -> dump();
      return ;
   }

   Matrix44 b_matrix;
   getEffectorMatrix( &b_matrix );

   *p = * m_affected_offsets[ i ];

   *p = b_matrix * *p;

   return ;
}

