/*
 * Copyright (C) 1998-9 Pancrazio `Ezio' de Mauro <p@demauro.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <dlfcn.h>
#include <syslog.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <dlfcn.h>

#include "localdecls.h"

#define LOGLEVEL (LOG_USER | LOG_INFO | LOG_PID)
#define BUFSIZE 1024

#define error(X) (X < 0 ? strerror(errno) : "success")

int __installwatch_refcount = 0;

#define REFCOUNT __installwatch_refcount++

static int(*true_chmod)(const char *, mode_t);
static int(*true_chown)(const char *, uid_t, gid_t);
static int(*true_chroot)(const char *);
static int(*true_creat)(const char *, mode_t);
static int(*true_fchmod)(int, mode_t);
static int(*true_fchown)(int, uid_t, gid_t);
static int(*true_ftruncate)(int, TRUNCATE_T);
static int(*true_lchown)(const char *, uid_t, gid_t);
static int(*true_link)(const char *, const char *);
static int(*true_mkdir)(const char *, mode_t);
static int(*true_open)(const char *, int, ...);
static int(*true_rename)(const char *, const char *);
static int(*true_rmdir)(const char *);
static int(*true_symlink)(const char *, const char *);
static int(*true_truncate)(const char *, TRUNCATE_T);
static int(*true_unlink)(const char *);

#if(GLIBC_MINOR >= 1)

static int(*true_creat64)(const char *, __mode_t);
static int(*true_ftruncate64)(int, __off64_t);
static int(*true_open64)(const char *, int, ...);
static int(*true_truncate64)(const char *, __off64_t);

#endif

static void log(const char *format, ...)
#ifdef __GNUC__
	/* Tell gcc that this function behaves like printf()
	 * for parameters 1 and 2                            */
	__attribute__((format(printf, 1, 2)))
#endif /* defined __GNUC__ */
;

void _init(void) {
	/* 
	 * There are two possibilities:
	 * a) I didn't understand well what RTLD_NEXT does, but it
	 *    happens to do the right thing under libc6
	 * b) RTLD_NEXT in libc5 (at least 5.4.38 and 5.3.12) is broken
	 */
	void *libc_handle;

#ifdef BROKEN_RTLD_NEXT
	libc_handle = dlopen(LIBC_VERSION, RTLD_LAZY);
#else
	libc_handle = RTLD_NEXT;
#endif
	true_chmod = dlsym(libc_handle, "chmod");
	true_chown = dlsym(libc_handle, "chown");
	true_chroot = dlsym(libc_handle, "chroot");
	true_creat = dlsym(libc_handle, "creat");
	true_fchmod = dlsym(libc_handle, "fchmod");
	true_fchown = dlsym(libc_handle, "fchown");
	true_ftruncate = dlsym(libc_handle, "ftruncate");
	true_lchown = dlsym(libc_handle, "lchown");
	true_link = dlsym(libc_handle, "link");
	true_mkdir = dlsym(libc_handle, "mkdir");
	true_open = dlsym(libc_handle, "open");
	true_rename = dlsym(libc_handle, "rename");
	true_rmdir = dlsym(libc_handle, "rmdir");
	true_symlink = dlsym(libc_handle, "symlink");
	true_truncate = dlsym(libc_handle, "truncate");
	true_unlink = dlsym(libc_handle, "unlink");

#if(GLIBC_MINOR >= 1)
	true_creat64 = dlsym(libc_handle, "creat64");
	true_ftruncate64 = dlsym(libc_handle, "ftruncate64");
	true_open64 = dlsym(libc_handle, "open64");
	true_truncate64 = dlsym(libc_handle, "truncate64");
#endif
}

static void log(const char *format, ...) {
	char buffer[BUFSIZE], *logname;
	va_list ap;
	int count, logfd;

	va_start(ap, format);
	count = vsnprintf(buffer, BUFSIZE, format, ap);
	va_end(ap);
	if(count == -1) {
		/* The buffer was not big enough */
		strcpy(&(buffer[BUFSIZE - 5]), "...\n");
		count = BUFSIZE - 1;
	}
	if((logname = getenv("INSTALLWATCHFILE"))) {
		logfd = true_open(logname, O_WRONLY | O_CREAT | O_APPEND, 0666);
		if(logfd >= 0) {
			if(write(logfd, buffer, count) != count)
				syslog(LOGLEVEL, "Count not write `%s' in `%s': %s\n", buffer, logname, strerror(errno));
			if(close(logfd) < 0)
				syslog(LOGLEVEL, "Could not close `%s': %s\n", logname, strerror(errno));
		} else
			syslog(LOGLEVEL, "Could not open `%s' to write `%s': %s\n", logname, buffer, strerror(errno));
		
	} else
		syslog(LOGLEVEL, buffer);
}

static void canonicalize(const char *path, char *resolved_path) {
	if(!realpath(path, resolved_path) && (path[0] != '/')) {
		/* The path could not be canonicalized, append it
		 * to the current working directory if it was not 
		 * an absolute path                               */
		getcwd(resolved_path, MAXPATHLEN - 2);
		strcat(resolved_path, "/");
		strncat(resolved_path, path, MAXPATHLEN - 1);
	}
} 

int chmod(const char *path, mode_t mode) {
	int result;
	char canonic[MAXPATHLEN];

	REFCOUNT;
	canonicalize(path, canonic);
	result = true_chmod(path, mode);
	log("%d\tchmod\t%s\t0%04o\t#%s\n", result, canonic, mode, error(result));
	return result;
}

int chown(const char *path, uid_t owner, gid_t group) {
	int result;
	char canonic[MAXPATHLEN];

	REFCOUNT;
	canonicalize(path, canonic);
	result = true_chown(path, owner, group);
	log("%d\tchown\t%s\t%d\t%d\t#%s\n", result, canonic, owner, group, error(result));
	return result;
}

int chroot(const char *path) {
	int result;
	char canonic[MAXPATHLEN];

	REFCOUNT;
	canonicalize(path, canonic);
	result = true_chroot(path);
	/* From now on, another log file will be written if *
	 * INSTALLWATCHFILE is set                          */
	log("%d\tchroot\t%s\t#%s\n", result, canonic, error(result));
	return result;
}

int creat(const char *pathname, mode_t mode) {
/* Is it a system call? */
	int result;
	char canonic[MAXPATHLEN];

	REFCOUNT;
	canonicalize(pathname, canonic);
	result = true_open(pathname, O_CREAT | O_WRONLY | O_TRUNC, mode);
	log("%d\tcreat\t%s\t#%s\n", result, canonic, error(result));
	return result;
}

int fchmod(int filedes, mode_t mode) {
	int result;

	REFCOUNT;
	result = true_fchmod(filedes, mode);
	log("%d\tfchmod\t%d\t0%04o\t#%s\n", result, filedes, mode, error(result));
	return result;
}

int fchown(int fd, uid_t owner, gid_t group) {
	int result;

	REFCOUNT;
	result = true_fchown(fd, owner, group);
	log("%d\tfchown\t%d\t%d\t%d\t#%s\n", result, fd, owner, group, error(result));
	return result;
}

int ftruncate(int fd, TRUNCATE_T length) {
	int result;

	REFCOUNT;
	result = true_ftruncate(fd, length);
	log("%d\tftruncate\t%d\t%d\t#%s\n", result, fd, (int)length, error(result));
	return result;
}

int lchown(const char *path, uid_t owner, gid_t group) {
/* Linux specific? */
	int result;
	char canonic[MAXPATHLEN];

	REFCOUNT;
	canonicalize(path, canonic);
	result = true_chown(path, owner, group);
	log("%d\tlchown\t%s\t%d\t%d\t#%s\n", result, canonic, owner, group, error(result));
	return result;
}

int link(const char *oldpath, const char *newpath) {
	int result;
	char old_canonic[MAXPATHLEN], new_canonic[MAXPATHLEN];

	REFCOUNT;
	canonicalize(oldpath, old_canonic);
	canonicalize(newpath, new_canonic);
	result = true_link(oldpath, newpath);
	log("%d\tlink\t%s\t%s\t#%s\n", result, old_canonic, new_canonic, error(result));
	return result;
}

int mkdir(const char *pathname, mode_t mode) {
	int result;
	char canonic[MAXPATHLEN];

	REFCOUNT;
	canonicalize(pathname, canonic);
	result = true_mkdir(pathname, mode);
	log("%d\tmkdir\t%s\t#%s\n", result, canonic, error(result));
	return result;
}

int open(const char *pathname, int flags, ...) {
/* Eventually, there is a third parameter: it's mode_t mode */
	va_list ap;
	mode_t mode;
	int result;
	char canonic[MAXPATHLEN];

	REFCOUNT;
	va_start(ap, flags);
	mode = va_arg(ap, mode_t);
	va_end(ap);
	canonicalize(pathname, canonic);
	result = true_open(pathname, flags, mode);
	if(flags & (O_WRONLY | O_RDWR)) 
		log("%d\topen\t%s\t#%s\n", result, canonic, error(result));
	/* TODO: add an option to backup files opened for writing */
	return result;
}

int rename(const char *oldpath, const char *newpath) {
	int result;
	char old_canonic[MAXPATHLEN], new_canonic[MAXPATHLEN];

	REFCOUNT;
	canonicalize(oldpath, old_canonic);
	canonicalize(newpath, new_canonic);
	result = true_rename(oldpath, newpath);
	log("%d\trename\t%s\t%s\t#%s\n", result, old_canonic, new_canonic, error(result));
	return result;
}

int rmdir(const char *pathname) {
	int result;
	char canonic[MAXPATHLEN];

	REFCOUNT;
	canonicalize(pathname, canonic);
	result = true_rmdir(pathname);
	log("%d\trmdir\t%s\t#%s\n", result, canonic, error(result));
	return result;
}

int symlink(const char *oldpath, const char *newpath) {
	int result;
	char old_canonic[MAXPATHLEN], new_canonic[MAXPATHLEN];

	REFCOUNT;
	canonicalize(oldpath, old_canonic);
	canonicalize(newpath, new_canonic);
	result = true_symlink(oldpath, newpath);
	log("%d\tsymlink\t%s\t%s\t#%s\n", result, old_canonic, new_canonic, error(result));
	return result;
}

int truncate(const char *path, TRUNCATE_T length) {
	int result;
	char canonic[MAXPATHLEN];

	REFCOUNT;
	canonicalize(path, canonic);
	result = true_truncate(path, length);
	log("%d\ttruncate\t%s\t%d\t#%s\n", result, path, (int)length, error(result));
	return result;
}

int unlink(const char *pathname) {
	int result;
	char canonic[MAXPATHLEN];

	REFCOUNT;
	canonicalize(pathname, canonic);
	result = true_unlink(pathname);
	log("%d\tunlink\t%s\t#%s\n", result, canonic, error(result));
	return result;
}

#if(GLIBC_MINOR >= 1)

int creat64(const char *pathname, __mode_t mode) {
/* Is it a system call? */
	int result;
	char canonic[MAXPATHLEN];

	REFCOUNT;
	canonicalize(pathname, canonic);
	result = true_open64(pathname, O_CREAT | O_WRONLY | O_TRUNC, mode);
	log("%d\tcreat\t%s\t#%s\n", result, canonic, error(result));
	return result;
}

int ftruncate64(int fd, __off64_t length) {
	int result;

	REFCOUNT;
	result = true_ftruncate64(fd, length);
	log("%d\tftruncate\t%d\t%d\t#%s\n", result, fd, (int)length, error(result));
	return result;
}

int open64(const char *pathname, int flags, ...) {
/* Eventually, there is a third parameter: it's mode_t mode */
	va_list ap;
	mode_t mode;
	int result;
	char canonic[MAXPATHLEN];

	REFCOUNT;
	va_start(ap, flags);
	mode = va_arg(ap, mode_t);
	va_end(ap);
	canonicalize(pathname, canonic);
	result = true_open64(pathname, flags, mode);
	if(flags & (O_WRONLY | O_RDWR)) 
		log("%d\topen\t%s\t#%s\n", result, canonic, error(result));
	/* TODO: add an option to backup files opened for writing */
	return result;
}

int truncate64(const char *path, __off64_t length) {
	int result;
	char canonic[MAXPATHLEN];

	REFCOUNT;
	canonicalize(path, canonic);
	result = true_truncate64(path, length);
	log("%d\ttruncate\t%s\t%d\t#%s\n", result, path, (int)length, error(result));
	return result;
}

#endif /* GLIBC_MINOR >= 1 */

