/*  Inti-GL: Integrated Foundation Classes
 *  Copyright (C) 2003 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

//! @file inti/gdk/gl/context.h
//! @brief A GdkGLContext C++ wrapper interface.
//!
//! Provides Context, an OpenGL rendering context object.

#ifndef INTI_GDK_GL_CONTEXT_H
#define INTI_GDK_GL_CONTEXT_H

#ifndef INTI_G_OBJECT_H
#include <inti/glib/object.h>
#endif

#ifndef __GDK_GL_CONTEXT_H__
#include <gdk/gdkglcontext.h>
#endif

#ifndef __GDK_GL_TOKENS_H__
#include <gdk/gdkgltokens.h>
#endif

#ifndef __gl_h_
#include <GL/gl.h>
#endif

namespace Inti {

namespace Gdk {

namespace GL {

class Config;
class ContextClass;
class Drawable;

//! @class Context context.h inti/gdk/gl/context.h
//! @brief A GdkGLContext C++ wrapper class.

class Context : public G::Object
{
	friend class G::Object;
	friend class ContextClass;

	Context(const Context&);
	Context& operator=(const Context&);

protected:
//! @name Constructors
//! @{

	explicit Context(GdkGLContext *context, bool reference = true);
	//!< Construct a new Context from an existing GdkGLContext.
	//!< @param context A pointer to a GdkGLContext.
	//!< @param reference Set false if the initial reference count is floating, set true if it's not.
	//!<
	//!< <BR>The <EM>context</EM> can be a newly created GdkGLContext or an existing
	//!< GdkGLContext. (see G::Object::Object).

//! @}

public:
//! @name Constructors
//! @{

	virtual ~Context();
	//!< Destructor.

//! @}

	static Pointer<Context> create(const Drawable& gldrawable, bool direct = true, int render_type = GDK_GL_RGBA_TYPE);
	//!< Creates a new OpenGL rendering context.
	//!< @param gldrawable A Gdk::GL::Drawable.
	//!< @param direct Whether rendering is to be done with a direct connection to the graphics system.
	//!< @param render_type Gdk::GL::RGBA_TYPE or Gdk::GL::COLOR_INDEX_TYPE (currently not used).
	//!< @return The new Context, or null if an error occurs.
	//!<
	//!< <BR>No context sharing takes place with contexts created by this method.

	static Pointer<Context> create(const Drawable& gldrawable, const Context& share_list, bool direct = true, int render_type = GDK_GL_RGBA_TYPE);
	//!< Creates a new OpenGL rendering context.
	//!< @param gldrawable A Gdk::GL::Drawable.
	//!< @param share_list The Context with which to share display lists and texture objects.
	//!< @param direct Whether rendering is to be done with a direct connection to the graphics system.
	//!< @param render_type Gdk::GL::RGBA_TYPE or Gdk::GL::COLOR_INDEX_TYPE (currently not used).
	//!< @return The new Context, or null if an error occurs.

//! @name Accessors
//! @{

	GdkGLContext* gdk_gl_context() const { return (GdkGLContext*)instance; }
	//!< Get a pointer to the GdkGLContext structure.

	GdkGLContextClass* gdk_gl_context_class() const;
	//!< Get a pointer to the GdkGLContextClass structure.

	operator GdkGLContext* () const;
	//!< Conversion operator; safely converts a Context to a GdkGLContext pointer.

	Drawable* get_gl_drawable() const;
	//!< Gets Gdk::GL::Drawable to which the context is bound.
	//!< @return The drawable, or null if no Gdk::GL::Drawable is bound.

	Config* get_gl_config() const;
	//!< Gets Gdk::GL::Config with which the context is configured.
	//!< @return A pointer to the Gdk::GL::Config object.

	Context* get_share_list() const;
	//!< Gets the Gdk::GL::Context with which the context shares the display lists and texture objects.
	//!< @return The Gdk::GL::Context, or null if the context is not sharing.

	bool is_direct() const;
	//!< Determines whether the context is a direct rendering context.
	//!< @return <EM>true</EM> if the context is a direct rendering context.

	int get_render_type() const;
	//!< Gets render_type of the context.
	//!< @return GDK_GL_RGBA_TYPE or GDK_GL_COLOR_INDEX_TYPE.

//! @}

	static Context* get_current();
	//!< Gets the current Gdk::GL::Context.
	//!< @return The current Gdk::GL::Context, or null if there is no current context.

//! @name Methods
//! @{

	virtual void dispose();
	//!< Disposes of the OpenGL resources associated with the context and decrements
	//!< context's reference count.

	bool copy(const Context& src, unsigned long mask = GL_ALL_ATTRIB_BITS);
	//!< Copy one or more state bits from <EM>src</EM> to this context.
	//!< @param src The source context.
	//!< @param mask The bitwise-OR of the same symbolic names that are passed to glPushAttrib().
	//!< @return <EM>true</EM> if successful, <EM>false</EM> otherwise.
	//!<
	//!< <BR>You can use GL_ALL_ATTRIB_BITS to copy all the rendering state information
	//!< (see the "GL/gl.h" header file).

//! @}
};

} // namespace GL

} // namespace Gdk

} // namespace Inti

#endif // INTI_GDK_GL_CONTEXT_H
