/*
 *
 * $Id: fetchipac.c,v 1.7 2000/06/05 22:53:25 moritz Exp $
 *
 * Fetch IP accounting stats
 * Copyright (C) 1997 - 2000 Moritz Both
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * The author can be reached via email: moritz@daneben.de, or by
 * snail mail: Moritz Both, Im Moore 26, 30167 Hannover,
 *             Germany. Phone: +49-511-1610129
 *
 */

#include <string.h>
#include <stdio.h>
#include <sys/stat.h>
#include <dirent.h>
#include <errno.h>
#include <search.h>
#include <time.h>
#include <stdlib.h>
#include <assert.h>
#include "ipac.h"

int	acc_style;		/* 0=ipfwadm, 1=ipchains */
char 	*me;			/* program name */

/* Call ipacset --fix-chains */
void fix_chains()
{
	char buf[PATH_MAX + 20];
	fprintf(stderr, "%s: Warning: ipac chains and/or jumps are corrupted. "
			"Trying to fix them\n", me);
	snprintf(buf, PATH_MAX+19, "%s --fix-chains", IPACSET);
	system(buf);
}

/* check if everything we need is in place and complain and exit if not *
 * run fix_chains() if neccessary
 */
void check_environment()
{
	struct stat stat_buf;
	FILE *f;
	int okay, count;
	char line_buf[256], *cp, *cp1;

	if (stat(IPFWADM_PROC, &stat_buf) == 0) 
	{
		acc_style = 0;
	}
	else if (stat(IPCHAINS_PROC_C, &stat_buf) == 0)
	{
		acc_style = 1;
		/* Check if all chains and jumps are there or if some
		 * firewall script deleted them. (This check is only an 
		 * estimate.)
		 */
		f = fopen(IPCHAINS_PROC_C, "r");
		if (f == NULL)
		{
			fprintf(stderr, "%s: cant open \"%s\": %s\n"
				"(must run as root... are we root?)\n",
				me, IPCHAINS_PROC_C, sys_errlist[errno]);
			exit(1);
		}
		okay = 0;
		count = 0;
		if (f != NULL) 
		{
			while(fgets(line_buf, 256, f) != NULL)
			{
				cp = strstr(line_buf, "input");
				if (cp == NULL)
					cp = strstr(line_buf, "output");
				if (cp != NULL)
				{
					cp1 = strstr(cp, CH_INNAME);
					if (cp1==NULL)
						cp1=strstr(cp, CH_OUTNAME);
					if (cp1==NULL)
						cp1=strstr(cp, CH_IONAME);
					if (cp1 != NULL)
						count++;
				}
			}
			fclose(f);
		}
		if (count == 4)
			okay = 1;
		if (okay == 0)
			fix_chains();
	}
	else
	{
		fprintf(stderr, "%s: no ip firewall / accounting code in the "
				"kernel", me);
		exit(1);
	}
}

/* structure for run file (rule file)
 */
struct runfile_line_type {
	char *line;
	struct runfile_line_type *next;
};

/* delete run file (rule file) from memory, freeing dynamically
 * allocated memory
 */
void destroy_runfile_lines(struct runfile_line_type *lines)
{
	struct runfile_line_type *next;
	while(lines != NULL)
	{
		if (lines->line != NULL)
			free(lines->line);
		next = lines->next;
		free(lines);
		lines = next;
	}
}

/* read run file (rule file) and store it in memory, using a singly 
 * linked list of instances of struct runfile_line_type
 * return the list read or NULL in case of error
 */
struct runfile_line_type *read_runfile()
{
	FILE *frunfile;
	char runfile_line[MAX_RULE_NAME_LENGTH + 50], *cp; 
	struct runfile_line_type *result, *lastline, *cur;

	frunfile = fopen(RUNFILE, "r");
	if (frunfile == NULL)
	{
		fprintf(stderr, "%s: cant open run file \"%s\": %s "
				"(ipacset not run?)\n",
			me, RUNFILE, sys_errlist[errno]);
		return NULL;
	}

	result = NULL;
	lastline = NULL;
	while(fgets(runfile_line, MAX_RULE_NAME_LENGTH+50, frunfile)
			!= NULL)
	{
		cp = strchr(runfile_line, '\n');
		if (cp)
			*cp = '\0';
		if (*runfile_line == '#')
			continue;

		cur = (struct runfile_line_type *)
				xmalloc(sizeof(struct runfile_line_type));
		cur->line = xstrdup(runfile_line);
		cur->next = NULL;
		if (result == NULL)
			result = cur;
		else 
			lastline->next = cur;
		lastline = cur;
	}
	if (!feof(frunfile))
	{
		fprintf(stderr, "%s: reading \"%s\": %s\n",
			me, RUNFILE, sys_errlist[errno]);
		fclose(frunfile);
		destroy_runfile_lines(result);
		result = NULL;
	}
	fclose(frunfile);
	return result;
}

void free_tree(void **ruletreep)
{
	/* The tree should be deleted here...
	 * Since fetchipac exists anyway, we don't NEED this.
	 */
}

/* read kernel accounting data in ipfwadm system
 * read from stream f
 * create records with data (packet and data counters) and
 * rule names and store them into instances of rule_type (declared
 * in ipac.h) using rule names from runfile
 * if a rule name is equal to a previously used rule name, add the
 * counters and re-use the old record
 * complain if something is wrong with the data.
 * return 0 for success, 1 otherwise
 */
int read_ipfwadm(FILE *f, struct runfile_line_type *runfile,
			rule_type **firstrule)
{
	char procfile_line[MAX_PROCFILE_LINE_LENGTH], *cp;
	rule_type *rule, *lastrule;
	struct runfile_line_type *nextline;
	void *node;
	int i;
	unsigned long bytes=0, pkts=0;
	void *ruletree = NULL;
	
	/* 1st line: 'IP Accounting rules' */
	if (fgets(procfile_line, MAX_PROCFILE_LINE_LENGTH, f) == NULL)
	{
		fprintf(stderr, "%s: error reading proc file: %s",
			me, sys_errlist[errno]);
		return 1;
	}

	lastrule = *firstrule = NULL;
	while(fgets(procfile_line, MAX_PROCFILE_LINE_LENGTH, f) != NULL)
	{
		if (runfile == NULL)
		{
			fprintf(stderr, "%s: more kernel accounting data than "
				"rules in \"%s\" - extra ignored\n"
				"(run ipacset to fix this)\n",
				me, RUNFILE);
			break;
		}
		rule = new_rule();
		strncpy(rule->name, runfile->line, MAX_RULE_NAME_LENGTH);
		/* use a binary tree to find rules with same name */
		node = tsearch(rule, &ruletree, rule_compare);
		if (*(rule_type **)node != rule)
		{
			free(rule);
			rule=*(rule_type **)node;
		}
		else
		{
			if (lastrule != NULL)
				lastrule->next = rule;
			lastrule = rule;
			if (*firstrule == NULL)
				*firstrule = rule;
		}
		/* word 6 and 7 are pkt and byte counters */
		cp = procfile_line;
		for (i=0; i<7; i++)
		{
			cp = strchr(cp, ' ');
			if (cp == NULL)
				break;
			while(*cp == ' ')
				cp++;
			if (i==5)
				pkts = atoi(cp);
			else if (i==6)
				bytes = atoi(cp);
		}
		if (cp == NULL)
		{
			fprintf(stderr, "%s: parse error reading proc accounting file\n",
					me);
			return 1;
		}
		rule->pkts += pkts;
		rule->bytes += bytes;
		nextline = runfile->next;
		free(runfile->line);
		free(runfile);
		runfile = nextline;
	}
	if (runfile)
	{
		fprintf(stderr, "%s: more rules in \"%s\" than in kernel\n"
			"(run ipacset to fix this)\n",
			me, RUNFILE);
	}
	free_tree(&ruletree);
	return 0;
}

/* read kernel accounting data in ipchains system
 * read from stream f
 * create records with data (packet and data counters) and
 * rule names and store them into instances of rule_type (declared
 * in ipac.h) using rule names from runfile
 * if a rule name is equal to a previously used rule name, add the
 * counters and re-use the old record
 * complain if something is wrong with the data.
 * return 0 for success, 1 otherwise
 */
int read_ipchains(FILE *f, struct runfile_line_type *runfile,
			rule_type **firstrule)
{
	char procfile_line[MAX_PROCFILE_LINE_LENGTH], *cp;
	rule_type *rule, *lastrule, search_rule;
	struct runfile_line_type *nextline, *line_before;
	void *node;
	char chain_name[10];
	unsigned long bytes_hi, bytes_lo, pkts_hi, pkts_lo;
	UINT64 bytes, pkts;
	void *ruletree = NULL;
	
	/* create the rule_type records in correct order as from 
	 * run file.
	 */
	lastrule = *firstrule = NULL;
	for (nextline=runfile; nextline!=NULL; nextline=nextline->next)
	{
		cp = strchr(nextline->line, ' ');
		if (cp == 0)
			continue;	/* bad entry */
		rule = new_rule();
		strncpy(rule->name, cp+1, MAX_RULE_NAME_LENGTH);
		/* use a binary tree to find rules with same name */
		node = tsearch(rule, &ruletree, rule_compare);
		if (*(rule_type **)node != rule)
		{
			free(rule);
		}
		else
		{
			if (lastrule != NULL)
				lastrule->next = rule;
			lastrule = rule;
			if (*firstrule == NULL)
				*firstrule = rule;
		}

	}

	while(fgets(procfile_line, MAX_PROCFILE_LINE_LENGTH, f) != NULL)
	{
		if (runfile == NULL)
		{
			fprintf(stderr, "%s: more kernel accounting data than "
				"rules in \"%s\" - extra ignored\n"
				"(run ipacset to fix this)\n",
				me, RUNFILE);
			break;
		}
		if (sscanf(procfile_line, " %9s %*s %*s %*s %*s %*s "
				"%lu %lu %lu %lu",
				chain_name, &pkts_hi, &pkts_lo,
					&bytes_hi, &bytes_lo) != 5)
		{
			fprintf(stderr, "%s: parse error reading proc accounting file\n",
					me);
			return 1;
		}
		/* is this ours ? */
		if (strcmp(chain_name, CH_INNAME) == 0
			|| strcmp(chain_name, CH_OUTNAME) == 0
			|| strcmp(chain_name, CH_IONAME) == 0)
		{	

			bytes = bytes_lo + ((UINT64)bytes_hi << 32);
			pkts = pkts_lo + ((UINT64)pkts_hi << 32);
		
			/* find appropriate runfile line */
			line_before = NULL;
			for (nextline = runfile; nextline != NULL; 
					line_before=nextline, 
					nextline=nextline->next)
			{
				cp = strchr(nextline->line, ' ');
				if (cp == 0)
					continue;	/* bad entry */
				if (strncmp(nextline->line, chain_name, 
						cp-nextline->line) == 0)
					break;
			}
			if (nextline == NULL)
			{
				fprintf(stderr, "%s: out of rule names for "
					"chain %s in \"%s\"\n"
					" (something corrupted ipac settings, r"
					"un ipacset to restore)\n",
					me, chain_name, RUNFILE);
				return 1;
			}

			/* use the binary tree to find the rule */
			strncpy(search_rule.name, cp+1, MAX_RULE_NAME_LENGTH);
			node = tfind(&search_rule, 
				(const void **)&ruletree, rule_compare);
			assert(node != NULL);
			rule=*(rule_type **)node;
			rule->pkts += pkts;
			rule->bytes += bytes;
			if (line_before != NULL)
				line_before->next = nextline->next;
			else
				runfile = nextline->next;
			free(nextline->line);
			free(nextline);
		}
	}
	if (runfile)
	{
		fprintf(stderr, "%s: more rules in \"%s\" than in kernel\n"
			"(run ipacset to fix this)\n",
			me, RUNFILE);
	}
	free_tree(&ruletree);
	return 0;
}

/* read the files - RUNFILE (rule file) and the correct /proc/net/... file -
 * and store the data into a list.
 */
int collect_data(rule_type **firstrule)
{
	FILE *fprocfile;
	char *procfile;
	struct runfile_line_type *runfile;
	int retval;

	runfile = read_runfile();
	if (runfile == NULL)
		return 1;

	procfile = (acc_style == 0 ? IPFWADM_PROC : IPCHAINS_PROC_C);

	/* open in read/write mode to reset counters */
	fprocfile = fopen(procfile, "r+");
	if (fprocfile == NULL)
	{
		fprintf(stderr, "%s: cant open \"%s\": %s\n",
				me, procfile, sys_errlist[errno]);
		return 1;
	}

	retval = (acc_style == 0 ?
		read_ipfwadm(fprocfile, runfile, firstrule)
		: read_ipchains(fprocfile, runfile, firstrule));

	fclose(fprocfile);
	
	return retval;
}

int write_oldstyle_file(rule_type *firstrule)
{
	FILE *f;
	char filename[PATH_MAX+20];
	time_t t;
	struct tm *tmp;
	int i;
	rule_type *rule;

	t = time(NULL);
	tmp = localtime(&t);

	strncpy(filename, ACCTDIR, PATH_MAX-1);
	strcat(filename, "/");
	i = strlen(filename);
	strftime(filename + i, PATH_MAX-i, "%Y%m%d-%H%M%S", tmp);

	f = fopen(filename, "w");
	if (f == NULL)
	{
		fprintf(stderr, "%s: opening \"%s\": %s\n",
			me, filename, sys_errlist[errno]);
		return 1;
	}

	for (rule = firstrule; rule; rule=rule->next)
		fprintf(f, "%s\n", rule->name);
	fprintf(f, "%s\n", DATDELIM);
	for (rule = firstrule; rule; rule=rule->next)
		fprintf(f, "%Lu %Lu\n", rule->pkts,
			rule->bytes);
	if (ferror(f))
	{
		fprintf(stderr, "%s: error writing file \"%s\": %s\n",
			me, filename, sys_errlist[errno]);
		fclose(f);
		return 1;
	}
	fclose(f);
	return 0;
}

int main(int argc, char **argv)
{
	rule_type *firstrule = NULL;

	me = argv[0];

	if (argc > 1)
	{
		printf("fetchipac Version %s Usage: fetchipac\n"
			"See fetchipac(8) for further information.\n",
				VERSION);
		exit(1);
	}
	check_environment();
	if (lock(LOCKFILE) == 0)
	{
		if (collect_data(&firstrule) == 0)
		{
			write_oldstyle_file(firstrule);
		}

		unlock(LOCKFILE);
	}
	return 0;
}
