/**************************************************************************** 
** File: datalink.c
**
** Author: Mike Borella
**
** Comments: Generic datalink module
**
** $Id: datalink.c,v 1.20 2001/09/07 22:16:29 mborella Exp $
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU Library General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
*****************************************************************************/

#include "global.h"
#include "error.h"
#include "datalink.h"
#include "ethernet.h"
#include "loopback.h"
#include "slip.h"
#include "raw.h"
#include "ppp.h"
#include "padding.h"
#include "payload.h"
#include "stats.h"

extern struct arg_t * my_args;
extern int            packet_displayed;
extern int            start_of_packet;
extern FILE *         outfile;

/*----------------------------------------------------------------------------
**
** datalink()
**
**----------------------------------------------------------------------------
*/

void datalink(u_char *user, const struct pcap_pkthdr *h, u_char *pkt)
{
  packet_t    packet;
  u_int16_t   len;
  u_int32_t * link;
  static int  count = 1; /* count of the number of packets */

  /*
   * For minimal mode.  We haven't displayed part of a packet yet...
   * Set the start of packet flag so that we can display a nicer separator
   */

  packet_displayed = 0;
  start_of_packet = 1;

  /*
   * Reset the stats counted flag and unpause the counting if paused.
   * This avoids the double-counting of encapsulated headers.
   */

  stats_reset();
  stats_unpause();

  /*
   * Determine real length of the data that we received
   */

  len = h->caplen;

  /*
   * Set up packet into data structure
   */

  packet.contents = my_malloc(len+1);
  memcpy(packet.contents, pkt, len);
  packet.current = &packet.contents[0];
  packet.end = &packet.contents[0] + len;
  packet.apparent_end = packet.end;
  packet.media_length = h->len;
  sprintf(packet.timestamp, "%u.%06u", (u_int32_t) h->ts.tv_sec, 
	  (u_int32_t) h->ts.tv_usec);

  /*
   * If we're in minimal mode, start off with the count number
   * and the optional packet timestamp
   */
  
  if (my_args->m)
    {
      display_minimal((u_int8_t *) &count, 4, DISP_DEC);
      display_minimal_string(" ");
      count ++;

      if (!my_args->T)
	{
	  display_minimal_string(packet.timestamp);
	  display_minimal_string(" ");
	}
    }
  
  /* 
   * Delete any remaining hex buffer contents
   */
  
  hexbuffer_kill();
  
  /*
   * Determine what to do next
   */
  
  link = (u_int32_t *) user;
  switch(*link)
    {
    case DLT_NULL:
      dump_loopback(&packet);
      break;
      
    case DLT_EN10MB:
      dump_ethernet(&packet);
      break;
      
    case DLT_SLIP:
      dump_slip(&packet);
      break;
      
    case DLT_PPP:
      dump_ppp(&packet);
      break;
      
#ifdef DLT_RAW /* Not supported in some arch or older pcap versions */
    case DLT_RAW:
      dump_raw(&packet);
      break;
#endif
    default:
      error_fatal("\ncannot handle data link type %d", link);
    }
  
  /* 
   * Both payload and padding shouldn't be displayed in minimal mode 
   */
  if (!my_args->m)
    {
      /*
       * If there is any remaining data and the user wants to see it, dump it 
       * out now
       */
      
      if (my_args->p && get_packet_apparentbytesleft(&packet))
	dump_payload(&packet);
      
      /* 
       * If there is a padding and the user wants to see it, dump it out now
       */
      
      if (my_args->d && packet_haspadding(&packet))
	dump_padding(&packet);
    }

  /*
   * For minimal mode, finish with a carriage return
   */
  
  if (my_args->m && packet_displayed)
    {
      display_minimal_cr();
    }

  /*
   * Deallocate memory
   */
  
  my_free(packet.contents);
      
}
