/*
 * Configurable ps-like program.
 * Column data and display routines.
 *
 * Copyright (c) 2008 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#include <signal.h>
#include <time.h>
#include "ips.h"


/*
 * Macro to define the column functions and to fill in the COLUMN
 * structure with values.
 */
#define	DEFCOL(symbol, name, header, width, justify, flags)		\
	static const char *	Show##symbol(const PROC * proc);	\
	static int	Sort##symbol(const PROC * proc1, const PROC * proc2); \
	static void	Eval##symbol(const PROC * proc, VALUE * val);	\
	static BOOL	Test##symbol(const PROC * proc);		\
									\
	static	COLUMN	Column##symbol =				\
	{								\
		#name, #header, sizeof(#header) - 1, width, width,	\
		justify, flags,						\
		Show##symbol,	Sort##symbol,				\
		Eval##symbol,	Test##symbol				\
	};


/*
 * The column definitions.
 */
DEFCOL(Pid,		pid,		Pid,
	5,	RIGHT,	USE_NONE)

DEFCOL(State,		state,		Stat,
	4,	CENTER,	USE_NONE)

DEFCOL(ThreadCount,	threads,	Threads,
	7,	RIGHT,	USE_NONE)

DEFCOL(ParentPid,	parentpid,	PPid,
	5,	RIGHT,	USE_NONE)

DEFCOL(SystemTime,	systemtime,	System Time,
	10,	RIGHT,	USE_NONE)

DEFCOL(UserTime,	usertime,	User Time,
	10,	RIGHT,	USE_NONE)

DEFCOL(RunTime,		runtime,	Runtime,
	10,	RIGHT,	USE_NONE)

DEFCOL(ChildRunTime,	childruntime,	ChildRuntime,
	10,	RIGHT,	USE_NONE)

DEFCOL(PageSwaps,	pageswaps,	PageSwaps,
	10,	RIGHT,	USE_NONE)

DEFCOL(Policy,		policy,		Policy,
	6,	RIGHT,	USE_NONE)

DEFCOL(Priority,	priority,	Pri,
	3,	RIGHT,	USE_NONE)

DEFCOL(RealTimePriority, realtimepriority, RTPri,
	5,	RIGHT,	USE_NONE)

DEFCOL(ChildPageSwaps,	childpageswaps,	ChildPgSwp,
	10,	RIGHT,	USE_NONE)

DEFCOL(Eip,		eip,		EIP,
	LONG_HEX_DIGITS, RIGHT,	USE_NONE)

DEFCOL(Esp,		esp,		ESP,
	LONG_HEX_DIGITS, RIGHT,	USE_NONE)

DEFCOL(WaitChannel,	waitchannel,	WaitChan,
	LONG_HEX_DIGITS, RIGHT,	USE_NONE)

DEFCOL(WaitSymbol,	waitsymbol,	WaitSymbol,
	16,	LEFT,	USE_WCHAN)

DEFCOL(Program,		program,	Program,
	12,	LEFT,	USE_NONE)

DEFCOL(Command,		command,	Command,
	20,	LEFT,	USE_COMMAND)

DEFCOL(Environment,	environment,	Environment,
	25,	LEFT,	USE_ENVIRON)

DEFCOL(ProcessGroup,	processgroup,	PGrp,
	5,	RIGHT,	USE_NONE)

DEFCOL(TtyProcessGroup,	ttyprocessgroup,TtyPG,
	5,	RIGHT,	USE_NONE)

DEFCOL(IdleTime,	idletime,	Idle,
	8,	RIGHT,	USE_INIT)

DEFCOL(DeadTime,	deadtime,	Dead,
	8,	RIGHT,	USE_NONE)

DEFCOL(TtyName,		ttyname,	TtyName,
	8,	LEFT,	USE_DEV_NAME)

DEFCOL(TtyDevice,	ttydevice,	TtyD,
	4,	RIGHT,	USE_NONE)

DEFCOL(UserName,	user,		User,
	8,	LEFT,	USE_USER_NAME)

DEFCOL(UserId,		uid,		Uid,
	5,	RIGHT,	USE_NONE)

DEFCOL(GroupName,	group,		Group,
	8,	LEFT,	USE_GROUP_NAME)

DEFCOL(GroupId,		gid,		Gid,
	5,	RIGHT,	USE_NONE)

DEFCOL(PercentCpu,	percentcpu,	%Cpu,
	5,	RIGHT,	USE_INIT)

DEFCOL(PercentMemory,	percentmemory,	%Mem,
	4,	RIGHT,	USE_NONE)

DEFCOL(Rss,		residentsetsize,Rss,
	5,	RIGHT,	USE_NONE)

DEFCOL(StartTime,	starttime,	Start Time,
	10,	LEFT,	USE_NONE)

DEFCOL(Age,		age,		Age,
	8,	RIGHT,	USE_NONE)

DEFCOL(Flags,		flags,		Flags,
	8,	RIGHT,	USE_NONE)

DEFCOL(PageFaults,	pagefaults,	Faults,
	10,	RIGHT,	USE_NONE)

DEFCOL(MinorPageFaults,	minorpagefaults,MinorFault,
	10,	RIGHT,	USE_NONE)

DEFCOL(MajorPageFaults,	majorpagefaults,MajorFault,
	10,	RIGHT,	USE_NONE)

DEFCOL(SignalCatch,	signalcatch,	SigCatch,
	8,	RIGHT,	USE_NONE)

DEFCOL(SignalIgnore,	signalignore,	SigIgnor,
	8,	RIGHT,	USE_NONE)

DEFCOL(SignalBlock,	signalblock,	SigBlock,
	8,	RIGHT,	USE_NONE)

DEFCOL(OpenFileCount,	openfiles,	Files,
	5,	RIGHT,	USE_OPEN_FILE)

DEFCOL(RunOrder,	runorder,	RunOrder,
	10,	RIGHT,	USE_NONE)

DEFCOL(CurrentDirectory,currentdirectory,Current Dir,
	25,	LEFT,	USE_CURR_DIR)

DEFCOL(RootDirectory,	rootdirectory,	Root Dir,
	25,	LEFT,	USE_ROOT_DIR)

DEFCOL(Executable,	executable,	Executable,
	25,	LEFT,	USE_EXEC_INODE)

DEFCOL(Summary,		summary,	Summary,
	14,	LEFT,	USE_NONE)

DEFCOL(Nice,		nice,		Nic,
	3,	RIGHT,	USE_NONE)

DEFCOL(Size,		size,		Size,
	5,	RIGHT,	USE_NONE)

DEFCOL(RealTimer,	realtimer,	RealTimer,
	10,	RIGHT,	USE_NONE)

DEFCOL(Stdin,		stdin,		Stdin,
	20,	LEFT,	USE_STDIN)

DEFCOL(Stdout,		stdout,		Stdout,
	20,	LEFT,	USE_STDOUT)

DEFCOL(Stderr,		stderr,		Stderr,
	20,	LEFT,	USE_STDERR)

DEFCOL(Stdio,		stdio,		Stdio,
	5,	CENTER,	USE_STDIN | USE_STDOUT | USE_STDERR)

DEFCOL(Active,		active,		Active,
	5,	CENTER,	USE_INIT)

DEFCOL(Processor,	processor,	Proc,
	4,	RIGHT,	USE_NONE)


/*
 * Table of all columns.
 * The table is NULL terminated.
 */
static	COLUMN *	column_table[] =
{
	&ColumnPid,
	&ColumnState,
	&ColumnParentPid,
	&ColumnSystemTime,
	&ColumnUserTime,
	&ColumnRunTime,
	&ColumnChildRunTime,
	&ColumnPageSwaps,
	&ColumnChildPageSwaps,
	&ColumnEip,
	&ColumnEsp,
	&ColumnWaitChannel,
	&ColumnWaitSymbol,
	&ColumnProgram,
	&ColumnCommand,
	&ColumnEnvironment,
	&ColumnProcessGroup,
	&ColumnTtyProcessGroup,
	&ColumnThreadCount,
	&ColumnIdleTime,
	&ColumnDeadTime,
	&ColumnTtyName,
	&ColumnTtyDevice,
	&ColumnUserName,
	&ColumnGroupName,
	&ColumnUserId,
	&ColumnGroupId,
	&ColumnPercentCpu,
	&ColumnPercentMemory,
	&ColumnRss,
	&ColumnStartTime,
	&ColumnAge,
	&ColumnFlags,
	&ColumnPageFaults,
	&ColumnMinorPageFaults,
	&ColumnMajorPageFaults,
	&ColumnSignalCatch,
	&ColumnSignalIgnore,
	&ColumnSignalBlock,
	&ColumnOpenFileCount,
	&ColumnRunOrder,
	&ColumnCurrentDirectory,
	&ColumnRootDirectory,
	&ColumnExecutable,
	&ColumnSummary,
	&ColumnNice,
	&ColumnPriority,
	&ColumnRealTimePriority,
	&ColumnPolicy,
	&ColumnProcessor,
	&ColumnSize,
	&ColumnRealTimer,
	&ColumnStdin,
	&ColumnStdout,
	&ColumnStderr,
	&ColumnStdio,
	&ColumnActive,
	NULL
};


/*
 * Default columns when nothing else has been set.
 */
static	COLUMN *	default_columns[] =
{
	&ColumnPid,
	&ColumnParentPid,
	&ColumnTtyName,
	&ColumnUserName,
	&ColumnSummary,
	&ColumnRunTime,
	&ColumnCommand,
	NULL
};


/*
 * Temporary buffer to contain constructed column strings.
 */
static	char	show_buffer[256];


/*
 * Other static routines.
 */
static	void	TicksToString(char * buf, long ticks);


/*
 * Default the widths of all the columns.
 */
void
DefaultColumnWidths(void)
{
	COLUMN **	columnptr;
	COLUMN *	column;

	for (columnptr = column_table; *columnptr; columnptr++)
	{
		column = *columnptr;

		column->width = column->initwidth;
	}
}


/*
 * List the available columns to stdout.
 */
void
ListColumns(void)
{
	COLUMN **	columnptr;
	const COLUMN *	column;

	printf("Column Name          Displayed As    Width\n");
	printf("-----------          ------------    -----\n");

	for (columnptr = column_table; *columnptr; columnptr++)
	{
		column = *columnptr;

		printf("%-20s %-15s %3d\n", column->name, column->heading,
			column->width);
	}

	printf("\n");
	printf("Note: Column names may be abbreviated.  They are used in\n");
	printf("several ways: displaying, sorting, and for the cond option.\n");
}


/*
 * Set the default columns.
 */
void
DefaultColumns(void)
{
	COLUMN **	src;
	COLUMN **	dest;

	dest = show_list;
	src = default_columns;
	show_count = 0;

	while (*src)
	{
		*dest++ = *src++;
		show_count++;
	}
}


/*
 * Find the column with the given name.
 * Abbreviations are allowed.
 * Returns NULL if there was no match or it was ambiguous.
 */
COLUMN *
FindColumn(const char * name)
{
	COLUMN **	columnptr;
	COLUMN *	column;
	COLUMN *	match;
	int		count;
	int		len;
	int		namelen;

	match = NULL;
	count = 0;

	namelen = strlen(name);

	for (columnptr = column_table; *columnptr; columnptr++)
	{
		column = *columnptr;

		len = strlen(column->name);

		if ((len < namelen) ||
			((memcmp(name, column->name, namelen) != 0)))
				continue;

		if (len == namelen)
			return column;

		match = column;
		count++;
	}

	if (count == 1)
		return match;

	return NULL;
}


/*
 * Functions to sort various columns based on comparing two processes.
 * This returns a negative value if the item for the first process is less
 * than the second process, a positive value if the item is greater, or
 * zero if the item is the same for both processes.
 */
static int
SortPid(const PROC * proc1, const PROC * proc2)
{
	return proc1->pid - proc2->pid;
}


static int
SortState(const PROC * proc1, const PROC * proc2)
{
	return proc1->state - proc2->state;
}


static int
SortParentPid(const PROC * proc1, const PROC * proc2)
{
	return proc1->parent_pid - proc2->parent_pid;
}


static int
SortThreadCount(const PROC * proc1, const PROC * proc2)
{
	return proc1->threadCount - proc2->threadCount;
}


static int
SortSystemTime(const PROC * proc1, const PROC * proc2)
{
	return proc1->stime - proc2->stime;
}


static int
SortUserTime(const PROC * proc1, const PROC * proc2)
{
	return proc1->utime - proc2->utime;
}


static int
SortRunTime(const PROC * proc1, const PROC * proc2)
{
	return (proc1->stime + proc1->utime) - (proc2->stime + proc2->utime);
}


static int
SortChildRunTime(const PROC * proc1, const PROC * proc2)
{
	return (proc1->cstime + proc1->cutime) -
		(proc2->cstime + proc2->cutime);
}


static int
SortPageSwaps(const PROC * proc1, const PROC * proc2)
{
	return proc1->nswap - proc2->nswap;
}


static int
SortChildPageSwaps(const PROC * proc1, const PROC * proc2)
{
	return proc1->cnswap - proc2->cnswap;
}


static int
SortEip(const PROC * proc1, const PROC * proc2)
{
	return proc1->eip - proc2->eip;
}


static int
SortEsp(const PROC * proc1, const PROC * proc2)
{
	return proc1->esp - proc2->esp;
}


static int
SortWaitChannel(const PROC * proc1, const PROC * proc2)
{
	return proc1->wchan - proc2->wchan;
}


static int
SortWaitSymbol(const PROC * proc1, const PROC * proc2)
{
	return strcmp(proc1->wchan_symbol, proc2->wchan_symbol);
}


static int
SortProgram(const PROC * proc1, const PROC * proc2)
{
	return strcmp(proc1->program, proc2->program);
}


static int
SortCommand(const PROC * proc1, const PROC * proc2)
{
	return strcmp(proc1->command, proc2->command);
}


static int
SortEnvironment(const PROC * proc1, const PROC * proc2)
{
	return strcmp(proc1->environment, proc2->environment);
}


static int
SortProcessGroup(const PROC * proc1, const PROC * proc2)
{
	return proc1->process_group - proc2->process_group;
}


static int
SortTtyProcessGroup(const PROC * proc1, const PROC * proc2)
{
	return proc1->tty_pgrp - proc2->tty_pgrp;
}


static int
SortIdleTime(const PROC * proc1, const PROC * proc2)
{
	return proc2->last_active_time - proc1->last_active_time;
}


static int
SortDeadTime(const PROC * proc1, const PROC * proc2)
{
	return proc2->deathTime - proc1->deathTime;
}


static int
SortTtyName(const PROC * proc1, const PROC * proc2)
{
	const char *	name1;
	const char *	name2;

	if (proc1->tty_dev == proc2->tty_dev)
		return 0;

	name1 = FindDeviceName(proc1->tty_dev);
	name2 = FindDeviceName(proc2->tty_dev);

	if (name1 && name2)
		return strcmp(name1, name2);

	if (!name1 && name2)
		return -1;

	if (name1 && !name2)
		return 1;

	return ((long) proc1->tty_dev) - ((long) proc2->tty_dev);
}


static int
SortTtyDevice(const PROC * proc1, const PROC * proc2)
{
	return ((long) proc1->tty_dev) - ((long) proc2->tty_dev);
}


static int
SortUserName(const PROC * proc1, const PROC * proc2)
{
	const char *	name1;
	const char *	name2;

	if (proc1->uid == proc2->uid)
		return 0;

	name1 = FindUserName(proc1->uid);
	name2 = FindUserName(proc2->uid);

	if (name1 && name2)
		return strcmp(name1, name2);

	if (!name1 && name2)
		return -1;

	if (name1 && !name2)
		return 1;

	return ((long) proc1->uid) - ((long) proc2->uid);
}


static int
SortUserId(const PROC * proc1, const PROC * proc2)
{
	return ((long) proc1->uid) - ((long) proc2->uid);
}


static int
SortGroupName(const PROC * proc1, const PROC * proc2)
{
	const char *	name1;
	const char *	name2;

	if (proc1->gid == proc2->gid)
		return 0;

	name1 = FindGroupName(proc1->gid);
	name2 = FindGroupName(proc2->gid);

	if (name1 && name2)
		return strcmp(name1, name2);

	if (!name1 && name2)
		return -1;

	if (name1 && !name2)
		return 1;

	return ((long) proc1->gid) - ((long) proc2->gid);
}


static int
SortGroupId(const PROC * proc1, const PROC * proc2)
{
	return ((long) proc1->gid) - ((long) proc2->gid);
}


static int
SortPercentCpu(const PROC * proc1, const PROC * proc2)
{
	return proc1->percent_cpu - proc2->percent_cpu;
}


static int
SortPercentMemory(const PROC * proc1, const PROC * proc2)
{
	return proc1->percent_memory - proc2->percent_memory;
}


static int
SortRss(const PROC * proc1, const PROC * proc2)
{
	return proc1->rss - proc2->rss;
}


static int
SortStartTime(const PROC * proc1, const PROC * proc2)
{
	return proc1->start_time_ticks - proc2->start_time_ticks;
}


static int
SortAge(const PROC * proc1, const PROC * proc2)
{
	return proc2->start_time_ticks - proc1->start_time_ticks;
}


static int
SortFlags(const PROC * proc1, const PROC * proc2)
{
	return proc1->flags - proc2->flags;
}


static int
SortPageFaults(const PROC * proc1, const PROC * proc2)
{
	return (proc1->min_flt + proc1->maj_flt) -
		(proc2->min_flt + proc2->maj_flt);
}


static int
SortMinorPageFaults(const PROC * proc1, const PROC * proc2)
{
	return proc1->min_flt - proc2->min_flt;
}


static int
SortMajorPageFaults(const PROC * proc1, const PROC * proc2)
{
	return proc1->maj_flt - proc2->maj_flt;
}


static int
SortSignalCatch(const PROC * proc1, const PROC * proc2)
{
	return proc1->sigcatch - proc2->sigcatch;
}


static int
SortSignalIgnore(const PROC * proc1, const PROC * proc2)
{
	return proc1->sigignore - proc2->sigignore;
}


static int
SortSignalBlock(const PROC * proc1, const PROC * proc2)
{
	return proc1->sigblock - proc2->sigblock;
}


static int
SortOpenFileCount(const PROC * proc1, const PROC * proc2)
{
	if (proc1->openfiles < proc2->openfiles)
		return -1;

	if (proc1->openfiles > proc2->openfiles)
		return 1;

	return 0;
}


static int
SortRunOrder(const PROC * proc1, const PROC * proc2)
{
	if (proc1->run_order < proc2->run_order)
		return -1;

	if (proc1->run_order > proc2->run_order)
		return 1;

	return 0;
}


static int
SortCurrentDirectory(const PROC * proc1, const PROC * proc2)
{
	return strcmp(proc1->cwd_path, proc2->cwd_path);
}


static int
SortRootDirectory(const PROC * proc1, const PROC * proc2)
{
	return strcmp(proc1->root_path, proc2->root_path);
}


static int
SortExecutable(const PROC * proc1, const PROC * proc2)
{
	return strcmp(proc1->exec_path, proc2->exec_path);
}


static int
SortStdin(const PROC * proc1, const PROC * proc2)
{
	return strcmp(proc1->stdio_paths[0], proc2->stdio_paths[0]);
}


static int
SortStdout(const PROC * proc1, const PROC * proc2)
{
	return strcmp(proc1->stdio_paths[1], proc2->stdio_paths[1]);
}


static int
SortStderr(const PROC * proc1, const PROC * proc2)
{
	return strcmp(proc1->stdio_paths[2], proc2->stdio_paths[2]);
}


static int
SortStdio(const PROC * proc1, const PROC * proc2)
{
	char	buf[8];

	strcpy(buf, ShowStdio(proc1));

	return strcmp(buf, ShowStdio(proc2));
}


static int
SortSummary(const PROC * proc1, const PROC * proc2)
{
	char	buf[30];

	strcpy(buf, ShowSummary(proc1));

	return strcmp(buf, ShowSummary(proc2));
}


static int
SortPriority(const PROC * proc1, const PROC * proc2)
{
	if (proc1->priority < proc2->priority)
		return -1;

	if (proc1->priority > proc2->priority)
		return 1;

	return 0;
}


static int
SortPolicy(const PROC * proc1, const PROC * proc2)
{
	if (proc1->policy < proc2->policy)
		return -1;

	if (proc1->policy > proc2->policy)
		return 1;

	return 0;
}


static int
SortRealTimePriority(const PROC * proc1, const PROC * proc2)
{
	if (proc1->realTimePriority < proc2->realTimePriority)
		return -1;

	if (proc1->realTimePriority > proc2->realTimePriority)
		return 1;

	return 0;
}


static int
SortNice(const PROC * proc1, const PROC * proc2)
{
	if (proc1->nice < proc2->nice)
		return -1;

	if (proc1->nice > proc2->nice)
		return 1;

	return 0;
}


static int
SortProcessor(const PROC * proc1, const PROC * proc2)
{
	if (proc1->processor < proc2->processor)
		return -1;

	if (proc1->processor > proc2->processor)
		return 1;

	return 0;
}


static int
SortSize(const PROC * proc1, const PROC * proc2)
{
	if (proc1->vsize < proc2->vsize)
		return -1;

	if (proc1->vsize > proc2->vsize)
		return 1;

	return 0;
}


static int
SortRealTimer(const PROC * proc1, const PROC * proc2)
{
	if (proc1->it_real_value < proc2->it_real_value)
		return -1;

	if (proc1->it_real_value > proc2->it_real_value)
		return 1;

	return 0;
}


static int
SortActive(const PROC * proc1, const PROC * proc2)
{
	if (!proc1->isactive && proc2->isactive)
		return -1;

	if (proc1->isactive && !proc2->isactive)
		return 1;

	return 0;
}


/*
 * Routines to evaluate a column, and return it with the specified type.
 * (This is either a string or a number.)
 */
static void
EvalPid(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->pid;
}


static void
EvalState(const PROC * proc, VALUE * val)
{
	char *	cp;

	cp = AllocTempString(2);
	cp[0] = proc->state;
	cp[1] = '\0';

	val->type = VALUE_STRING;
	val->strval = cp;
}


static void
EvalParentPid(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->parent_pid;
}


static void
EvalThreadCount(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->threadCount;
}


static void
EvalSystemTime(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->stime;
}


static void
EvalUserTime(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->utime;
}


static void
EvalRunTime(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->stime + proc->utime;
}


static void
EvalChildRunTime(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->cstime + proc->cutime;
}


static void
EvalPageSwaps(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->nswap;
}


static void
EvalChildPageSwaps(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->cnswap;
}


static void
EvalEip(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->eip;
}


static void
EvalEsp(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->eip;
}


static void
EvalWaitChannel(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->wchan;
}


static void
EvalWaitSymbol(const PROC * proc, VALUE * val)
{
	val->type = VALUE_STRING;
	val->strval = proc->wchan_symbol;
}


static void
EvalProgram(const PROC * proc, VALUE * val)
{
	val->type = VALUE_STRING;
	val->strval = proc->program;
}


static void
EvalCommand(const PROC * proc, VALUE * val)
{
	val->type = VALUE_STRING;
	val->strval = proc->command;
}


static void
EvalEnvironment(const PROC * proc, VALUE * val)
{
	val->type = VALUE_STRING;
	val->strval = proc->environment;
}


static void
EvalProcessGroup(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->process_group;
}


static void
EvalTtyProcessGroup(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->tty_pgrp;
}


static void
EvalIdleTime(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = (current_time - proc->last_active_time) / 60;
}


static void
EvalDeadTime(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = 0;

	if (proc->deathTime > 0)
		val->intval = (current_time - proc->deathTime) / 60;
}


static void
EvalTtyName(const PROC * proc, VALUE * val)
{
	val->type = VALUE_STRING;
	val->strval = FindDeviceName(proc->tty_dev);

	if (val->strval == NULL)
		val->strval = "";
}


static void
EvalTtyDevice(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = (long) proc->tty_dev;
}


static void
EvalUserName(const PROC * proc, VALUE * val)
{
	val->type = VALUE_STRING;
	val->strval = FindUserName(proc->uid);

	if (val->strval == NULL)
		val->strval = "";
}


static void
EvalGroupName(const PROC * proc, VALUE * val)
{
	val->type = VALUE_STRING;
	val->strval = FindGroupName(proc->uid);

	if (val->strval == NULL)
		val->strval = "";
}


static void
EvalUserId(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = (long) proc->uid;
}


static void
EvalGroupId(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = (long) proc->gid;
}


static void
EvalPercentCpu(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->percent_cpu;
}


static void
EvalPercentMemory(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->percent_memory;
}


static void
EvalRss(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->rss * 4;
}


static void
EvalStartTime(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->start_time_clock;
}


static void
EvalAge(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = current_time - proc->start_time_clock;

	if (val->intval < 0)
		val->intval = 0;
}


static void
EvalFlags(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->flags;
}


static void
EvalPageFaults(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->min_flt + proc->maj_flt;
}


static void
EvalMinorPageFaults(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->min_flt;
}


static void
EvalMajorPageFaults(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->maj_flt;
}


static void
EvalSignalCatch(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->sigcatch;
}


static void
EvalSignalIgnore(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->sigignore;
}


static void
EvalSignalBlock(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->sigblock;
}


static void
EvalOpenFileCount(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->openfiles;
}


static void
EvalRunOrder(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->run_order;
}


static void
EvalCurrentDirectory(const PROC * proc, VALUE * val)
{
	val->type = VALUE_STRING;
	val->strval = proc->cwd_path;
}


static void
EvalRootDirectory(const PROC * proc, VALUE * val)
{
	val->type = VALUE_STRING;
	val->strval = proc->root_path;
}


static void
EvalExecutable(const PROC * proc, VALUE * val)
{
	val->type = VALUE_STRING;
	val->strval = proc->exec_path;
}


static void
EvalStdin(const PROC * proc, VALUE * val)
{
	val->type = VALUE_STRING;
	val->strval = proc->stdio_paths[0];
}


static void
EvalStdout(const PROC * proc, VALUE * val)
{
	val->type = VALUE_STRING;
	val->strval = proc->stdio_paths[1];
}


static void
EvalStderr(const PROC * proc, VALUE * val)
{
	val->type = VALUE_STRING;
	val->strval = proc->stdio_paths[2];
}


static void
EvalStdio(const PROC * proc, VALUE * val)
{
	val->type = VALUE_STRING;
	val->strval = CopyTempString(ShowStdio(proc));
}


static void
EvalSummary(const PROC * proc, VALUE * val)
{
	val->type = VALUE_STRING;
	val->strval = CopyTempString(ShowSummary(proc));
}


static void
EvalPolicy(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->policy;
}


static void
EvalPriority(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->priority;
}


static void
EvalRealTimePriority(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->realTimePriority;
}


static void
EvalNice(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->nice;
}


static void
EvalProcessor(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->processor;
}


static void
EvalSize(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = (proc->vsize + 1023) / 1024;
}


static void
EvalRealTimer(const PROC * proc, VALUE * val)
{
	val->type = VALUE_NUMBER;
	val->intval = proc->it_real_value;
}


static void
EvalActive(const PROC * proc, VALUE * val)
{
	val->type = VALUE_BOOLEAN;
	val->intval = proc->isactive;
}


/*
 * Routines to test whether a column value is "TRUE" in some sense.
 * The meaning differs from column to column, but generally means that
 * the column value is nonzero or non-null.  These functions are used when
 * the "test" attribute is used on a column name.
 */
static BOOL
TestPid(const PROC * proc)
{
	return (proc->pid != 0);
}


static BOOL
TestState(const PROC * proc)
{
	return (proc->state != 'Z');
}


static BOOL
TestThreadCount(const PROC * proc)
{
	return (proc->threadCount > 1);
}


static BOOL
TestParentPid(const PROC * proc)
{
	return (proc->parent_pid && (proc->parent_pid != 1));
}


static BOOL
TestSystemTime(const PROC * proc)
{
	return (proc->stime != 0);
}


static BOOL
TestUserTime(const PROC * proc)
{
	return (proc->utime != 0);
}


static BOOL
TestRunTime(const PROC * proc)
{
	return ((proc->stime != 0) || (proc->utime != 0));
}


static BOOL
TestChildRunTime(const PROC * proc)
{
	return ((proc->cstime != 0) || (proc->cutime != 0));
}


static BOOL
TestPageSwaps(const PROC * proc)
{
	return (proc->nswap != 0);
}


static BOOL
TestChildPageSwaps(const PROC * proc)
{
	return (proc->cnswap != 0);
}


static BOOL
TestEip(const PROC * proc)
{
	return (proc->eip != 0);
}


static BOOL
TestEsp(const PROC * proc)
{
	return (proc->esp != 0);
}


static BOOL
TestWaitChannel(const PROC * proc)
{
	return (proc->wchan != 0);
}


static BOOL
TestWaitSymbol(const PROC * proc)
{
	return (proc->wchan_symbol[0] != '\0');
}


static BOOL
TestProgram(const PROC * proc)
{
	return (proc->program[0] != '\0');
}


static BOOL
TestCommand(const PROC * proc)
{
	return proc->hascommand;
}


static BOOL
TestEnvironment(const PROC * proc)
{
	return (proc->environment[0] != '\0');
}


static BOOL
TestProcessGroup(const PROC * proc)
{
	return (proc->process_group != 0);
}


static BOOL
TestTtyProcessGroup(const PROC * proc)
{
	return ((proc->tty_pgrp != 0) && (proc->tty_pgrp != -1));
}


static BOOL
TestIdleTime(const PROC * proc)
{
	return (proc->last_active_time != current_time);
}


static BOOL
TestDeadTime(const PROC * proc)
{
	return (proc->deathTime != 0);
}


static BOOL
TestTtyName(const PROC * proc)
{
	return ((proc->tty_dev != 0) && (proc->tty_dev != BAD_DEVID));
}


static BOOL
TestTtyDevice(const PROC * proc)
{
	return ((proc->tty_dev != 0) && (proc->tty_dev != BAD_DEVID));
}


static BOOL
TestUserName(const PROC * proc)
{
	return (FindUserName(proc->uid) != NULL);
}


static BOOL
TestGroupName(const PROC * proc)
{
	return (FindGroupName(proc->gid) != NULL);
}


static BOOL
TestUserId(const PROC * proc)
{
	return (proc->uid != 0);
}


static BOOL
TestGroupId(const PROC * proc)
{
	return (proc->gid != 0);
}


static BOOL
TestPercentCpu(const PROC * proc)
{
	return (proc->percent_cpu != 0);
}


static BOOL
TestPercentMemory(const PROC * proc)
{
	return (proc->percent_memory != 0);
}


static BOOL
TestRss(const PROC * proc)
{
	return (proc->rss != 0);
}


static BOOL
TestStartTime(const PROC * proc)
{
	return (proc->start_time_ticks != 0);
}


static BOOL
TestAge(const PROC * proc)
{
	return (proc->start_time_clock < current_time);
}


static BOOL
TestFlags(const PROC * proc)
{
	return (proc->flags != 0);
}


static BOOL
TestPageFaults(const PROC * proc)
{
	return ((proc->min_flt != 0) || (proc->maj_flt != 0));
}


static BOOL
TestMinorPageFaults(const PROC * proc)
{
	return (proc->min_flt != 0);
}


static BOOL
TestMajorPageFaults(const PROC * proc)
{
	return (proc->maj_flt != 0);
}


static BOOL
TestSignalCatch(const PROC * proc)
{
	return (proc->sigcatch != 0);
}


static BOOL
TestSignalIgnore(const PROC * proc)
{
	return (proc->sigignore != 0);
}


static BOOL
TestSignalBlock(const PROC * proc)
{
	return (proc->sigblock != 0);
}


static BOOL
TestOpenFileCount(const PROC * proc)
{
	return (proc->openfiles > 0);
}


static BOOL
TestRunOrder(const PROC * proc)
{
	return (proc->run_order > 0);
}


static BOOL
TestCurrentDirectory(const PROC * proc)
{
	return (proc->cwd_path != empty_string);
}


static BOOL
TestRootDirectory(const PROC * proc)
{
	return (proc->root_path != empty_string);
}


static BOOL
TestExecutable(const PROC * proc)
{
	return (proc->exec_path != empty_string);
}


static BOOL
TestStdin(const PROC * proc)
{
	return (proc->stdio_paths[0] != empty_string);
}


static BOOL
TestStdout(const PROC * proc)
{
	return (proc->stdio_paths[1] != empty_string);
}


static BOOL
TestStderr(const PROC * proc)
{
	return (proc->stdio_paths[2] != empty_string);
}


static BOOL
TestStdio(const PROC * proc)
{
	return ((proc->stdio_paths[0] != '\0') ||
		(proc->stdio_paths[1] != '\0') ||
		(proc->stdio_paths[2] != '\0'));
}


static BOOL
TestSummary(const PROC * proc)
{
	return TRUE;
}


static BOOL
TestPolicy(const PROC * proc)
{
	return (proc->policy != 0);
}


static BOOL
TestPriority(const PROC * proc)
{
	return (proc->priority != 0);
}


static BOOL
TestRealTimePriority(const PROC * proc)
{
	return (proc->realTimePriority != 0);
}


static BOOL
TestNice(const PROC * proc)
{
	return (proc->nice != 0);
}


static BOOL
TestProcessor(const PROC * proc)
{
	return (proc->processor != 0);
}


static BOOL
TestSize(const PROC * proc)
{
	return (proc->vsize != 0);
}


static BOOL
TestRealTimer(const PROC * proc)
{
	return (proc->it_real_value != 0);
}


static BOOL
TestActive(const PROC * proc)
{
	return proc->isactive;
}


/*
 * Functions to show various columns.
 * This just means return a pointer to a string containing the desired
 * information.  The string should be of minimal length, since left or
 * right space padding is done elsewhere.  The string only has to be valid
 * until the next ShowXXX function is called, and so a common buffer can
 * be used for many routines.
 */
static const char *
ShowPid(const PROC * proc)
{
	sprintf(show_buffer, "%ld", (long) proc->pid);

	return show_buffer;
}


static const char *
ShowState(const PROC * proc)
{
	show_buffer[0] = proc->state;
	show_buffer[1] = '\0';

	return show_buffer;
}


static const char *
ShowThreadCount(const PROC * proc)
{
	sprintf(show_buffer, "%d", proc->threadCount);

	return show_buffer;
}


static const char *
ShowParentPid(const PROC * proc)
{
	sprintf(show_buffer, "%ld", (long) proc->parent_pid);

	return show_buffer;
}


static const char *
ShowSystemTime(const PROC * proc)
{
	TicksToString(show_buffer, proc->stime);

	return show_buffer;
}


static const char *
ShowUserTime(const PROC * proc)
{
	TicksToString(show_buffer, proc->utime);

	return show_buffer;
}


static const char *
ShowRunTime(const PROC * proc)
{
	TicksToString(show_buffer, proc->stime + proc->utime);

	return show_buffer;
}


static const char *
ShowChildRunTime(const PROC * proc)
{
	TicksToString(show_buffer, proc->cstime + proc->cutime);

	return show_buffer;
}


static const char *
ShowPageSwaps(const PROC * proc)
{
	sprintf(show_buffer, "%ld", proc->nswap);

	return show_buffer;
}


static const char *
ShowChildPageSwaps(const PROC * proc)
{
	sprintf(show_buffer, "%ld", proc->cnswap);

	return show_buffer;
}


static const char *
ShowEip(const PROC * proc)
{
	sprintf(show_buffer, LONG_HEX_FORMAT, proc->eip);

	return show_buffer;
}


static const char *
ShowEsp(const PROC * proc)
{
	sprintf(show_buffer, LONG_HEX_FORMAT, proc->esp);

	return show_buffer;
}


static const char *
ShowWaitChannel(const PROC * proc)
{
	if (proc->wchan == 0)
		return "";

	sprintf(show_buffer, LONG_HEX_FORMAT, proc->wchan);

	return show_buffer;
}


static const char *
ShowWaitSymbol(const PROC * proc)
{
	return proc->wchan_symbol;
}


static const char *
ShowProgram(const PROC * proc)
{
	return proc->program;
}


static const char *
ShowCommand(const PROC * proc)
{
	return proc->command;
}


static const char *
ShowEnvironment(const PROC * proc)
{
	return proc->environment;
}


static const char *
ShowProcessGroup(const PROC * proc)
{
	sprintf(show_buffer, "%ld", (long) proc->process_group);

	return show_buffer;
}


static const char *
ShowTtyProcessGroup(const PROC * proc)
{
	sprintf(show_buffer, "%ld", (long) proc->tty_pgrp);

	return show_buffer;
}


static const char *
ShowIdleTime(const PROC * proc)
{
	char *	cp;
	long	idle;

	cp = show_buffer;

	if (proc->isancient)
		*cp++ = '+';

	idle = (current_time - proc->last_active_time) / 60;

	if (idle < 60)
		sprintf(cp, "%ld", idle);
	else
		sprintf(cp, "%ld:%02ld", idle / 60, idle % 60);

	return show_buffer;
}


static const char *
ShowDeadTime(const PROC * proc)
{
	char *	cp;
	long	minutes;

	if (proc->deathTime == 0)
		return "-";

	cp = show_buffer;

	minutes = (current_time - proc->deathTime) / 60;

	if (minutes < 60)
		sprintf(cp, "%ld", minutes);
	else
		sprintf(cp, "%ld:%02ld", minutes / 60, minutes % 60);

	return show_buffer;
}


static const char *
ShowTtyName(const PROC * proc)
{
	const char *	name;

	name = FindDeviceName(proc->tty_dev);

	if (name != NULL)
		return name;

	sprintf(show_buffer, "%lx", (long) proc->tty_dev);

	return show_buffer;
}


static const char *
ShowTtyDevice(const PROC * proc)
{
	sprintf(show_buffer, "%lx", (long) proc->tty_dev);

	return show_buffer;
}


static const char *
ShowUserName(const PROC * proc)
{
	const char *	name;

	name = FindUserName(proc->uid);

	if (name)
		return name;

	sprintf(show_buffer, "%ld", (long) proc->uid);

	return show_buffer;
}


static const char *
ShowGroupName(const PROC * proc)
{
	const char *	name;

	name = FindGroupName(proc->gid);

	if (name)
		return name;

	sprintf(show_buffer, "%ld", (long) proc->gid);

	return show_buffer;
}


static const char *
ShowUserId(const PROC * proc)
{
	sprintf(show_buffer, "%ld", (long) proc->uid);

	return show_buffer;
}


static const char *
ShowGroupId(const PROC * proc)
{
	sprintf(show_buffer, "%ld", (long) proc->gid);

	return show_buffer;
}


static const char *
ShowPercentCpu(const PROC * proc)
{
	sprintf(show_buffer, "%d.%02d", proc->percent_cpu / 100,
		proc->percent_cpu % 100);

	return show_buffer;
}


static const char *
ShowPercentMemory(const PROC * proc)
{
	sprintf(show_buffer, "%d.%d", proc->percent_memory / 10,
		proc->percent_memory % 10);

	return show_buffer;
}


static const char *
ShowRss(const PROC * proc)
{
	sprintf(show_buffer, "%ld", proc->rss * 4);

	return show_buffer;
}


static const char *
ShowStartTime(const PROC * proc)
{
	int	proc_age_days;
	char *	cp;

	/*
	 * Get the formatted string for the process's start time:
	 *	"Wed Jun 30 21:49:08 1993\n"
	 */
	cp = ctime(&proc->start_time_clock);

	/*
	 * Show the start time as just the hour and minute, unless the
	 * process started more than one day ago, in which case also give
	 * the number of days it has been around.
	 */
	memcpy(show_buffer, cp + 11, 5);
	show_buffer[5] = '\0';

	proc_age_days = (current_time - proc->start_time_clock)
		/ (60 * 60 * 24);

	if (proc_age_days > 0)
		sprintf(show_buffer + 5, "-%dd", proc_age_days);

	return show_buffer;
}


static const char *
ShowAge(const PROC * proc)
{
	long	minutes;
	long	hours;
	long	days;

	minutes = (current_time - proc->start_time_clock) / 60;

	if (minutes < 0)
		minutes = 0;

	hours = minutes / 60;
	minutes %= 60;

	days = hours / 24;
	hours %= 24;

	if (days)
		sprintf(show_buffer, "%ldd%02ld:%02ld", days, hours, minutes);
	else if (hours)
		sprintf(show_buffer, "%ld:%02ld", hours, minutes);
	else
		sprintf(show_buffer, "%ld", minutes);

	return show_buffer;
}


static const char *
ShowFlags(const PROC * proc)
{
	sprintf(show_buffer, "%lx", proc->flags);

	return show_buffer;
}


static const char *
ShowPageFaults(const PROC * proc)
{
	sprintf(show_buffer, "%ld", proc->min_flt + proc->maj_flt);

	return show_buffer;
}


static const char *
ShowMinorPageFaults(const PROC * proc)
{
	sprintf(show_buffer, "%ld", proc->min_flt);

	return show_buffer;
}


static const char *
ShowMajorPageFaults(const PROC * proc)
{
	sprintf(show_buffer, "%ld", proc->maj_flt);

	return show_buffer;
}


static const char *
ShowSignalCatch(const PROC * proc)
{
	sprintf(show_buffer, "%08lx", proc->sigcatch);

	return show_buffer;
}


static const char *
ShowSignalIgnore(const PROC * proc)
{
	sprintf(show_buffer, "%08lx", proc->sigignore);

	return show_buffer;
}


static const char *
ShowSignalBlock(const PROC * proc)
{
	sprintf(show_buffer, "%08lx", proc->sigblock);

	return show_buffer;
}


static const char *
ShowOpenFileCount(const PROC * proc)
{
	if (proc->openfiles < 0)
		return "-";

	sprintf(show_buffer, "%d", proc->openfiles);

	return show_buffer;
}


static const char *
ShowRunOrder(const PROC * proc)
{
	sprintf(show_buffer, "%lu", proc->run_order);

	return show_buffer;
}


static const char *
ShowCurrentDirectory(const PROC * proc)
{
	return proc->cwd_path;
}


static const char *
ShowRootDirectory(const PROC * proc)
{
	return proc->root_path;
}


static const char *
ShowExecutable(const PROC * proc)
{
	return proc->exec_path;
}


static const char *
ShowStdin(const PROC * proc)
{
	return proc->stdio_paths[0];
}


static const char *
ShowStdout(const PROC * proc)
{
	return proc->stdio_paths[1];
}


static const char *
ShowStderr(const PROC * proc)
{
	return proc->stdio_paths[2];
}


static const char *
ShowStdio(const PROC * proc)
{
	char *	path;
	char *	tag;
	int	fd;

	strcpy(show_buffer, "---");

	/*
	 * Loop over the stdio file descriptors.
	 */
	for (fd = 0; fd <= 2; fd++)
	{
		path = proc->stdio_paths[fd];
		tag = &show_buffer[fd];

		/*
		 * If there is no path then leave the dash.
		 */
		if (*path == '\0')
			continue;

		/*
		 * If this is a pipe then say so.
		 */
		if (memcmp(path, "pipe:", 5) == 0)
		{
			*tag = 'P';

			continue;
		}

		/*
		 * If this is a socket then say so.
		 */
		if (memcmp(path, "socket:", 7) == 0)
		{
			*tag = 'S';

			continue;
		}

		/*
		 * If this is the null device say so.
		 */
		if (strcmp(path, "/dev/null") == 0)
		{
			*tag = 'N';

			continue;
		}

		/*
		 * If this is a terminal say so.
		 */
		if ((memcmp(path, "/dev/tty", 8) == 0) ||
			(memcmp(path, "/dev/pts/", 9) == 0) ||
			(memcmp(path, "/dev/vc/", 8) == 0))
		{
			*tag = 'T';

			continue;
		}

		/*
		 * Don't know what it is, say it is just some file.
		 */
		*tag = 'F';
	}

	return show_buffer;
}


static const char *
ShowSummary(const PROC * proc)
{
	long	age;
	char *	cp;

	cp = show_buffer;
	strcpy(cp, "--------------");

	if (proc->deathTime != 0)
		cp[0] = '-';
	else if ((proc->state == 'R') || (proc->state == 'Z') ||
		(proc->state == 'T') || (proc->state == 'D'))
	{
		cp[0] = proc->state;	/* copy state */
	}
	else if (proc->isactive)
		cp[0] = 'A';		/* active */
	else
		cp[0] = 'I';		/* or idle */

	if ((proc->rss == 0) && (proc->vsize != 0) && (proc->state != 'Z'))
		cp[1] = 'W';

	if (proc->nice > 0)
		cp[2] = 'N';
	else if (proc->nice < 0)
		cp[2] = 'H';

	if (proc->session_id == proc->pid)
		cp[3] = 'S';		/* session id leader */

	if (proc->process_group == proc->pid)
		cp[4] = 'P';		/* process group leader */

	if ((proc->tty_dev != 0) && (proc->tty_dev != BAD_DEVID))
		cp[5] = 'T';		/* on a terminal */

	if (proc->tty_pgrp == proc->process_group)
		cp[6] = 'F';		/* foreground process */

	if (proc->parent_pid == 1)
		cp[7] = 'I';		/* owned by init */

	if (proc->sigignore & (1 << (SIGHUP - 1)))
		cp[8] = 'H';		/* ignores SIGHUP */
	else if (proc->sigcatch & (1 << (SIGHUP - 1)))
		cp[8] = 'h';		/* catches SIGHUP */

	if (proc->sigignore & (1 << (SIGTERM - 1)))
		cp[9] = 'T';		/* ignores SIGTERM */
	else if (proc->sigcatch & (1 << (SIGTERM - 1)))
		cp[9] = 't';		/* catches SIGTERM */

	if (proc->uid == my_uid)
		cp[10] = 'U';		/* has my user id */

	if (proc->gid == my_gid)
		cp[11] = 'G';		/* has my group id */

	if (proc->uid == 0)
		cp[12] = 'R';		/* root process */
	else if (proc->uid < BASE_USER_UID)
		cp[12] = 'S';		/* server process */

	age = current_time - proc->start_time_clock;

	if (age >= (60 * 60 * 24 * 7))
		cp[13] = 'W';		/* week old */
	else if (age >= (60 * 60 * 24))
		cp[13] = 'D';		/* day old */
	else if (age >= (60 * 60))
		cp[13] = 'H';		/* hour old */
	else if (age >= (60 * 10))
		cp[13] = 'T';		/* ten minutes old */
	else if (age >= (60 * 5))
		cp[13] = 'F';		/* five minutes old */
	else if (age >= 60)
		cp[13] = 'M';		/* minute old */
	else
		cp[13] = 'N';		/* new process */

	return show_buffer;
}


static const char *
ShowPolicy(const PROC * proc)
{
	sprintf(show_buffer, "%ld", proc->policy);

	return show_buffer;
}


static const char *
ShowPriority(const PROC * proc)
{
	sprintf(show_buffer, "%ld", proc->priority);

	return show_buffer;
}


static const char *
ShowRealTimePriority(const PROC * proc)
{
	sprintf(show_buffer, "%ld", proc->realTimePriority);

	return show_buffer;
}


static const char *
ShowNice(const PROC * proc)
{
	sprintf(show_buffer, "%ld", proc->nice);

	return show_buffer;
}


static const char *
ShowProcessor(const PROC * proc)
{
	sprintf(show_buffer, "%ld", (long) proc->processor);

	return show_buffer;
}


static const char *
ShowSize(const PROC * proc)
{
	sprintf(show_buffer, "%ld", (proc->vsize + 1023) / 1024);

	return show_buffer;
}


static const char *
ShowRealTimer(const PROC * proc)
{
	long	intpart;
	long	fracpart;

	intpart = proc->it_real_value / TICKS;
	fracpart = proc->it_real_value % TICKS;

	if (fracpart)
	{
		sprintf(show_buffer, "%ld.%02ld", intpart,
			((fracpart * 100) / TICKS));
	} else
		sprintf(show_buffer, "%ld", intpart);

	return show_buffer;
}


static const char *
ShowActive(const PROC * proc)
{
	return (proc->isactive ? "active" : "idle");
}


/*
 * Convert a time in ticks into hours, minutes, seconds and hundreths of
 * seconds in the form:
 *	hhh:mm:ss.hh
 * and store that into the supplied buffer.
 */
static void
TicksToString(char * buf, long ticks)
{
	int	hundreths;
	int	seconds;
	int	minutes;
	long	hours;

	hundreths = ticks % TICKS;
	ticks /= TICKS;

	seconds = ticks % 60;
	ticks /= 60;

	minutes = ticks % 60;
	ticks /= 60;

	hours = ticks;

	if (hours > 0)
	{
		sprintf(buf, "%ld:%02d:%02d.%02d", hours, minutes, seconds,
			hundreths);
	}
	else if (minutes > 0)
		sprintf(buf, "%d:%02d.%02d", minutes, seconds, hundreths);
	else
		sprintf(buf, "%d.%02d", seconds, hundreths);
}

/* END CODE */
