;;;; conditions.lisp -- various error conditions

(in-package :ironclad)

(define-condition ironclad-error (simple-error)
  ()
  (:documentation "All errors signaled by IRONCLAD are of this type."))

(define-condition initialization-vector-not-supplied (ironclad-error)
  ((mode :initarg :mode :reader cipher-mode))
  (:report (lambda (condition stream)
             (format stream "Mode ~A requires an initialization vector"
                     (cipher-mode condition))))
  (:documentation "Signaled when an initialization vector is required
for a particular mode of operation but not supplied."))

(define-condition invalid-initialization-vector (ironclad-error)
  ((cipher :initarg :cipher :reader cipher)
   (block-length :initarg :block-length :reader block-length))
  (:report (lambda (condition stream)
             (format stream "Cipher ~A requires an initialization vector of length ~D"
                     (cipher condition)
                     (block-length condition))))
  (:documentation "Signaled when an invalid initialization vector is supplied to MAKE-CIPHER."))

(define-condition invalid-key-length (ironclad-error)
  ((cipher :initarg :cipher :reader cipher)
   (lengths :initarg :accepted-lengths :reader accepted-lengths))
  (:report (lambda (condition stream)
             (format stream "Cipher ~A only accepts keys of these lengths: ~A"
                     (cipher condition)
                     (accepted-lengths condition))))
  (:documentation "Signaled when a key is not the proper length for a cipher."))

(define-condition unsupported-cipher (ironclad-error)
  ((cipher :initarg :name :reader cipher))
  (:report (lambda (condition stream)
             (format stream "Cipher ~A is not a supported cipher"
                     (cipher condition))))
  (:documentation "Signaled when an invalid cipher name is provided to MAKE-CIPHER."))

(define-condition unsupported-mode (ironclad-error)
  ((mode :initarg :mode :reader mode))
  (:report (lambda (condition stream)
             (format stream "Mode ~A is not a supported mode"
                     (mode condition))))
  (:documentation "Signaled when an invalid mode name is provided to MAKE-CIPHER."))
