/*
 bot-events.c : IRC bot plugin for irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "bot.h"

static gint get_flags(USER_REC *user, CHANNEL_REC *channel)
{
    USER_CHAN_REC *userchan;

    userchan = g_hash_table_lookup(user->channels, channel->name);
    return (user->flags | (userchan == NULL ? 0 : userchan->flags)) & (~user->not_flags);
}

static gboolean event_massjoin(CHANNEL_REC *channel, GList *users)
{
    USER_REC *user;
    GString *opstr, *vstr;
    gint flags;

    g_return_val_if_fail(channel != NULL, FALSE);
    g_return_val_if_fail(users != NULL, FALSE);

    opstr = g_string_new(NULL);
    vstr = g_string_new(NULL);
    for (; users != NULL; users = users->next)
    {
	user = users->data;

	if (user->nickrec == NULL)
            continue; /* shouldn't happen! */

	flags = get_flags(user, channel);
	if (user->nickrec->op || ((flags & (USER_AUTO_OP|USER_AUTO_VOICE)) == 0) ||
	    (user->nickrec->voice && (flags & USER_AUTO_OP) == 0))
	{
	    /* No need to do anything to this user.. */
	    continue;
	}

	g_string_sprintfa((flags & USER_AUTO_OP) ? opstr : vstr, "%s ", user->nickrec->nick);
    }
    if (opstr->len > 0)
	signal_emit("command op", 3, opstr->str, channel->server, channel);
    if (vstr->len > 0)
	signal_emit("command voice", 3, vstr->str, channel->server, channel);

    g_string_free(opstr, TRUE);
    g_string_free(vstr, TRUE);
    return TRUE;
}

/* Parse channel mode string */
static void parse_channel_mode(CHANNEL_REC *channel, gchar *modestr, gchar *nick, gchar *address)
{
    NICK_REC *nickrec, *splitnick;
    USER_REC *user;
    GString *str;
    gchar *ptr, *mode, type;
    gint flags;

    g_return_if_fail(channel != NULL);
    g_return_if_fail(nick != NULL);
    g_return_if_fail(modestr != NULL);

    user = botuser_find(nick, address);
    flags = user == NULL ? 0 : get_flags(user, channel);

    if (!channel->chanop || g_strcasecmp(nick, channel->server->nick) == 0 || flags & USER_MASTER)
    {
	/* can't do anything or we/master did mode change,
	   don't bother checking what */
	return;
    }

    str = g_string_new(NULL);
    type = '+';
    for (mode = cmd_get_param(&modestr); *mode != '\0'; mode++)
    {
	if (*mode == '+' || *mode == '-')
	{
	    type = *mode;
	    continue;
	}

	if (*mode != 'o' || type == '-')
	    continue;

	/* opped someone, check if it's a valid op */
	ptr = cmd_get_param(&modestr);
	if (*ptr == '\0')
	    continue;

	if (strcmp(channel->server->nick, ptr) == 0)
	{
	    /* we aren't chanop anymore .. */
	    g_string_truncate(str, 0);
	    break;
	}

	/* find out who was opped.. */
	nickrec = nicklist_find(channel, ptr);
	if (nickrec == NULL || nickrec->host == NULL)
	    continue;

	user = botuser_find(ptr, nickrec->host);
	flags = user == NULL ? 0 : get_flags(user, channel);
	if (flags & USER_OP)
	    continue;

	if (address == NULL)
	{
	    /* server opped, check if user was opped before netsplit. */
	    splitnick = netsplit_find_channel(channel->server, nickrec->nick, nickrec->host, channel->name);
	    if (splitnick != NULL && splitnick->op)
		continue;
	}

	/* this one isn't supposed to get ops! */
	g_string_sprintfa(str, "%s ", ptr);
    }

    if (str->len != 0)
	signal_emit("command deop", 3, str->str, channel->server, channel);
    g_string_free(str, TRUE);
}

static gboolean event_mode(gchar *data, SERVER_REC *server, gchar *nick, gchar *address)
{
    CHANNEL_REC *chanrec;
    gchar *params, *channel, *mode;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 2 | PARAM_FLAG_GETREST, &channel, &mode);

    if (ischannel(*channel))
    {
        /* channel mode change */
        chanrec = channel_find(server, channel);
        if (chanrec != NULL)
            parse_channel_mode(chanrec, mode, nick, address);
    }

    g_free(params);
    return TRUE;
}

void plugin_bot_events(PLUGIN_REC *plugin)
{
    plugin_bind(plugin, "bot massjoin", (SIGNAL_FUNC) event_massjoin, TRUE);
    plugin_bind(plugin, "event mode", (SIGNAL_FUNC) event_mode, FALSE);
}
