/*
 dialog-plugins.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

static GtkWidget *clist;
static gint plugin_menus;

static void clist_append_plugin(GtkWidget *clist, PLUGIN_SETUP_REC *plugin)
{
    gchar *titles[4];

    g_return_if_fail(clist != NULL);
    g_return_if_fail(plugin != NULL);

    titles[0] = plugin_find(plugin->name) ? _("Yes") : _("No");
    titles[1] = plugin->autorun ? _("Yes") : _("No");
    titles[2] = plugin->name;
    titles[3] = plugin->description;

    gtk_clist_set_row_data(GTK_CLIST(clist), gtk_clist_append(GTK_CLIST(clist), titles), plugin);
}

static gboolean dialog_plugins_update(void)
{
    GList *tmp;

    if (clist == NULL) return TRUE;

    gtk_clist_freeze(GTK_CLIST(clist));
    gtk_clist_clear(GTK_CLIST(clist));
    for (tmp = g_list_first(availplugins); tmp != NULL; tmp = tmp->next)
        clist_append_plugin(clist, tmp->data);
    gtk_clist_thaw(GTK_CLIST(clist));
    return TRUE;
}

static void sig_load(GtkWidget *clist)
{
    PLUGIN_SETUP_REC *rec;
    gint selection;

    g_return_if_fail(clist != NULL);

    selection = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(clist), "selection"));
    if (selection == -1) return;

    rec = gtk_clist_get_row_data(GTK_CLIST(clist), selection);

    if (!plugin_load(rec->path, rec->args))
        gui_dialog(DIALOG_ERROR, _("Can't load plugin: %s"), rec->path);
}

static void sig_unload(GtkWidget *clist)
{
    PLUGIN_SETUP_REC *rec;
    PLUGIN_REC *plugin;
    gint selection;

    g_return_if_fail(clist != NULL);

    selection = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(clist), "selection"));
    if (selection == -1) return;

    rec = gtk_clist_get_row_data(GTK_CLIST(clist), selection);
    plugin = plugin_find(rec->name);
    if (plugin == NULL)
        gui_dialog(DIALOG_ERROR, _("Plugin not loaded: %s"), rec->name);
    else
        plugin_destroy(plugin);
}

static void sig_select_row(GtkWidget *clist, gint row)
{
    g_return_if_fail(clist != NULL);

    gtk_object_set_data(GTK_OBJECT(clist), "selection", GINT_TO_POINTER(row));
}

static void sig_unselect_row(GtkWidget *clist)
{
    g_return_if_fail(clist != NULL);

    gtk_object_set_data(GTK_OBJECT(clist), "selection", GINT_TO_POINTER(-1));
}

static void sig_autorun_clicked(GtkWidget *dialog)
{
    gint autorun;

    autorun = !GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(dialog), "autorun"));
    gtk_object_set_data(GTK_OBJECT(dialog), "autorun", GINT_TO_POINTER(autorun));
}

static void sig_edit_ok(GtkWidget *dialog)
{
    proplist_t pprop, iprop;
    PLUGIN_SETUP_REC *plugin;
    GtkWidget *entry;
    gboolean autorun;
    gchar *args;

    autorun = (gboolean) GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(dialog), "autorun"));
    entry = gtk_object_get_data(GTK_OBJECT(dialog), "args");
    plugin = gtk_object_get_data(GTK_OBJECT(dialog), "plugin");
    args = gtk_entry_get_text(GTK_ENTRY(entry));

    plugin->autorun = autorun;
    if (plugin->args != NULL) g_free(plugin->args);
    plugin->args = *args == '\0' ? NULL : g_strdup(args);

    pprop = config_section(&cprop, "plugins");
    iprop = config_section(&pprop, plugin->name);
    config_set_bool(iprop, "autorun", autorun);
    if (args != NULL) config_set_str(iprop, "args", args);
    if (!readonly) PLSave(cprop, TRUE);

    gtk_widget_destroy(dialog);
    dialog_plugins_update();
}

static void sig_edit(GtkWidget *clist)
{
    GtkWidget *dialog, *button, *label, *entry;
    PLUGIN_SETUP_REC *plugin;
    gint selection;
    gchar *str;

    g_return_if_fail(clist != NULL);

    selection = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(clist), "selection"));
    if (selection == -1) return;
    plugin = gtk_clist_get_row_data(GTK_CLIST(clist), selection);

    dialog = gnome_dialog_new(PACKAGE, GNOME_STOCK_BUTTON_OK,
                              GNOME_STOCK_BUTTON_CANCEL, NULL);
    gtk_object_set_data(GTK_OBJECT(dialog), "plugin", plugin);
    gtk_signal_connect_object(GTK_OBJECT(dialog), "delete_event",
                              GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));
    gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);
    gtk_object_set_data(GTK_OBJECT(dialog), "clist", clist);

    str = g_strdup_printf(_("Plugin %s properties:"), plugin->name);
    label = gtk_label_new(str); g_free(str);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), label, FALSE, FALSE, 0);

    button = gtk_check_button_new_with_label(_("Automatically start plugin at startup"));
    gtk_object_set_data(GTK_OBJECT(dialog), "autorun", GINT_TO_POINTER((gint) plugin->autorun));
    if (plugin->autorun) gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
    gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                              GTK_SIGNAL_FUNC(sig_autorun_clicked), GTK_OBJECT(dialog));
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), button, FALSE, FALSE, 0);

    entry = gui_create_labelentry(GNOME_DIALOG(dialog)->vbox, _("Arguments for plugin:"), plugin->args, FALSE);
    gtk_object_set_data(GTK_OBJECT(dialog), "args", entry);

    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, GTK_SIGNAL_FUNC(sig_edit_ok), GTK_OBJECT(dialog));
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 1, GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(dialog));

    gtk_widget_show_all(dialog);
}

static void sig_destroy(GtkWidget **dialog)
{
    g_return_if_fail(dialog != NULL);
    g_return_if_fail(*dialog != NULL);

    gtk_widget_destroy(*dialog);
    *dialog = NULL;
    clist = NULL;
}

static void clist_click_column(GtkCList *clist, gint column)
{
    if (column != clist->sort_column)
	gtk_clist_set_sort_column(clist, column);
    else
    {
	if (clist->sort_type == GTK_SORT_ASCENDING)
	    clist->sort_type = GTK_SORT_DESCENDING;
	else
	    clist->sort_type = GTK_SORT_ASCENDING;
    }

    gtk_clist_sort(clist);
}

void dialog_plugins(void)
{
    static GtkWidget *dialog = NULL;
    GtkWidget *scrollbox;
    gchar *titles[4];
    GList *tmp;

    if (dialog != NULL)
    {
        /* dialog already open */
        gdk_window_raise(dialog->window);
        return;
    }

    dialog = gnome_dialog_new(PACKAGE, _("Load"), _("Unload"), _("Settings..."), GNOME_STOCK_BUTTON_CLOSE, NULL);
    gtk_widget_set_usize(dialog, -1, 250);
    gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, TRUE, FALSE);
    gtk_signal_connect_object(GTK_OBJECT(dialog), "destroy",
                              GTK_SIGNAL_FUNC(sig_destroy), (GtkObject *) &dialog);
    gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
                       GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);

    /* Create plugin clist widget */
    scrollbox = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollbox),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), scrollbox, TRUE, TRUE, 0);

    titles[0] = _("Running"); titles[1] = _("Autorun"); titles[2] = _("Name"); titles[3] = _("Description");
    clist = gtk_clist_new_with_titles(4, titles);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 0, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 1, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 2, TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 3, TRUE);
    gtk_container_add(GTK_CONTAINER(scrollbox), clist);
    gtk_object_set_data(GTK_OBJECT(clist), "selection", GINT_TO_POINTER(-1));
    gtk_clist_set_selection_mode(GTK_CLIST(clist), GTK_SELECTION_BROWSE);
    gtk_signal_connect(GTK_OBJECT(clist), "select_row",
                       GTK_SIGNAL_FUNC(sig_select_row), NULL);
    gtk_signal_connect(GTK_OBJECT(clist), "unselect_row",
                       GTK_SIGNAL_FUNC(sig_unselect_row), NULL);
    gtk_signal_connect(GTK_OBJECT(clist), "click_column",
		       GTK_SIGNAL_FUNC(clist_click_column), NULL);

    gtk_clist_freeze(GTK_CLIST(clist));
    for (tmp = g_list_first(availplugins); tmp != NULL; tmp = tmp->next)
        clist_append_plugin(clist, tmp->data);
    gtk_clist_thaw(GTK_CLIST(clist));
    gtk_clist_select_row(GTK_CLIST(clist), 0, 0);

    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 0, GTK_SIGNAL_FUNC(sig_load), GTK_OBJECT(clist));
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 1, GTK_SIGNAL_FUNC(sig_unload), GTK_OBJECT(clist));
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 2, GTK_SIGNAL_FUNC(sig_edit), GTK_OBJECT(clist));
    gnome_dialog_button_connect_object(GNOME_DIALOG(dialog), 3, GTK_SIGNAL_FUNC(sig_destroy), (GtkObject *) &dialog);

    /* Make columns resizeable */
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 0, TRUE);
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 1, TRUE);
    gtk_clist_set_column_resizeable(GTK_CLIST(clist), 2, TRUE);
    gtk_widget_show_all(dialog);
}

static void plugin_window_add_menu(gchar *after, GnomeUIInfo *menu, PLUGIN_REC *plugin)
{
    GList *tmp;

    for (tmp = g_list_first(mainwindows); tmp != NULL; tmp = tmp->next)
        gui_menus_insert(tmp->data, after, menu, plugin);
}

static gboolean plugin_add_menu(gchar *label, gpointer func, PLUGIN_REC *plugin)
{
    gchar *str;
    GnomeUIInfo menu[] =
    {
        { GNOME_APP_UI_ITEM, NULL, NULL, NULL, NULL, NULL, 0, 0, 0, 0, NULL },
        GNOMEUIINFO_END
    };
    GnomeUIInfo submenu[] =
    {
        { GNOME_APP_UI_SUBTREE, NULL, NULL, NULL, NULL, NULL, 0, 0, 0, 0, NULL },
        GNOMEUIINFO_END
    };
    GnomeUIInfo mainmenu[] =
    {
        { GNOME_APP_UI_SUBTREE, NULL, NULL, NULL, NULL, NULL, 0, 0, 0, 0, NULL },
        GNOMEUIINFO_END
    };

    g_return_val_if_fail(plugin != NULL, FALSE);

    if (label == NULL)
        menu[0].type = GNOME_APP_UI_SEPARATOR;
    else
        menu[0].label = label;
    menu[0].moreinfo = func;

    if (plugin->gui_data == NULL)
    {
        /* first menu item for this plugin, create the plugin's main menu first.. */
	submenu[0].label = plugin->name;
        submenu[0].moreinfo = menu;

        plugin->gui_data = GINT_TO_POINTER(TRUE);
	if (plugin_menus > 0)
	    plugin_window_add_menu(_("Plugins/"), submenu, plugin);
	else
	{
	    /* First plugin, create plugins menu. */
	    mainmenu[0].label = _("Plugins");
	    mainmenu[0].moreinfo = submenu;
	    plugin_window_add_menu(NULL, mainmenu, plugin);
	    plugin_menus++;
	}
    }
    else
    {
        str = g_strdup_printf(_("Plugins/%s/"), plugin->name);
        plugin_window_add_menu(str, menu, plugin);
        g_free(str);
    }

    return TRUE;
}

static gboolean plugin_destroyed(PLUGIN_REC *plugin)
{
    GList *tmp;
    gchar *str;

    g_return_val_if_fail(plugin != NULL, FALSE);

    plugin_menus--;
    str = plugin_menus == 0 ?
	g_strdup(_("Plugins")) :
	g_strdup_printf(_("Plugins/%s"), plugin->name);;

    for (tmp = g_list_first(mainwindows); tmp != NULL; tmp = tmp->next)
	gui_menus_remove(tmp->data, str);

    g_free(str);

    dialog_plugins_update();
    return TRUE;
}

void dialog_plugins_init(void)
{
    plugin_menus = 0;
    clist = NULL;

    signal_add("plugin created", (SIGNAL_FUNC) dialog_plugins_update);
    signal_add("plugin destroyed", (SIGNAL_FUNC) plugin_destroyed);
    signal_add("plugin add menu", (SIGNAL_FUNC) plugin_add_menu);
}

void dialog_plugins_deinit(void)
{
    signal_remove("plugin created", (SIGNAL_FUNC) dialog_plugins_update);
    signal_remove("plugin destroyed", (SIGNAL_FUNC) plugin_destroyed);
    signal_remove("plugin add menu", (SIGNAL_FUNC) plugin_add_menu);
}
