/*
 gui-depends.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

typedef struct
{
    GtkWidget *widget;
    gchar *signal;
    gpointer data;
}
DEPEND_REC;

static GList *depends = NULL;

static int gui_depends_destroy_node(GNode *node, gpointer data)
{
    gtk_widget_destroy(GTK_WIDGET(node->data));
    return 0;
}

static void gui_depends_destroyed(GtkWidget *widget, GNode **nodes)
{
    static gboolean destroying = FALSE;
    GNode *node;

    g_return_if_fail(widget != NULL);
    g_return_if_fail(nodes != NULL);

    if (destroying) return;
    destroying = TRUE;

    node = g_node_find(*nodes, G_IN_ORDER, G_TRAVERSE_ALL, widget);
    g_return_if_fail(node != NULL);

    g_node_traverse(node, G_IN_ORDER, G_TRAVERSE_ALL, -1,
                    (GNodeTraverseFunc) gui_depends_destroy_node, NULL);
    g_node_destroy(node);

    if (node == *nodes) *nodes = NULL; /* root killed. */
    destroying = FALSE;
}

void gui_widget_depends(GtkWidget *parent, GtkWidget *child)
{
    static GNode *nodes = NULL;
    GNode *parentnode;

    g_return_if_fail(parent != NULL);
    g_return_if_fail(child != NULL);

    /* find parent */
    parentnode = nodes == NULL ? NULL :
        g_node_find(nodes, G_IN_ORDER, G_TRAVERSE_ALL, parent);

    if (parentnode == NULL)
    {
        /* paren't doesn't exist yet, create */
        parentnode = g_node_new(parent);
	if (nodes == NULL)
	    nodes = parentnode;
	else
	    g_node_append(nodes, parentnode);

        gtk_signal_connect(GTK_OBJECT(parent), "destroy",
                           GTK_SIGNAL_FUNC(gui_depends_destroyed), &nodes);
    }

    g_node_append_data(parentnode, child);

    /* add child to tree */
    gtk_signal_connect(GTK_OBJECT(child), "destroy",
                       GTK_SIGNAL_FUNC(gui_depends_destroyed), &nodes);
}

static DEPEND_REC *gui_depends_find_signal(gchar *signal)
{
    GList *node;

    for (node = g_list_first(depends); node != NULL; node = node->next)
    {
        DEPEND_REC *rec = node->data;

        if (g_strcasecmp(rec->signal, signal) == 0)
            return rec;
    }

    return NULL;
}

static gboolean gui_depends_data_destroyed(gpointer data);

static void gui_depends_remove_signals(GList *signals)
{
    GList *tmp;

    for (tmp = g_list_first(signals); tmp != NULL; tmp = tmp->next)
    {
        if (gui_depends_find_signal(tmp->data) == NULL)
            signal_remove(tmp->data, (SIGNAL_FUNC) gui_depends_data_destroyed);
        g_free(tmp->data);
    }

    g_list_free(signals);
}

static gboolean gui_depends_data_destroyed(gpointer data)
{
    GList *node, *next, *signals;

    if (data == NULL) return TRUE;

    /* data destroyed, destroy widgets related to it */
    signals = NULL;
    for (node = g_list_first(depends); node != NULL; node = next)
    {
        DEPEND_REC *rec = node->data;

        next = node->next;
        if (rec->data == data)
        {
            depends = g_list_remove_link(depends, node);
            g_list_free_1(node);

            if (glist_find_icase_string(signals, rec->signal) == NULL)
                signals = g_list_append(signals, rec->signal);
            else
                g_free(rec->signal);
            gtk_widget_destroy(rec->widget);
            g_free(rec);
            break;
        }
    }

    gui_depends_remove_signals(signals);
    return TRUE;
}

static void gui_depends_data_widget_destroyed(GtkWidget *widget)
{
    GList *node, *next, *signals;

    /* widget destroyed, remove everything related to it */
    signals = NULL;
    for (node = g_list_first(depends); node != NULL; node = next)
    {
        DEPEND_REC *rec = node->data;

        next = node->next;
        if (rec->widget == widget)
        {
            depends = g_list_remove_link(depends, node);
            g_list_free_1(node);

            if (glist_find_icase_string(signals, rec->signal) == NULL)
                signals = g_list_append(signals, rec->signal);
            else
                g_free(rec->signal);
            g_free(rec);
            break;
        }
    }

    gui_depends_remove_signals(signals);
}

void gui_widget_depends_data(GtkWidget *widget, gchar *signal, gpointer data)
{
    DEPEND_REC *rec;

    g_return_if_fail(widget != NULL);
    g_return_if_fail(signal != NULL);
    g_return_if_fail(data != NULL);

    /* check if signal is already grabbed */
    if (gui_depends_find_signal(signal) == NULL)
        signal_add(signal, (SIGNAL_FUNC) gui_depends_data_destroyed);

    rec = g_new0(DEPEND_REC, 1);
    depends = g_list_append(depends, rec);

    rec->widget = widget;
    rec->signal = g_strdup(signal);
    rec->data = data;

    gtk_signal_connect(GTK_OBJECT(widget), "destroy",
                       GTK_SIGNAL_FUNC(gui_depends_data_widget_destroyed), NULL);
}
