/*
 lag.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

#define LAG_SEND_ANOTHER_PING 30

typedef struct
{
    SERVER_REC *server;

    gchar *address;
    struct timeval time;
}
LAG_REC;

static gint timeout_tag;
static GList *lags;

static void lag_free(LAG_REC *rec)
{
    lags = g_list_remove(lags, rec);

    g_free(rec->address);
    g_free(rec);
}

void lag_get(SERVER_REC *server, gchar *address)
{
    LAG_REC *lag;
    gchar *str;

    g_return_if_fail(server != NULL);

    lag = g_new0(LAG_REC, 1);
    lags = g_list_append(lags, lag);
    lag->server = server;
    lag->address = g_strdup(address != NULL ? address : server->real_address);

    if (gettimeofday(&lag->time, NULL) != 0)
    {
	g_warning("check_lag() : gettimeofday() failed\n");
	lag_free(lag);
        return;
    }

    if (address == NULL)
    {
	if (server->lag_sent == 0)
	    server->lag_sent = time(NULL);
	server->lag_last_check = time(NULL);
    }

    /* NOTE: the function that calls lag_get() should check that there's no
       other commands waiting in buffer or this will give wrong results! */
    str = g_strdup_printf("PING LAG%ld.%ld %s",
			  lag->time.tv_sec, lag->time.tv_usec, lag->address);
    irc_send_cmd(server, str);
    g_free(str);

    server_redirect_event(server, lag->address, 1, "event pong", "lag event pong", 0, NULL);
}

static gboolean event_pong(gchar *data, SERVER_REC *server)
{
    LAG_REC *lag;
    GList *tmp;
    struct timeval now, sent;
    gchar *params, *servername, *msg;
    glong secs, usecs, lagtime;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 2, &servername, &msg);

    /* find the lag record */
    lag = NULL;
    for (tmp = lags; tmp != NULL; tmp = tmp->next)
    {
	lag = tmp->data;

	if (lag->server == server && g_strcasecmp(lag->address, servername) == 0)
	    break;
    }

    if (tmp == NULL)
    {
	/* not found - just ignore it */
	g_free(params);
	return TRUE;
    }

    if (g_strcasecmp(servername, server->real_address) == 0)
	server->lag_sent = 0;

    if (gettimeofday(&now, NULL) != 0)
    {
        g_warning("check_lag() : gettimeofday() failed\n");
	g_free(params);
	lag_free(lag);
        return TRUE;
    }

    if (strncmp(msg, "LAG", 3) == 0)
    {
	/* great, server returned string we sent in PING, we can get the
	   send time here easily (ircnet doesn't do this) */
	if (sscanf(msg+3, "%ld.%ld", &sent.tv_sec, &sent.tv_usec) == 0)
	{
	    /* failed ?? */
	    g_free(params);
	    lag_free(lag);
            return TRUE;
	}
    }
    else
    {
	/* get the PING send time */
	sent = lag->time;
    }

    /* calculate time difference */
    secs = now.tv_sec - sent.tv_sec;
    usecs = now.tv_usec - sent.tv_usec;
    if (usecs < 0)
    {
	usecs += 1000000;
	secs--;
    }

    /* and we've got the lag */
    lagtime = usecs/1000 + secs * 1000;
    if (g_strcasecmp(server->real_address, lag->address) == 0)
    {
	/* server lag.. */
	server->lag = lagtime;
	signal_emit("server lag", 1, server);
    }
    else
    {
	/* some other lag */
	signal_emit("lag", 2, lag->address, GINT_TO_POINTER((gint) lagtime));
    }

    g_free(params);
    lag_free(lag);
    return TRUE;
}

static gint sig_check_lag(void)
{
    GList *tmp, *next;
    gchar *str;
    time_t now;
    gint min_lag_check_time;

    min_lag_check_time = setup_get_int("min_lag_check_time");

    now = time(NULL);
    for (tmp = servers; tmp != NULL; tmp = next)
    {
	SERVER_REC *rec = tmp->data;

	next = tmp->next;
	if (rec->lag_sent != 0)
	{
	    /* waiting for PONG .. */
	    if (rec->lag_last_check+LAG_SEND_ANOTHER_PING < now)
	    {
		/* send the PING again.. */
		rec->lag_last_check = time(NULL);
		str = g_strdup_printf("PING AGAIN %s", rec->real_address);
		irc_send_cmd(rec, str);
		g_free(str);

		server_redirect_event(rec, rec->real_address, 1, "event pong", "lag event pong", 0, NULL);
	    }
	    else if (setup_get_int("max_lag_before_disconnect") > 1 && now-rec->lag_sent > setup_get_int("max_lag_before_disconnect"))
	    {
		/* too much lag, disconnect */
		signal_emit("server lag disconnect", 1, rec);
		rec->connection_lost = TRUE;
		server_disconnect(rec);
	    }
	}
	else if (rec->lag_last_check+min_lag_check_time < now && rec->cmdcount == 0 && rec->connected && !rec->no_lag_check)
	{
	    /* no commands in buffer - get the lag */
	    lag_get(rec, NULL);
	}
    }

    return 1;
}

void lag_init(void)
{
    lags = NULL;
    timeout_tag = gui_timeout_add(1000, (GUITimeoutFunction) sig_check_lag, NULL);
    signal_add("lag event pong", (SIGNAL_FUNC) event_pong);
}

void lag_deinit(void)
{
    gui_timeout_remove(timeout_tag);
    while (lags != NULL)
	lag_free(lags->data);
    signal_remove("lag event pong", (SIGNAL_FUNC) event_pong);
}
