/*
 misc.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

#include <sys/wait.h>
#include <errno.h>

static gint pids = 0;

/* Read a line */
gint read_line(gboolean socket, gint handle, GString *output, GString *buffer)
{
    gchar tmpbuf[512];
    gint recvlen, pos;

    g_return_val_if_fail(handle != -1, -1);
    g_return_val_if_fail(output != NULL, -1);
    g_return_val_if_fail(buffer != NULL, -1);

    g_string_truncate(output, 0);

    recvlen = socket ?
        net_receive(handle, tmpbuf, sizeof(tmpbuf)-1) :
        read(handle, tmpbuf, sizeof(tmpbuf)-1);

    if (recvlen <= 0)
    {
        if (buffer->len > 0)
        {
            /* no new data got but still something in buffer.. */
            for (pos = 0; pos < buffer->len; pos++)
            {
                if (buffer->str[pos] == 13 || buffer->str[pos] == 10)
                {
                    recvlen = 0;
                    break;
                }
            }
            if (recvlen < 0 && buffer->len > 0)
            {
                /* connection closed and last line is missing \n ..
                   just add it so we can see if it had anything useful.. */
                recvlen = 0;
                g_string_append_c(buffer, '\n');
            }
        }

        if (recvlen < 0) return -1;
    }
    else
    {
        /* append received data to buffer */
        tmpbuf[recvlen] = '\0';
        g_string_append(buffer, tmpbuf);
    }

    for (pos = 0; pos < buffer->len; pos++)
    {
        if (buffer->str[pos] == 13 || buffer->str[pos] == 10)
        {
	    /* end of line */
	    gchar chr;

	    chr = buffer->str[pos];
	    buffer->str[pos] = '\0';
            g_string_assign(output, buffer->str);

            if (chr == 13 && buffer->str[pos+1] == 10)
            {
                /* skip \n too */
                pos++;
            }

	    g_string_erase(buffer, 0, pos+1);
            return 1;
        }
    }

    /* EOL wasn't found, wait for more data.. */
    return 0;
}

gboolean copyfile(gchar *src, gchar *dest)
{
    FILE *fs, *fd;
    gboolean ret;

    g_return_val_if_fail(src != NULL, FALSE);
    g_return_val_if_fail(dest != NULL, FALSE);

    ret = FALSE;
    fs = fopen(src, "rb");
    if (fs != NULL)
    {
        remove(dest); /* mm. just to be sure there's no link already in dest */
        fd = fopen(dest, "w+");
        if (fd != NULL)
        {
            gint len;
            gchar buf[1024];

            while ((len = fread(buf, 1, sizeof(buf), fs)) > 0)
                if (fwrite(buf, 1, len, fd) != len) break;
            fclose(fd);
            ret = TRUE;
        }
        fclose(fs);
    }

    return ret;
}

gint pidcheck_func(void)
{
    if (waitpid(-1, NULL, WNOHANG) > 0) pids--;
    return pids > 0;
}

gint execute(gchar *cmd)
{
    gchar **args;
    gchar *prev;
    gint pid, max, cnt;

    g_return_val_if_fail(cmd != NULL, -1);

    pid = fork();
    if (pid == -1) return FALSE;
    if (pid != 0)
    {
        pids++;
        if (pids == 1) gui_timeout_add(2000, (GUITimeoutFunction) pidcheck_func, NULL);
        return pid;
    }

    max = 5; cnt = 0;
    args = g_malloc(sizeof(gchar *)*max);
    for (prev = cmd; ; cmd++)
    {
        if (*cmd == '\0' || *cmd == ' ')
        {
            args[cnt++] = prev;
            if (cnt == max)
            {
                max += 5;
                args = g_realloc(args, sizeof(gchar *)*max);
            }
            if (*cmd == '\0') break;
            *cmd++ = '\0';
            prev = cmd;
        }
    }
    args[cnt] = NULL;

    execvp(args[0], args);

    _exit(99);
    return -1;
}

gboolean find_substr(gchar *list, gchar *item)
{
    gchar *ptr;

    g_return_val_if_fail(list != NULL, FALSE);
    g_return_val_if_fail(item != NULL, FALSE);

    if (*item == '\0') return FALSE;

    for (;;)
    {
        while (isspace((gint) *list)) list++;
        if (*list == '\0') break;

        ptr = strchr(list, ' ');
        if (ptr == NULL) ptr = list+strlen(list);

        if (g_strncasecmp(list, item, ptr-list) == 0 && item[ptr-list] == '\0')
            return TRUE;

        list = ptr;
    }

    return FALSE;
}

GList *str2list(gchar *list, gchar sep)
{
    GList *glist;

    g_return_val_if_fail(list != NULL, FALSE);

    if (*list == '\0') return NULL;

    list = g_strdup(list);
    glist = NULL;
    for (;;)
    {
	glist = g_list_append(glist, list);

	while (*list != '\0' && *list != sep) list++;
        if (*list == '\0') break;
	*list++ = '\0';

	while (*list == sep) list++;
    }

    return glist;
}

gchar *list2str(GList *list, gchar *sep)
{
    int len, seplen;
    GList *tmp;
    gchar *s, *str;

    len = 0;
    seplen = strlen(sep);
    for (tmp = list = g_list_first(list); tmp!=NULL; tmp=tmp->next)
	len += strlen(tmp->data) + seplen;
    len -= seplen; /* we added one too much */

    s = str = (gchar*) g_malloc(len+1);
    for (tmp = list; tmp!=NULL; tmp = tmp->next)
    {
	g_memmove(s, tmp->data, strlen(tmp->data));
	s += strlen(tmp->data);
	if (tmp->next!=NULL) {
	    g_memmove(s, sep, seplen);
	    s += seplen;
	}
    }
    *s = '\0';
    return str;
}

GList *glist_find_string(GList *list, gchar *key)
{
    for (list = g_list_first(list); list != NULL; list = list->next)
        if (strcmp(list->data, key) == 0) return list;

    return NULL;
}

GList *glist_find_icase_string(GList *list, gchar *key)
{
    for (list = g_list_first(list); list != NULL; list = list->next)
        if (g_strcasecmp(list->data, key) == 0) return list;

    return NULL;
}

gchar *stristr(gchar *data, gchar *key)
{
    gchar *pos, *max;
    gint len;

    if (strlen(key) > strlen(data)) return NULL;

    len = strlen(key);
    max = data+strlen(data)-len;
    for (pos = data; pos <= max; pos++)
        if (g_strncasecmp(pos, key, len) == 0) return pos;

    return NULL;
}

/* Get next parameter */
gchar *event_get_param(gchar **data)
{
    gchar *pos;

    g_return_val_if_fail(data != NULL, NULL);
    g_return_val_if_fail(*data != NULL, NULL);

    if (**data == ':')
    {
        /* easy. :) */
        pos = *data;
        *data += strlen(*data);
        return pos+1;
    }

    pos = *data;
    while (**data != '\0' && **data != ' ') (*data)++;
    if (**data == ' ') *(*data)++ = '\0';

    return pos;
}

/* Get count parameters from data */
gchar *event_get_params(gchar *data, gint count, ...)
{
    gchar **str, *tmp, *duprec;
    gboolean rest;
    va_list args;

    g_return_val_if_fail(data != NULL, NULL);

    va_start(args, count);
#ifdef MEM_DEBUG
    ig_set_data(data);
#endif
    duprec = data = g_strdup(data);
#ifdef MEM_DEBUG
    ig_set_data("");
#endif
    rest = count & PARAM_FLAG_GETREST;
    count = PARAM_WITHOUT_FLAGS(count);
    while (count-- > 0)
    {
        str = (gchar **) va_arg(args, gchar **);
        if (count == 0 && rest)
        {
            /* put the rest to last parameter */
            tmp = *data == ':' ? data+1 : data;
        }
        else
        {
            tmp = event_get_param(&data);
        }
        if (str != NULL) *str = tmp;
    }
    va_end(args);

    return duprec;
}

gchar *cmd_get_param(gchar **data)
{
    gchar *pos;

    g_return_val_if_fail(data != NULL, NULL);
    g_return_val_if_fail(*data != NULL, NULL);

    while (**data == ' ') (*data)++;
    pos = *data;

    while (**data != '\0' && **data != ' ') (*data)++;
    if (**data == ' ') *(*data)++ = '\0';

    return pos;
}

static gchar *get_opt_channel(gchar **data, CHANNEL_REC *channel)
{
    gchar *ret, *p;

    g_return_val_if_fail(data != NULL, NULL);
    g_return_val_if_fail(*data != NULL, NULL);

    p = *data;
    if (ischannel(*p) || (*p == '*' && (p[1] == '\0' || p[1] == ' ')))
    {
        ret = cmd_get_param(data);
        if (*ret == '!')
        {
            /* whenever trying to send something to !channel, change it to
               the real joined !XXXXXchannel */
            CHANNEL_REC *chan;

            chan = channel_find(channel->server, ret);
            if (chan != NULL) ret = chan->name;
        }
    }
    else
        ret = NULL;

    if (ret == NULL || strcmp(ret, "*") == 0)
        ret = channel == NULL ? NULL : channel->name;

    return ret;
}

static gchar *get_opt_args(gchar **data)
{
    gchar *p, *ret;

    g_return_val_if_fail(data != NULL, NULL);
    g_return_val_if_fail(*data != NULL, NULL);

    ret = NULL;
    for (p = *data;;)
    {
        if (*p != '-')
        {
            if (p == *data) return "";
            while (isspace(p[-1]) && p > *data) p--;
            if (*p != '\0') *p++ = '\0';
            ret = *data;
            *data = p;
            return ret;
        }
        while (!isspace(*p) && *p != '\0') p++;
        while (isspace(*p)) p++;
    }
}

/* Get parameters from data, if channel is TRUE, the first parameter is
   optional channel */
gchar *cmd_get_params(gchar *data, gint count, ...)
{
    CHANNEL_REC *channel;
    gchar **str, *tmp, *duprec;
    va_list args;
    gint cnt;

    g_return_val_if_fail(data != NULL, NULL);

    va_start(args, count);
    channel = (count & PARAM_FLAG_OPTCHAN) == 0 ? NULL :
        (CHANNEL_REC *) va_arg(args, CHANNEL_REC *);

    duprec = data = g_strdup(data);
    cnt = PARAM_WITHOUT_FLAGS(count);
    while (cnt-- > 0)
    {
        str = (gchar **) va_arg(args, gchar **);
        if (count & PARAM_FLAG_OPTCHAN)
        {
            /* get channel name */
            tmp = get_opt_channel(&data, channel);
            count &= ~PARAM_FLAG_OPTCHAN;
        }
        else if (count & PARAM_FLAG_OPTARGS)
        {
            tmp = get_opt_args(&data);
	    count &= ~PARAM_FLAG_OPTARGS;
        }
        else if (cnt == 0 && count & PARAM_FLAG_GETREST)
        {
            /* get rest */
            tmp = data;
        }
        else
        {
            tmp = cmd_get_param(&data);
        }

        if (str != NULL) *str = tmp;
    }
    va_end(args);

    return duprec;
}

gchar *convert_home(gchar *path)
{
    return *path == '~' && (*(path+1) == '/' || *(path+1) == '\0') ?
        g_strconcat(g_get_home_dir(), path+1, NULL) :
        g_strdup(path);
}

gboolean match_wildcards(gchar *mask, gchar *data)
{
    gchar *p1, *p2, *orig;
    gboolean ret;

    g_return_val_if_fail(mask != NULL, FALSE);
    g_return_val_if_fail(data != NULL, FALSE);

    orig = mask = g_strdup(mask);
    for (; *mask != '\0' && *data != '\0'; mask++)
    {
        if (*mask == '*')
        {
            mask++;
            if (*mask == '\0')
            {
                data += strlen(data);
                break;
            }
            while (*mask == '?' || *mask == '*') mask++;

            p1 = strchr(mask, '*');
            p2 = strchr(mask, '?');
            if (p1 == NULL || (p2 < p1 && p2 != NULL)) p1 = p2;

            if (p1 != NULL) *p1 = '\0';

            data = stristr(data, mask);
            if (data == NULL) break;

            data += strlen(mask);
            mask += strlen(mask)-1;

            if (p1 != NULL) *p1 = p1 == p2 ? '?' : '*';
        }
        else if (toupper(*mask) == toupper(*data) || *mask == '?')
            data++;
        else
            break;
    }

    ret = data != NULL && *data == '\0' && *mask == '\0';
    g_free(orig);

    return ret;
}

gint g_istr_equal (gconstpointer v, gconstpointer v2)
{
  return g_strcasecmp ((const gchar*) v, (const gchar*)v2) == 0;
}

/* a char* hash function from ASU */
guint g_istr_hash (gconstpointer v)
{
  const char *s = (char*)v;
  const char *p;
  guint h=0, g;

  for(p = s; *p != '\0'; p += 1) {
    h = ( h << 4 ) + toupper(*p);
    if ( ( g = h & 0xf0000000 ) ) {
      h = h ^ (g >> 24);
      h = h ^ g;
    }
  }

  return h /* % M */;
}

#ifndef HAVE_STRERROR
static char * strerror(int errno)
{
    extern int sys_nerr;
    extern char * sys_errlist[];

    if ((0 <= errno) && (errno < sys_nerr))
	return sys_errlist[errno];
    else
	return "unknown errno";
}
#endif

/* vim: set shiftwidth=4 sts=4: */
