/*
 modes.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

/* Max mode changes allowed in MODE command, 3 in ircnet */
#define MAX_MODES_PER_COMMAND 3

static void invitelist_free(CHANNEL_REC *channel)
{
    g_return_if_fail(channel != NULL);

    g_list_foreach(channel->invitelist, (GFunc) g_free, NULL);
    g_list_free(channel->invitelist);
}

void invitelist_add(CHANNEL_REC *channel, gchar *mask)
{
    g_return_if_fail(channel != NULL);
    g_return_if_fail(mask != NULL);

    channel->invitelist = g_list_append(channel->invitelist, g_strdup(mask));

    signal_emit("invitelist new", 2, channel, mask);
}

void invitelist_remove(CHANNEL_REC *channel, gchar *mask)
{
    GList *tmp;

    g_return_if_fail(channel != NULL);
    g_return_if_fail(mask != NULL);

    tmp = glist_find_icase_string(channel->invitelist, mask);
    if (tmp != NULL)
    {
        signal_emit("invitelist remove", 2, channel, tmp->data);
        g_free(tmp->data);
        channel->invitelist = g_list_remove_link(channel->invitelist, tmp);
        g_list_free_1(tmp);
    }
}


/* Change nick's mode in channel */
static void nick_mode_change(CHANNEL_REC *channel, gchar *nick, gchar mode, gboolean set)
{
    NICK_REC *nickrec;

    g_return_if_fail(channel != NULL);
    g_return_if_fail(nick != NULL);

    nickrec = nicklist_find(channel, nick);
    if (nickrec == NULL) return; /* No /names list yet */

    if (mode == '@') nickrec->op = set;
    if (mode == '+') nickrec->voice = set;

    signal_emit("nick mode changed", 2, channel, nickrec);
}

/* Parse channel mode string */
void modes_parse_channel(CHANNEL_REC *channel, gchar *setby, gchar *modestr)
{
    gchar *ptr, *mode, type;

    g_return_if_fail(channel != NULL);
    g_return_if_fail(setby != NULL);
    g_return_if_fail(modestr != NULL);

    type = '+';
    for (mode = cmd_get_param(&modestr); *mode != '\0'; mode++)
    {
        switch (*mode)
        {
            case '+':
            case '-':
                type = *mode;
                break;

            case 'b':
                ptr = cmd_get_param(&modestr);
                if (*ptr == '\0') break;

                if (type == '+')
                    ban_add(channel, ptr, setby, time(NULL));
                else
                    ban_remove(channel, ptr);
                break;

            case 'e':
                ptr = cmd_get_param(&modestr);
                if (*ptr == '\0') break;

                if (type == '+')
                    ban_exception_add(channel, ptr, setby, time(NULL));
                else
                    ban_exception_remove(channel, ptr);
                break;

            case 'I':
                ptr = cmd_get_param(&modestr);
                if (*ptr == '\0') break;

                if (type == '+')
                    invitelist_add(channel, ptr);
                else
                    invitelist_remove(channel, ptr);
                break;

            case 'v':
                ptr = cmd_get_param(&modestr);
                if (*ptr != '\0')
                    nick_mode_change(channel, ptr, '+', type == '+');
                break;

            case 'o':
                ptr = cmd_get_param(&modestr);
                if (*ptr != '\0')
                {
                    if (strcmp(channel->server->nick, ptr) == 0)
                        channel->chanop = type == '+' ? TRUE : FALSE;
                    nick_mode_change(channel, ptr, '@', type == '+');
                }
                break;

            case 'l':
                if (type == '-')
                    channel->limit = 0;
                else
                {
                    ptr = cmd_get_param(&modestr);
                    sscanf(ptr, "%d", &channel->limit);
                }
                signal_emit("channel mode changed", 1, channel);
                break;
            case 'k':
                ptr = cmd_get_param(&modestr);
                if (*ptr != '\0' || type == '-')
                {
                    if (channel->key != NULL)
                    {
                        g_free(channel->key);
                        channel->key = NULL;
                    }
		    if (type == '+') channel->key = g_strdup(ptr);
		    channel->mode_key = type == '+' ? TRUE : FALSE;
                }
                signal_emit("channel mode changed", 1, channel);
                break;

            default:
                switch (*mode)
                {
                    case 'i':
                        channel->mode_invite = type == '+' ? TRUE : FALSE;
                        break;
                    case 'm':
                        channel->mode_moderate = type == '+' ? TRUE : FALSE;
                        break;
                    case 's':
                        channel->mode_secret = type == '+' ? TRUE : FALSE;
                        break;
                    case 'p':
                        channel->mode_private = type == '+' ? TRUE : FALSE;
                        break;
                    case 'n':
                        channel->mode_nomsgs = type == '+' ? TRUE : FALSE;
                        break;
                    case 't':
                        channel->mode_optopic = type == '+' ? TRUE : FALSE;
                        break;
                    case 'a':
                        channel->mode_anonymous = type == '+' ? TRUE : FALSE;
                        break;
                    case 'r':
                        channel->mode_reop = type == '+' ? TRUE : FALSE;
                        break;
                }
                signal_emit("channel mode changed", 1, channel);
                break;
        }
    }
}

/* Parse user mode string */
static void parse_user_mode(SERVER_REC *server, gchar *modestr)
{
    GString *newmode;
    gchar *p, type, c;
    gboolean ok;
    gint n;

    g_return_if_fail(server != NULL);
    g_return_if_fail(modestr != NULL);

    type = '+';
    newmode = g_string_new(server->usermode);
    while (*modestr != '\0' && *modestr != ' ')
    {
	if (*modestr != '+' && *modestr != '-')
	{
	    p = strchr(newmode->str, *modestr);

	    if (type == '+' && p == NULL)
		g_string_append_c(newmode, *modestr);
	    else if (type == '-' && p != NULL)
		g_string_erase(newmode, (gint) (p-newmode->str), 1);
	}

	switch (*modestr)
	{
	    case '+':
	    case '-':
		type = *modestr;
		break;
            case 'i':
                server->usermode_invisible = type == '+' ? TRUE : FALSE;
                break;
            case 'w':
                server->usermode_wallops = type == '+' ? TRUE : FALSE;
                break;
            case 's':
                server->usermode_servernotes = type == '+' ? TRUE : FALSE;
		break;
	}
	modestr++;
    }

    /* my nice little sorting algorithm :) */
    do
    {
	ok = TRUE;
	for (n = 1; n < newmode->len; n++)
	{
	    if (newmode->str[n] < newmode->str[n-1])
	    {
		c = newmode->str[n];
		newmode->str[n] = newmode->str[n-1];
		newmode->str[n-1] = c;
		ok = FALSE;
	    }
	}
    }
    while (!ok);

    if (server->usermode != NULL) g_free(server->usermode);
    server->usermode = newmode->str;
    g_string_free(newmode, FALSE);

    signal_emit("user mode changed", 1, server);
}

static gboolean event_user_mode(gchar *data, SERVER_REC *server)
{
    gchar *params, *nick, *mode;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 3, NULL, &nick, &mode);
    parse_user_mode(server, mode);

    g_free(params);
    return TRUE;
}

static gboolean event_mode(gchar *data, SERVER_REC *server, gchar *nick)
{
    CHANNEL_REC *chanrec;
    gchar *params, *channel, *mode;

    g_return_val_if_fail(data != NULL, FALSE);

    params = event_get_params(data, 2 | PARAM_FLAG_GETREST, &channel, &mode);

    if (!ischannel(*channel))
    {
        /* user mode change */
        parse_user_mode(server, mode);
    }
    else
    {
        /* channel mode change */
        chanrec = channel_find(server, channel);
        if (chanrec != NULL)
            modes_parse_channel(chanrec, nick, mode);
    }

    g_free(params);
    return TRUE;
}

static gboolean event_away(gchar *data, SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    server->usermode_away = TRUE;
    signal_emit("user mode changed", 1, server);
    return TRUE;
}

static gboolean event_unaway(gchar *data, SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    server->usermode_away = FALSE;
    if (server->away_reason != NULL)
    {
	g_free(server->away_reason);
	server->away_reason = NULL;
    }
    signal_emit("user mode changed", 1, server);
    return TRUE;
}

static gboolean modes_channel_destroyed(CHANNEL_REC *channel)
{
    g_return_val_if_fail(channel != NULL, FALSE);

    invitelist_free(channel);
    return TRUE;
}

gboolean modes_set(gchar *data, gchar *mode, SERVER_REC *server, CHANNEL_REC *curchan)
{
    gchar *params, *channel, *nicks, *nick;
    GString *str;
    gint num, modepos;

    g_return_val_if_fail(data != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    params = cmd_get_params(data, 2 | PARAM_FLAG_OPTCHAN | PARAM_FLAG_GETREST,
                            curchan, &channel, &nicks);
    if (!ischannel(*channel)) cmd_param_error(CMDERR_NOT_JOINED);
    if (*nicks == '\0') cmd_param_error(CMDERR_NOT_ENOUGH_PARAMS);

    str = g_string_new(NULL); num = modepos = 0;
    while (*nicks != '\0')
    {
        nick = cmd_get_param(&nicks);

        if (num == 0)
        {
            g_string_sprintf(str, "MODE %s %s", channel, mode);
            modepos = str->len;
        }
        else
        {
            /* insert the mode string */
            g_string_insert(str, modepos, mode);
        }

        g_string_sprintfa(str, " %s", nick);

        num++;
        if (num == MAX_MODES_PER_COMMAND)
        {
            /* max. modes / command reached, send to server */
            irc_send_cmd(server, str->str);
            num = 0;
        }
    }
    if (num > 0) irc_send_cmd(server, str->str);

    g_string_free(str, TRUE);

    g_free(params);
    return TRUE;
}

static gboolean cmd_op(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    return modes_set(data, "+o", server, channel);
}

static gboolean cmd_deop(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    return modes_set(data, "-o", server, channel);
}

static gboolean cmd_voice(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    return modes_set(data, "+v", server, channel);
}

static gboolean cmd_devoice(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    return modes_set(data, "-v", server, channel);
}

#define MODE_HAS_ARG(c) ((c) == 'b' || (c) == 'e' || (c) == 'I' || (c) == 'v' || (c) == 'o' || (c) == 'l' || (c) == 'k')

static gboolean cmd_mode(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    gchar *params, *target, *mode, *modestr, type, *arg;
    GString *cmd;
    gint count, pos;

    g_return_val_if_fail(data != NULL, FALSE);
    g_return_val_if_fail(channel != NULL, FALSE);
    if (server == NULL || !server->connected) cmd_return_error(CMDERR_NOT_CONNECTED);

    params = cmd_get_params(data, 2 | PARAM_FLAG_GETREST, &target, &modestr);
    if (strcmp(target, "*") == 0)
        target = channel->name;
    if (*target == '\0') cmd_param_error(CMDERR_NOT_ENOUGH_PARAMS);

    /* send the modes in blocks of 3. */
    cmd = g_string_new(NULL);
    type = '+'; count = 0; pos = 0;
    for (mode = cmd_get_param(&modestr); ; mode++)
    {
	if (*mode == '\0' || (count == MAX_MODES_PER_COMMAND && MODE_HAS_ARG(*mode)))
	{
	    g_string_prepend(cmd, "MODE  ");
	    g_string_insert(cmd, 5, target);
	    irc_send_cmd(server, cmd->str);
	    if (*mode == '\0') break;

	    count = 0; pos = 0;
	    g_string_truncate(cmd, 0);
	}

	g_string_insert_c(cmd, pos, *mode);
	pos++;

	if (*mode == '+' || *mode == '-')
        {
	    type = *mode;
	    continue;
	}

        if (ischannel(*target) && MODE_HAS_ARG(*mode))
	{
	    count++;
	    arg = cmd_get_param(&modestr);
	    if (*arg != '\0') g_string_sprintfa(cmd, " %s", arg);
	}
    }

    g_string_free(cmd, TRUE);

    g_free(params);
    return TRUE;
}

void modes_init(void)
{
    signal_add("event 221", (SIGNAL_FUNC) event_user_mode);
    signal_add("event 305", (SIGNAL_FUNC) event_unaway);
    signal_add("event 306", (SIGNAL_FUNC) event_away);
    signal_add("event mode", (SIGNAL_FUNC) event_mode);
    signal_add("channel destroyed", (SIGNAL_FUNC) modes_channel_destroyed);

    command_bind("op", NULL, (SIGNAL_FUNC) cmd_op);
    command_bind("deop", NULL, (SIGNAL_FUNC) cmd_deop);
    command_bind("voice", NULL, (SIGNAL_FUNC) cmd_voice);
    command_bind("devoice", NULL, (SIGNAL_FUNC) cmd_devoice);
    command_bind("mode", NULL, (SIGNAL_FUNC) cmd_mode);
}

void modes_deinit(void)
{
    signal_remove("event 221", (SIGNAL_FUNC) event_user_mode);
    signal_remove("event 305", (SIGNAL_FUNC) event_unaway);
    signal_remove("event 306", (SIGNAL_FUNC) event_away);
    signal_remove("event mode", (SIGNAL_FUNC) event_mode);
    signal_remove("channel destroyed", (SIGNAL_FUNC) modes_channel_destroyed);

    command_unbind("op", (SIGNAL_FUNC) cmd_op);
    command_unbind("deop", (SIGNAL_FUNC) cmd_deop);
    command_unbind("voice", (SIGNAL_FUNC) cmd_voice);
    command_unbind("devoice", (SIGNAL_FUNC) cmd_devoice);
    command_unbind("mode", (SIGNAL_FUNC) cmd_mode);
}
