/****************************************************************************
 *                                 DFIO.cc  
 *
 * Author: Matthew Ballance
 * Desc:   Implements the core functionality of the DFIO.
 *
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 ****************************************************************************/
#include "DFIOTrace.h"
#include "vector.h"
#include "WidgetMgrTypes.h"
#include "WidgetManager.h"
#include "CmdSwitcher.h"


/************************************************************
 * DFIO()
 ************************************************************/
DFIO::DFIO(
        Tcl_Interp          *interp,
        Uint32               argc,
        Char               **argv) : d_instName(argv[1]), d_interp(interp)
{
    d_ok = false;

    if (WidgetMgr_GetObjHandle(interp, "DFIO", argv[1])) {
        Tcl_AppendResult(d_interp, "DFIO instance ", argv[1], 
                " already exists", 0);
        return;
    }

    WidgetMgr_AddInst(interp, WIDGET_TYPE_DFIO, argv[1], this);
    Tcl_SetObjResult(interp, Tcl_NewStringObj(argv[1], -1));

    Tcl_CreateCommand(d_interp, argv[1], &DFIO::InstCmd, this, 0);

    traces = new Vector<DFIOTrace>();
    signalHash = new HashTable<DFIOTrace>();
    rfp = 0;
    wfp = 0;
    readPos = writePos = 0;
    currentTime = 0; 
    currentTime64.low = currentTime64.high = 0; 
    maxTime     = 0; 
    d_separateTraceMax = false;

    timeUnit = -9;
    timeMult = 1;

    d_ok = true;
}

/************************************************************
 * ~DFIO()
 ************************************************************/
DFIO::~DFIO()
{
    Tcl_DeleteCommand(d_interp, d_instName.value());
    WidgetMgr_DelInst(d_interp, "DFIO", d_instName.value());
}

/************************************************************
 * Delete()
 ************************************************************/
void DFIO::Delete()
{
    Tcl_DoWhenIdle(&DFIO::Delete, this);
}

typedef enum {
    DFC_Delete,
    DFC_NumCmds
} DFIOCmds;

static CmdSwitchStruct dfio_cmds[] = {
    { "delete",              DFC_Delete        },
    { "",                    0                 }
};

/************************************************************
 * InstCmd()
 ************************************************************/
int DFIO::InstCmd(int argc, const char **argv)
{
    int cmd;

    cmd = CmdSwitch(dfio_cmds, (char *)argv[1]);

    switch (cmd) {
        case DFC_Delete:
            Delete();
            break;

        default:
            Tcl_AppendResult(d_interp, "Unknown command - ", argv[1], 0);
            return TCL_ERROR;
            break;
    }

    return TCL_OK;
}

/********************************************************************
 * setTimeUnit()
 ********************************************************************/
void DFIO::setTimeUnit(Int32 unit)
{
    base_setTimeUnit(unit);
}

/************************************************************
 * newTrace()
 ************************************************************/
DFIOTrace *DFIO::base_newTrace(
        Char        *name,
        Uint32       rows,
        Uint32       msb,
        Uint32       lsb,
        Uint32       traceType)
{
    DFIOTrace *trace = new DFIOTrace(name, rows, msb, lsb, traceType);
    return trace;
}

/************************************************************
 * base_openFile()
 ************************************************************/
Int32 DFIO::base_openFile(
    Char        *fileName,
    Char        *mode)
{
    Char    modebuf[16] = "";

    fprintf(stderr, "NOTE: opening \"%s\"\n", fileName);

    if (!mode || !(*mode) ||
        (strlen(mode) > 2) ) {
        return -1;
    }

    if (mode[0] == 'w' || mode[1] == 'w') {
        wfp = fopen(fileName, "w");
        if (!wfp) {
            fprintf(stderr, "ERROR: Cannot open \"%s\" for writing\n", fileName);
            return -1;
        }
        openMode |= OM_Write;
    }

    if (mode[0] == 'r' || mode[1] == 'r') {
        rfp = fopen(fileName, "r");
        if (!rfp) {
            fprintf(stderr, "ERROR: Cannot open \"%s\" for reading\n", fileName);
            return -1;
        }
        openMode |= OM_Read;
    }

    fprintf(stderr, "Returning 0\n");
    return 0;
}

/************************************************************
 * base_addTrace()
 ************************************************************/
Int32 DFIO::base_addTrace(DFIOTrace *newTrace)
{
    newTrace->parent = this;
    traces->append(newTrace);
    signalHash->addEntry(newTrace);

    newTrace->setIndepTraceMax(d_separateTraceMax);

    return 0;
}

/************************************************************
 * base_findTrace()
 ************************************************************/
DFIOTrace *DFIO::base_findTrace(Char *traceName)
{
    DFIOTrace *ret;

    ret = signalHash->findEntry(traceName);

    return ret;
}

/************************************************************
 * setTime()
 ************************************************************/
void DFIO::setTime(Uint32 time) {
    base_setTime(time);
}

/************************************************************
 * base_setTime()
 ************************************************************/
void DFIO::base_setTime(Uint32       time) 
{
    currentTime       = (time*timeMult);
    currentTime64.low = (time*timeMult);

    if (currentTime > maxTime) {
        maxTime = currentTime;
    }
}

/************************************************************
 *
 ************************************************************/
Uint32 DFIO::getMaxTime()
{
    return maxTime;
}

/************************************************************
 * setMaxTime()
 ************************************************************/
void DFIO::setMaxTime(Uint32 max)
{
    maxTime = (max * timeMult);
}

/************************************************************
 * base_setTimeUnit()
 ************************************************************/
void DFIO::base_setTimeUnit(Int32   unit)
{
    timeUnit = unit;
}

/************************************************************
 * getTimeUnit()
 ************************************************************/
Int32 DFIO::getTimeUnit()
{
    return timeUnit;
}

/************************************************************
 * setUnitMultiplier()
 ************************************************************/
void DFIO::setUnitMultiplier(Int32 multiplier)
{
    timeMult = multiplier;
}

/************************************************************
 * getUnitMultiplier()
 ************************************************************/
Int32 DFIO::getUnitMultiplier()
{
    return timeMult;
}


